/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.report.data.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.hibernate.Session;
import org.hibernate.Query;

import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.data.QueryCriteria;
import gov.va.med.fw.report.data.ReportDataException;
import gov.va.med.fw.util.StringUtils;

import gov.va.med.esr.common.model.lookup.EnrollmentProcessStatus;
import gov.va.med.esr.common.model.lookup.EnrollmentStatusComponent;
import gov.va.med.esr.common.model.lookup.ReportFacilityDisplayBy;
import gov.va.med.esr.common.model.lookup.ReportFormat;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.report.ReportEnrollmentStatusComponent;
import gov.va.med.esr.common.model.report.ReportFacility;
import gov.va.med.esr.common.model.report.ReportParameterSet;
import gov.va.med.esr.common.model.report.ReportSetup;
import gov.va.med.esr.common.report.data.CommonCriteria;
import gov.va.med.esr.common.report.data.StandardReportCriteria;

/**
 * QM3 Reports Data DAO.
 * 
 * @author Muddaiah Ranga
 *
 */
public class QM3ReportDataDAOImpl extends AbstractStandardReportDataDAOImpl {

    private static final long serialVersionUID = 1178123948624532738L;

    public void preDataRetrieval( ReportConfiguration config ) throws ReportDataException {
	      
	      QueryCriteria reportCriteria = config.getQueryCriteria();
		  StandardReportCriteria criteria = (StandardReportCriteria)reportCriteria;
		  ReportSetup setup = criteria.getReportSetup();
		  ReportParameterSet parameters = setup.getParameterSet();
			
	      if( reportCriteria instanceof StandardReportCriteria ) {
	         criteria.setStartDate( this.getDate( config, true ) );
	         criteria.setEndDate( this.getDate( config, false ) );
	         
	            //Optional fields for Detailed report
            if (ReportFormat.DETAIL.getName().equals(parameters.getReportFormat().getCode())){
        		//outlier days
        		Integer days = parameters.getOutlierDays();
        		if (days == null) days = new Integer(0);
        		criteria.setExceedXDays(days.toString());
        		
                Set enrollmentStatuses = parameters.getEnrollmentStatusComponents();
                Iterator iter = enrollmentStatuses.iterator();
                while (iter.hasNext()) {
                	ReportEnrollmentStatusComponent comp = (ReportEnrollmentStatusComponent)iter.next();
                	String code = comp.getEnrollmentStatusComponent().getCode();
                	criteria.addCriterion("enrollmentStatusComponent" + code, "Y");
                }
            }
	      }
    }
    
    protected Query buildQuery( ReportConfiguration config, Session session ) throws ReportDataException {
        
        QueryCriteria reportCriteria = config.getQueryCriteria();
        Query query = null;

        if( reportCriteria instanceof StandardReportCriteria ) {

            // Get a named query
            query = this.getNamedQuery( session );
            
            // Get a criteria
            StandardReportCriteria criteria = (StandardReportCriteria)reportCriteria;
            ReportSetup setup = criteria.getReportSetup();
            ReportParameterSet parameters = setup != null ? setup.getParameterSet() : null;
            if( parameters == null ) {
                throw new ReportDataException("Missing report parameters in report " + config.getReportID() );
            }

            Date startDate = parameters.getFromDate();
            if(startDate == null) {
                throw new ReportDataException( "Missing Start Date in report " + config.getReportID() );
            }
            
            Date endDate = parameters.getToDate();
            if(endDate == null) {
                throw new ReportDataException( "Missing End Date in report " + config.getReportID() );
            }
            
            query.setDate(CommonCriteria.START_DATE, startDate);
            query.setDate(CommonCriteria.END_DATE,endDate);
            
            
            ReportFacilityDisplayBy facilityDisplayBy = parameters.getFacilityDisplayBy();
            if(facilityDisplayBy == null || StringUtils.isEmpty(facilityDisplayBy.getCode())) {
                //No additional parameters are required to be passed
            } else {    
                //When 1 or more VISNs selected
                if(StringUtils.equals(facilityDisplayBy.getCode(),ReportFacilityDisplayBy.CODE_VISN.getCode())) {
                    List visns = this.getFacilitityIds(parameters.getFacilities());
                    if(visns != null && !visns.isEmpty()) {
                        query.setParameterList( CommonCriteria.VISNS, visns.toArray() );
                    }
                    //case where novisn is selected
                    else {
                        List facilities = new ArrayList();
                        facilities.add(new Integer(0));
                        query.setParameterList( CommonCriteria.VISNS, facilities);
                    }
                    //NOVISN is selected
                    if (parameters.isNoVISNPresent()){
                        query.setString(CommonCriteria.IS_NO_VISN,CommonCriteria.IS_NO_VISN);
                    }
                    else {
                        query.setString(CommonCriteria.IS_NO_VISN,"VISN");
                    }                    
                //When 0, more or all facilities selected
                } else {
                    Set reportFacilities = parameters.getFacilities();
                    if(reportFacilities == null || reportFacilities.isEmpty()) {
                        List facilities = new ArrayList();
                        facilities.add(new Integer(0));
                        query.setParameterList( CommonCriteria.FACILITIES, facilities);
                        query.setString(CommonCriteria.IS_ALL,"all");
                    } else {
                        query.setParameterList( CommonCriteria.FACILITIES, this.getFacilitityIds(reportFacilities));
                        query.setString(CommonCriteria.IS_ALL,"notAll");
                    }
                }
            }
            
            
            EnrollmentProcessStatus processStatus = parameters.getEnrollmentProcessStatus();
            
            if (processStatus != null && EnrollmentProcessStatus.CODE_INCOMPLETE.equals(processStatus.getCodeObject()))
            {
            	query.setString("incompleteFlg", "Y");
            }
            else 
            {
            	query.setString("incompleteFlg", "N");
            }
            
            //Optional fields for Detailed report
            if (ReportFormat.DETAIL.getName().equals(parameters.getReportFormat().getCode())){
        		//outlier days
        		Integer days = parameters.getOutlierDays();
        		if (days == null) days = new Integer(0);
        		query.setInteger("days",days.intValue());
        		
                Set enrollmentStatuses = parameters.getEnrollmentStatusComponents();
                Iterator iter = enrollmentStatuses.iterator();
                ArrayList listComps = new ArrayList();
                while (iter.hasNext()) {
                	ReportEnrollmentStatusComponent comp = (ReportEnrollmentStatusComponent)iter.next();
                	String code = comp.getEnrollmentStatusComponent().getCode();
                	listComps.add(code);
                }
                query.setParameterList("enrollmentStatusComponent", listComps);
            }
        }
        return query;
    }
    
    private List getFacilitityIds(Set facilities) {
        Iterator i = facilities != null ? facilities.iterator() : null;
        ArrayList list = new ArrayList();
        while( i != null && i.hasNext() ) {
            ReportFacility rFacility = (ReportFacility)i.next();
            if(rFacility != null) list.add(((VAFacility)rFacility.getLookup()).getIdentifier());
        }
        return list;
    }
}