/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.esr.common.report.data.impl;

// Java classes
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

// Library classes
import org.hibernate.Query;
import org.hibernate.Session;

// Framework classes
import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.data.QueryCriteria;
import gov.va.med.fw.report.data.ReportDataException;
import gov.va.med.fw.util.StringUtils;

// ESR classes
import gov.va.med.esr.common.model.lookup.ReportFacilityDisplayBy;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.report.ReportFacility;
import gov.va.med.esr.common.model.report.ReportSetup;
import gov.va.med.esr.common.model.report.ReportParameterSet;
import gov.va.med.esr.common.report.data.CommonCriteria;
import gov.va.med.esr.common.report.data.StandardReportCriteria;

/**
 * EED3 reports DAO.
 *
 * @author Muddaiah Ranga
 */
public class EED3ReportDataDAOImpl extends AbstractStandardReportDataDAOImpl {

	private static final long serialVersionUID = -5657439440961807171L;
    
	/**
	 * A default constructor
	 */
	public EED3ReportDataDAOImpl() {
		super();
	}

    /* 
     * @see gov.va.med.fw.report.data.ReportDataDAO#preDataRetrieval(gov.va.med.fw.report.ReportConfiguration)
     */
    public void preDataRetrieval( ReportConfiguration config ) throws ReportDataException {
        QueryCriteria reportCriteria = config.getQueryCriteria();

        if( reportCriteria instanceof StandardReportCriteria ) {
           StandardReportCriteria criteria = (StandardReportCriteria)reportCriteria;
           ReportSetup setup = criteria.getReportSetup();
           ReportParameterSet parameters = setup != null ? setup.getParameterSet() : null;
           if( parameters == null ) {
                throw new ReportDataException("Missing report parameters in report " + config.getReportID() );
            }
           criteria.addCriterion(CommonCriteria.AS_OF_DATE,parameters.getAsOfDate());
           criteria.addCriterion(CommonCriteria.PH_UNCONFIRMED_DAYS,parameters.getPhUnconfirmedDays());
        }
    }
    
	/**
	 * @see gov.va.med.fw.report.data.hibernate.HibernateReportDataDAO#buildQuery(gov.va.med.fw.report.ReportConfiguration)
	 */
	protected Query buildQuery( ReportConfiguration config, Session session ) throws ReportDataException {
		
		QueryCriteria reportCriteria = config.getQueryCriteria();
		Query query = null;

		if( reportCriteria instanceof StandardReportCriteria ) {

			// Get a named query
			query = this.getNamedQuery( session );
			
			// Get a criteria
			StandardReportCriteria criteria = (StandardReportCriteria)reportCriteria;
			ReportSetup setup = criteria.getReportSetup();
			ReportParameterSet parameters = setup != null ? setup.getParameterSet() : null;
			if( parameters == null ) {
				throw new ReportDataException("Missing report parameters in report " + config.getReportID() );
			}

            Date asOfDate = parameters.getAsOfDate();
            
			if(asOfDate == null) {
				throw new ReportDataException( "Missing As Of Date in report " + config.getReportID() );
			}
            
            Integer phUnconfirmedDays = parameters.getPhUnconfirmedDays();
            if(phUnconfirmedDays == null) {
                throw new ReportDataException( "Missing PH Unconfirmed Days in report " + config.getReportID() );
            }
			
			query.setDate(CommonCriteria.AS_OF_DATE, parameters.getAsOfDate());
			query.setInteger(CommonCriteria.PH_UNCONFIRMED_DAYS, parameters.getPhUnconfirmedDays().intValue());
            
            ReportFacilityDisplayBy facilityDisplayBy = parameters.getFacilityDisplayBy();
            if(facilityDisplayBy == null || StringUtils.isEmpty(facilityDisplayBy.getCode())) {
                //No additional parameters are required to be passed
            } else {    
                //When 1 or more VISNs selected
                if(StringUtils.equals(facilityDisplayBy.getCode(),ReportFacilityDisplayBy.CODE_VISN.getCode())) {
                    List visns = this.getFacilitityIds(parameters.getFacilities());
                    if(visns != null && !visns.isEmpty()) {
                        query.setParameterList( CommonCriteria.VISNS, visns.toArray() );
                    }
                    //case where novisn is selected
                    else {
                        List facilities = new ArrayList();
                        facilities.add(new Integer(0));
                        query.setParameterList( CommonCriteria.VISNS, facilities);
                    }
                    //NOVISN is selected
                    if (parameters.isNoVISNPresent()){
                        query.setString(CommonCriteria.IS_NO_VISN,CommonCriteria.IS_NO_VISN);
                    }
                    else {
                        query.setString(CommonCriteria.IS_NO_VISN,"VISN");
                    }
                //When 0, more or all facilities selected
                } else {
                    Set reportFacilities = parameters.getFacilities();
                    if(reportFacilities == null || reportFacilities.isEmpty()) {
                        List facilities = new ArrayList();
                        facilities.add(new Integer(0));
                        query.setParameterList( CommonCriteria.FACILITIES, facilities);
                        query.setString(CommonCriteria.IS_ALL,"all");
                    } else {
                        query.setParameterList( CommonCriteria.FACILITIES, this.getFacilitityIds(reportFacilities));
                        query.setString(CommonCriteria.IS_ALL,"notAll");
                    }
                }
            }
		}
		return query;
	}
    
    private List getFacilitityIds(Set facilities) {
        Iterator i = facilities != null ? facilities.iterator() : null;
        ArrayList list = new ArrayList();
        while( i != null && i.hasNext() ) {
            ReportFacility rFacility = (ReportFacility)i.next();
            if(rFacility != null) list.add(((VAFacility)rFacility.getLookup()).getIdentifier());
        }
        return list;
    }
}