/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.esr.common.report.data;

// Java classes
import java.util.Date;
import java.util.Calendar;
import java.util.Locale;

// Library classes
import org.springframework.context.MessageSource;

// Framework classes
import gov.va.med.esr.common.model.lookup.AckType.Code;
import gov.va.med.fw.model.lookup.AbstractCode;
import gov.va.med.fw.util.StringUtils;

// ESR classes

/**
 * Encapsulates report data to generate OPP Common extract files 
 * such as the files defined in this class
 *
 * Project: Common</br>
 * Created on: 1:25:19 PM </br>
 *
 * @author DNS   LEV
 */
public class CommonExtractFileCriteria extends StandardReportCriteria {
	
	/**
	 * An instance of CDCONDITION_FILE
	 */
	public static final Type CDCONDITION_FILE = new Type("OPP 3");
	/**
	 * An instance of CDDIAGNOSIS_FILE
	 */
	public static final Type CDDIAGNOSIS_FILE = new Type("OPP 2");
	/**
	 * An instance of CDPROCEDURE_FILE
	 */
	public static final Type CDPROCEDURE_FILE = new Type("OPP 1");

	/**
	 * An instance of ELIGIBILITY_FILE
	 */
	public static final Type ELIGIBILITY_FILE = new Type("OPP 6");
	/**
	 * An instance of PERIODOFSERVICE_FILE
	 */
	public static final Type PERIODOFSERVICE_FILE = new Type("OPP 4");
	/**
	 * An instance of RATEDDISABILITIES_FILE
	 */
	public static final Type RATEDDISABILITIES_FILE = new Type("OPP 7");
	
	/**
	 * An instance of COMBATEPISODES_FILE 
	 */
	public static final Type COMBATEPISODES_FILE = new Type("OPP 8");	
	/**
	 * An instance of MAIN_FILE
	 */
	public static final Type MAIN_FILE = new Type("OPP 5");
	
	/**
	 * A report type
	 */
	public static class Type extends AbstractCode {
		
		/**
		 * An instance of serialVersionUID
		 */
		private static final long serialVersionUID = 6209605494205996214L;

		/**
		 * A default constructor
		 * @param type
		 */
		private Type(String code) {
			super(code);
		}

		/**
		 * @param code
		 * @return
		 */
		public static Code getByCode(String code) {
			return (Code) getCode(Code.class, code);
		}
	}
	
	/**
	 * An instance of RUN_DATE
	 */
	private static final String RUN_DATE = "runDate";
	/**
	 * An instance of RECORDS
	 */
	private static final String RECORDS = "records";
	/**
	 * An instance of EXTRACT_FILE
	 */
	private static final String EXTRACT_FILE = "filename";
	/**
	 * An instance of REPORT_ID
	 */
	private static final String REPORT_ID = "id";
	/**
	 * An instance of REPORT_TITLE
	 */
	private static final String REPORT_TITLE = "title";
	
	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -5789963986526186284L;

	/**
	 * An instance of reportType
	 */
	private CommonExtractFileCriteria.Type reportType = null;
	
	/**
	 * A default constructor
	 */
	public CommonExtractFileCriteria( CommonExtractFileCriteria.Type type ) {
		this( type, null );
	}
	
	/**
	 * A default constructor
	 * @param type
	 * @param bundle
	 */
	public CommonExtractFileCriteria( CommonExtractFileCriteria.Type type,  MessageSource bundle ) {
		super();
		this.reportType = type;
		if( bundle != null ) {
			
			// Set default values loaded from a bundle for filename, title, and id
			String code = StringUtils.deleteWhitespace( type.getCode() );
			String key = code + "." + EXTRACT_FILE;
			this.setExtractFileName( getDefaultValue( bundle, key ) );

			key = code + "." + REPORT_TITLE;
			this.setReportTitle( getDefaultValue( bundle, key ) );

			key = code + "." + REPORT_ID;
			this.setReportID( getDefaultValue( bundle, key ) );
		}
		// Set a default run date
		this.setRunDate( Calendar.getInstance().getTime() );
	}
	
	public void setRunDate( Date date ) {
		this.addCriterion( RUN_DATE, date );
	}
	public Date getRunDate() {
		Object criterion = this.getCriteria().get( RUN_DATE );
		return criterion instanceof Date ? (Date)criterion : null;
	}
	public void setExtractFileName( String name ) {
		this.addCriterion( EXTRACT_FILE, name );
	}
	public String getExtractFileName() {
		Object criterion = this.getCriteria().get( EXTRACT_FILE );
		return criterion instanceof String ? (String)criterion : null;
	}
	public void setRecords( Integer records ) {
		this.addCriterion( RECORDS, records );
	}
	public Integer getRecords() {
		Object criterion = this.getCriteria().get( RECORDS );
		return criterion instanceof Integer ? (Integer)criterion : null;
	}
	public void setReportID( String id ) {
		this.addCriterion( REPORT_ID, id );
	}
	public String getReportID() {
		Object criterion = this.getCriteria().get( REPORT_ID );
		return criterion instanceof String ? (String)criterion : null;
	}
	public void setReportTitle( String title ) {
		this.addCriterion( REPORT_TITLE, title );
	}
	public String getReportTitle() {
		Object criterion = this.getCriteria().get( REPORT_TITLE );
		return criterion instanceof String ? (String)criterion : null;
	}
	
	/**
	 * @return Returns the report type.
	 */
	public CommonExtractFileCriteria.Type getReportType() {
		return reportType;
	}
	
	private String getDefaultValue( MessageSource bundle, String key ) {
		String value = null;
		try {
			value = bundle.getMessage( key, null, Locale.getDefault() );
		}
		catch( Exception e ) {
			// Ignore exceptions for default values are not needed
		}
		return value;
	}
}