/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.persistent.security;

import gov.va.med.esr.common.model.lookup.FunctionalGroup;
import gov.va.med.esr.common.model.security.CapabilitySet;
import gov.va.med.esr.common.model.security.EEServiceRequest;
import gov.va.med.esr.common.model.security.EEServiceRequestLite;
import gov.va.med.esr.common.model.security.EEServiceUser;
import gov.va.med.esr.common.model.security.UserLastLoginInfo;
import gov.va.med.esr.common.model.security.UserLogin;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.RolePrincipalImpl;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.fw.security.UserPrincipal;

import java.util.List;
import java.util.Set;

/**
 * A DAO to add, update and find the UserPrincipal.
 * 
 * @author Ghazenfer Mansoor
 */
public interface SecurityDAO extends DAOOperations {
    
    /**
     * Find user given a username. If the user with this name does not exists, 
     * this method returns null.
     * 
     * @param username
     * @return UserPrincipalImpl if the user exists or null otherwise.
     * @throws DAOException if there is any error finding the user.
     * @throws IllegalArgumentException if input is null.
     */
    public UserPrincipal findUserByUsername(String username)
            throws DAOException;


    /**
     * Find the user given the unique user identifier.
     * 
     * @param userId
     * @return UserPrincipalImpl Object.
     * @throws DAOException if there is any error adding the user.
     * @throws IllegalArgumentException if input is null.
     */
    public UserPrincipal getUserById(EntityKey userId) throws DAOException;
    
    /**
     * Find the user given the userPrincipal attributes(givenname, familyName, middleName, userName)
     * @throws DAOException if there is any error while searching.
     * @throws IllegalArgumentException if input is null.
     */
    public UserPrincipal findUser(UserPrincipal userPrincipal) throws DAOException, MaxRecordsExceededException;    
    
    /**
     * Find last login (both failed and success dates for user
     * @param userId
     * @return
     * @throws DAOException
     */
    public UserLastLoginInfo getUserLastLoginInfo(EntityKey userId) throws DAOException;
    
    public List getAllCapabilitySets() throws DAOException;
    
    public List getAllRoles() throws DAOException;
    
    public CapabilitySet findCapabilitySetByName(String name) throws DAOException;
    
    public RolePrincipalImpl findRoleByName(String name) throws DAOException;
    
    public List findDeletedRoles() throws DAOException;
    
    public List findDeletedCapabilitySets() throws DAOException;
    
    public List findAllUsers() throws DAOException;
    
    public Lookup getByCode(String userName) throws DAOException;
    
    public Lookup getByFullname(String userName) throws DAOException;
    
    public void updateAndLog(UserPrincipal userPrincipal, UserLogin userLogin) throws DAOException;
    
    public void updatePassword(UserPrincipal userPrincipal) throws DAOException;
    
    public List findUsersByRoleName(String roleName) throws DAOException;
    
    public List findUsersByCapabilityCode(String capabilityCode) throws DAOException;
    
    public List findUsersByFunctionalGroup(FunctionalGroup functionalGroup) throws DAOException;
    
    public List findUsersByRoleAndFunctionalGroup(FunctionalGroup functionalGroup, String roleName) 
    	throws DAOException;
    
    public List getPasswordChangeHistory(EntityKey userId) throws DAOException;
    
    public List getEEServiceRequests() throws DAOException;
    
    public EEServiceRequest getEEServiceRequest(EntityKey eeServiceRequestId) throws DAOException;
    
    public Set getEEServiceRequestFieldsByRequest(EntityKey eeServiceRequestId) throws DAOException ;

    public EEServiceRequest getEEServiceRequestByName(String name) throws DAOException;

    
    public List getEEServiceLiteRequests() throws DAOException;
    
    public EEServiceRequestLite getEEServiceRequestLite(EntityKey eeServiceRequestId) throws DAOException;
    
    public EEServiceRequestLite getEEServiceRequestLiteByName(String name) throws DAOException;

    public List getEEServiceUsers() throws DAOException;
    
    public EEServiceUser getEEServiceUserByName(String userName) throws DAOException;
    
    public EEServiceUser getEEServiceUserById(EntityKey eeServiceUserId) throws DAOException;
        
    
}