/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.esr.common.persistent.report;

// Java classes
import java.io.Serializable;
import java.util.List;
import java.util.Map;

// Library classes

// Framework classes
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.fw.security.UserPrincipal;
// ESR classes
import gov.va.med.esr.common.model.report.CompletedReport;
import gov.va.med.esr.common.model.report.ReportSetup;
import gov.va.med.esr.common.model.report.SimpleCompletedReport;
import gov.va.med.esr.service.CompletedReportsSearchQueryInfo;
import gov.va.med.esr.service.ReportFilterSearchQueryInfo;

/**
 * Provides data access methods to perform CRUD operations 
 * on generated reports 
 *
 * Project: Common</br>
 * Created on: 11:49:36 AM </br>
 *
 * @author DNS   LEV
 */
public interface ReportDAO extends Serializable, DAOOperations {

	/**
	 * @param user The user that has the permission to work on these setups.
	 * @return A list of ReportSetupLite objects
	 */
	public List getReportSetups( UserPrincipal user ) throws DAOException ;

	/**
	 * 
	 * @param identifier
	 * @return ReportSetup with the specified id or null if not found
	 * @throws DAOException
	 */
	public ReportSetup getReportSetup(EntityKey identifier) throws DAOException ; 

	/**
	 * Add or update a CompletedReport
	 * @param reportSetup
	 * @throws DAOException
	 */
    public void saveReportSetup(ReportSetup reportSetup) throws DAOException;

	/**
	 * Delete a ReportSetup
	 * @param identifier
	 * @throws DAOException
	 */
    public void deleteReportSetup(EntityKey identifier) throws DAOException;

    /**
     * 
     * @param user The user that has permission to work on these reports
     * @return A list of SimpleCompletedReport objects
     * @throws DAOException
     */
	public List getCompletedReports(UserPrincipal user) throws DAOException ;

	/**
	 * 
	 * @param identifier
	 * @return CompletedReport with the specified id or null if not found.
	 * @throws DAOException
	 */
	public CompletedReport getCompletedReport(EntityKey identifier) throws DAOException;

	/**
	 * Add or update a CompletedReport
	 * @param completedReport
	 * @throws DAOException
	 */
    public void saveCompletedReport(CompletedReport completedReport) throws DAOException;
    
    /**
     * Search for completed reports using search criteria.
     * 
     * @param searchCriteria
     * @return A list of CompletedReport objects
     * @throws DAOException
     * @throws MaxRecordsExceededException 
     */
    public List searchCompletedReports(CompletedReportsSearchQueryInfo searchCriteria) throws DAOException, MaxRecordsExceededException;
    
    /**
     * Search for scheduled reports using search criteria.
     * 
     * @param searchCriteria
     * @return A list of ScheduledReport objects
     * 
     * @throws DAOException
     */
    public List searchScheduledReports(ReportFilterSearchQueryInfo searchCriteria) throws DAOException, MaxRecordsExceededException;
    
    /**
     * Delete a Completed Report.
     * @param identifier
     * @throws DAOException
     */
    public void deleteCompletedReport(EntityKey identifier) throws DAOException;
    
    /**
     * This method retrieves SimpleCompletedReport.
     * @param identifier
     * @return
     * @throws DAOException
     */
    
    public SimpleCompletedReport getSimpleCompletedReport(EntityKey identifier) throws DAOException;
    
    /**
     * This method retrieves distinct StandardReports from completed reports table
     * @return
     * @throws DAOException
     */
    public List getDistinctStandardReports() throws DAOException;
    
    /**
     * This method retrieves active and inactive reports counts.
     * activeReportsCount --> count of active reports.
     * inactiveReportsCount --> count of inactive reports.
     * @return
     * @throws DAOException
     */
    public Map getActiveInactiveReportsCount() throws DAOException;
    
    /**
     * This method retrieves distinct ReportExportTypes from completed reports table
     
     * @return
     * @throws DAOException
     */
    public List getDistinctReportExportTypes() throws DAOException;
 }
