/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.persistent.registry;

// Java classes
import java.util.List;

import gov.va.med.esr.common.model.ee.PrisonerOfWar;
import gov.va.med.esr.common.model.ee.PurpleHeart;
import gov.va.med.esr.common.model.ee.SHAD;
import gov.va.med.esr.common.model.lookup.RegistryType;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.registry.RegistryLoadStatistics;
import gov.va.med.esr.common.model.registry.Registry;
import gov.va.med.esr.common.model.registry.RegistryTrait;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.SearchQueryInfo;

/**
 * DAO iterface for registry.
 * 
 * @author Muddaiah Ranga
 * @version 3.0
 */
public interface RegistryDAO extends DAOOperations {
	
    public static final String FIND_PH_BY_IDENTIFIER = "purpleHeart_Identifier";
    public static final String FIND_POW_BY_IDENTIFIER = "prisonerOfWar_Identifier";
    public static final String FIND_POW_BY_ICN = "prisonerOfWar_Icn";
    public static final String FIND_SHAD_BY_IDENTIFIER = "shad_Identifier"; 
    public static final String PARAM_IDENTIFIER = "identifier";
    public static final String PARAM_ICN = "icn";
    
    //Registry Load Summary
    public static final String FIND_LATEST_SUMMARY_BY_REGISTRY_TYPE = "registrySummary_RegistryType";
    public static final String PARAM_REGISTRY_TYPE = "registryType";
    public static final String FIND_LATEST_SUMMARY = "registrySummary_Latest";    
    
    /**
     * Gets the registry matching the registry id and the registry type.
     * 
     * @param entityKey the registry entity key
     * @param registryType the registry type
     * @return
     * @throws DAOException
     */
    public Registry getById(EntityKey entityKey, RegistryType registryType) throws DAOException;
    
    /**
     * Gets the PH registry matching the registry id.
     * 
     * @param entityKey the registry entity key
     * @return
     * @throws DAOException
     */
    public PurpleHeart getPHRegistryById(EntityKey entityKey) throws DAOException;
    
    /**
     * Gets the POW registry matching the registry id.
     * 
     * @param entityKey the registry entity key
     * @return
     * @throws DAOException
     */
    public PrisonerOfWar getPOWRegistryById(EntityKey entityKey) throws DAOException;
    
    
    /**
     * Gets the POW registry matching the icn
     * 
     * @param icn
     * @return
     * @throws DAOException
     */
    public PrisonerOfWar getPOWRegistryByIcn(String icn) throws DAOException;
    
    /**
     * Gets the SHAD registry matching the registry id.
     * 
     * @param entityKey the registry entity key
     * @return
     * @throws DAOException
     */
    public SHAD getSHADRegistryById(EntityKey entityKey) throws DAOException;
    
    
    public Registry getRegistry(Person person, RegistryType type) throws DAOException;
    
    /**
     * Gets the matching RegistryTrait.
     * 
     * @param registry the Registry object
     * @throws DAOException
     */
    public RegistryTrait getRegistryTrait(Registry registry) throws DAOException;
    
    /**
     * Finds the registries matching the criteria.
     * 
     * @param criteria the criteria
     * @return 
     * @throws DAOException Thrown when a persistence exception occurs.
     */
    public List find(SearchQueryInfo criteria) throws DAOException, MaxRecordsExceededException;
    
    /**
     * Adds/updates the registry to the database.
     *  
     * @param registry the registry object
     * @throws DAOException
     */
    public void save(Registry registry) throws DAOException;
    
    /**
     * Gets the Most Recent entry in the Load Registry Summary table for the given RegistryType
     * 
     * @param registryType
     * @return
     * @throws ServiceException
     */
    public RegistryLoadStatistics getMostRecentRegistrySummaryByType(RegistryType registryType) throws DAOException;    

    
}