/********************************************************************
 * Copyright  2004-2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.persistent.person;

// Java Classes
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKey;
import gov.va.med.esr.common.model.person.id.VPIDEntityKey;
import gov.va.med.esr.service.PersonSearchQueryInfo;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.ps.model.PersonVPID;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * This interface defines various DAO methods for searching and retrieving Person objects.
 *
 * @author Martin Francisco & Andrew Pach
 * @version 3.0
 */
public interface PersonDAO extends DAOOperations
{
    /**
     * Find a list of person objects based on the passed in search criteria.
     *
     * @param personSearchQueryInfo The search criteria to filter on
     *
     * @return List of person objects with person Id, social security number, first name, middle name, and last name
     *         populated.
     * @throws DAOException if there was an error trying to find the search results.
     * @throws MaxRecordsExceededException if the maximum number of records allowed was exceeded.
     */
    public List find(PersonSearchQueryInfo personSearchQueryInfo) throws DAOException, MaxRecordsExceededException;
    
    /**
     * Gets a partially built Person object (i.e. the search results) for a VPID Key.
     * @param key The VPID Key
     * @return The partially built Person
     * @throws DAOException If any errors were encountered.
     */
    public Person getSearchResultsByVPID(VPIDEntityKey key) throws DAOException;

    /**
     * Method to retrieve a Person by VPID
     * 
     * @param key The VPID Key
     * @return The fully built Person
     * @throws DAOException If any errors were encountered.
     */
    public Person getByVPID(VPIDEntityKey key) throws DAOException;
    
	/**
	 * Performs a lookup to get VPID using a PersonId
	 * 
	 * @param personId
	 * @return
	 * @throws DAOException
	 */
	public VPIDEntityKey getVPIDByPersonId(PersonIdEntityKey personId)
			throws DAOException;

	/**
	 * Performs a lookup to get PersonId using a VPID
	 * 
	 * @param personId
	 * @return
	 * @throws DAOException
	 */
	public PersonIdEntityKey getPersonIdByVPID(VPIDEntityKey vpid)
			throws DAOException;
	
	/**
	 * converts the given vpids to person_id List<Long> 
	 * @param vpids
	 * @return
	 * @throws DAOException
	 */
    public Map getIdsByVPIDs(List vpids) throws DAOException;
    
    /**
     * converts the given ids to VPIDs List<String> 
     * @param Ids
     * @return
     * @throws DAOException
     */
    public Map getVPIDsbyIds(List Ids) throws DAOException;
    
	/**
	 * Queries ps_person_vpid table, retrieves the PersonVPID given the vpid value
	 * 
	 * @param vpid contains the vpid value
	 * @return
	 */
    public PersonVPID getPersonVPIDByVpidValue(VPIDEntityKey vpid) throws DAOException;
    
    public void updateVOAIndicator (Person person) throws DAOException;
    
}