/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.persistent.messaging;

// Java classes
import java.math.BigDecimal;
import java.util.List;

import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.pagination.SearchQueryInfo;

import gov.va.med.esr.common.model.messaging.ArchivedMessageLogEntry;
import gov.va.med.esr.common.model.messaging.ArchivedMessageLogEntryLite;
import gov.va.med.esr.common.model.messaging.MessageLogEntry;
import gov.va.med.esr.common.model.messaging.MessageLogEntryLite;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.service.MessageLogCriteria;

/**
 * Provides methods to retrieve a message log entry entity
 *
 * Project: Common</br>
 * Created on: 9:47:22 AM </br>
 *
 * @author DNS   LEV, Martin Francisco
 */
public interface ArchivedMessageLogEntryDAO extends DAOOperations {
	
    /**
     * A key to look up a a named query FIND_BY_IDENTIFIER
     */
    public static final String FIND_BY_IDENTIFIER = "arc_messageLogEntryQuery_Identifier";
    
    /**
     * A key to look up a a named query FIND_MESSAGE_LOG_ENTRY_LITE_BY_IDENTIFIER
     */
    public static final String FIND_MESSAGE_LOG_ENTRY_LITE_BY_IDENTIFIER = "arc_messageLogEntryQueryLite_Identifier";
    
	/**
	 * A key to look up a a named query FIND_BY_CONTROL_IDENTIFIER
	 */
	public static final String FIND_BY_CONTROL_IDENTIFIER = "arc_messageLogEntryQuery_ControlIdentifier";

	/**
	 * A key to look up a a named query FIND_BY_CONTROL_IDENTIFIER
	 */
	public static final String FIND_BY_BATCH_CONTROL_IDENTIFIER = "arc_messageLogEntryQuery_BatchControlIdentifier";
	
    /**
     * A name of a param PARAM_IDENTIFIER
     */
    public static final String PARAM_IDENTIFIER = "identifier";
    
	/**
	 * A name of a param PARAM_CONTROL_IDENTIFIER
	 */
	public static final String PARAM_CONTROL_IDENTIFIER = "controlIdentifier";
	
	public static final String PARAM_STATION_NUMBER = "stationNumber";
	
	
	/**
	 * A key to look up a a named query NAMED_QUERY_FIND_BY_PERSON
	 */
	public static final String FIND_BY_PERSON = "arc_messageLogEntryQuery_FindByPerson";
	
	/**
	 * A key to look up a a named query FIND_BY_ELGBTY_STATUS_AND_PERSON
	 */
	public static final String FIND_BY_ELGBTY_STATUS_AND_PERSON = "arc_messageLogEntryQuery_ElgbtyStatusAndPerson";

	/**
	 * A name of a param PARAM_ELGBTY_STATUS
	 */
	public static final String PARAM_ELGBTY_STATUS = "siteReceivedEligibilityStatus";
	
	/**
	 * Query to lookup the facility that sent the recent message
	 */
	public static final String FIND_SITE_LAST_TRANSMITTED_MSG = 
		"arc_messageLogEntryQuery_findSiteLastTransmittedMsg";
	/**
	 * Query Parameter person id
	 */
	public static final String PARAM_PERSON_ID = "personId"; 
	
	/**
	 * Query to lookup rows based on the transmitted date and number of times the message has been retransmitted
	 */
	public static final String FIND_BY_RETRANSMISSION_WAIT_PERIOD_AND_COUNT = 
	    "arc_messageLogEntryQuery_FindByRetransmissionWaitPeriodAndCount";

	/**
	 * Query to lookup rows based on the transmitted date and number of times the message has been retransmitted
	 * for the EGT(MFN-ZEG) messages
	 */
	public static final String FIND_BY_RETRANSMISSION_WAIT_PERIOD_AND_COUNT_FOR_EGT = 
	    "arc_messageLogEntryQuery_FindByRetransmissionWaitPeriodAndCountForEGT";
	
	public static final String FIND_INBOUND_MESSAGES = 
	    "arc_messageLogEntryQuery_findInboundMessages";
	
	public static final String PARAM_RETRANSMISSION_COUNT = "retransmissionCount";
	
	public static final String PARAM_RETRANSMISSION_WAIT_PERIOD= "retransmissionWaitPeriod";
	
	public static final String PARAM_MESSAGE_STATUS= "messageStatus";
	
	public static final String PARAM_MESSAGE_TYPE= "messageType";
	
	public static final String FIND_VOA_AA_ACK_BY_PERSON_ID = "arc_messageLogEntryQuery_findVoaAaAckLogEntryByPersonId";
	
	
    /**
     * Gets the message log entry for the message with the specified identifier.
     * 
     * @param controlIdentifier The identifier.
     * @return The message log entry for the message with the specified identifier.
     * @throws DAOException Thrown when a persistence exception occurs.
     */
    public ArchivedMessageLogEntry getById(BigDecimal identifier) throws DAOException;
    
    /**
     * Gets the message log entry lite for the message with the specified identifier.
     * 
     * @param controlIdentifier The identifier.
     * @return The message log entry lite for the message with the specified identifier.
     * @throws DAOException Thrown when a persistence exception occurs.
     */
    public ArchivedMessageLogEntryLite getMessageLogEntryLiteById(BigDecimal identifier) throws DAOException;
    
	/**
	 * Gets the message log entry for the message with the specified control
	 * identifier.
	 * 
	 * @param controlIdentifier The control identifier.
	 * @return The message log entry for the message with the specified control identifier.
	 * @throws DAOException Thrown when a persistence exception occurs.
	 */
	public ArchivedMessageLogEntry getByControlId(String controlIdentifier) throws DAOException;

	/**
	 * Gets the message log entry for the message with the specified control
	 * identifier.
	 * 
	 * @param controlIdentifier The control identifier.
	 * @return The message log entry for the message with the specified control identifier.
	 * @throws DAOException Thrown when a persistence exception occurs.
	 */
	public ArchivedMessageLogEntry getByBatchControlId(String controlIdentifier) throws DAOException;
	
	/**
	 * Finds the message log entries for a person.
	 * 
	 * @param person The person whose message log entries will be returned.
	 * @return A list of message log entries for the specified person.
	 * @throws DAOException Thrown when a persistence exception occurs.
	 */
	public List find(Person person) throws DAOException;
	
    
    /**
     * Finds the message log entries for a criteria.
     * 
     * @param criteria The criteria for retrieving the message log entries will be returned.
     * @return A list of message log entries for the specified criteria.
     * @throws DAOException Thrown when a persistence exception occurs.
     * @throws MaxRecordsExceededException Thrown when a number of records exceeded.
     */
    public List find(SearchQueryInfo criteria) throws DAOException, MaxRecordsExceededException;
    
	/**
	 * Finds the message log entries for a person and the specific received eligibility status 
	 * 
	 * @param person The person whose message log entries will be returned.
	 * @param eligibilityStatus The received eligibility status.
	 * @return The message log entry for the specified person and received eligibility status
	 * @throws DAOException Thrown when a persistence exception occurs.
	 */
	public ArchivedMessageLogEntry findByReceivedEligiblityStatus(Person person, String eligibilityStatus ) throws DAOException;
	
	/**
	 * Find the Facility from whih the last message has been received and successfully processed
	 * 
	 * @param personId
	 * @return
	 * @throws DAOException
	 */
	public String findSiteLastTransmittedMsg(EntityKey personId) throws DAOException; 
	
	/**
	 * The message is not in an error status or completed.
	 * The Date of Transmission plus the Re-Transmit Wait Period<retransmissionCount>
	 * is less than or equal to the Current Date.
	 * The message has been tranmitted less than the <retransmissionCount>
	 * 
	 * 
	 * @param retransmissionWaitPeriod
	 * @param retransmissionCount
	 * @return
	 * @throws DAOException
	 */
	public List findByRetransmissionWaitPeriodAndCount(Integer retransmissionWaitPeriod,
            Integer retransmissionCount, String messageStatusCode, boolean isEGTRetransmission) throws DAOException;

	/** Specific query to determine if already processed an inbound message */
	public boolean hasProcessedInboundMessage(String messageControlNumber, String stationNumber) throws DAOException;
	
	
    /**
     * find message log entries for VOA that has AA Ack sent, given a person id, in decending datetime order (latest first)
     * 
     * @param identifier person id.
     * @return list of log entries
     * @throws ServiceException
     */	
	public List findVoaAaAckLogEntryByPersonId(EntityKey personId) throws DAOException; 
}