/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.esr.common.persistent.history;

// Java classes
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.esr.service.EEResultInfo;

import java.util.Date;
import java.util.List;

// Common classes

/** A generic interface to provide standard operations for retrieving history for an object
 * 
 * @author DNS   CHENJ2
 */
public interface RulesEnrollmentHistoryDAO extends HistoryDAO {

    public Date getEffectiveDateForEarliestVerifiedUnlessCancelledOrRejectedBelowEnrollmentThreshold(EntityKey personId)
    throws DAOException;
    
	// For ee calculcation
	
    public Date getEnrollmentApplicationDateForEarliestSpecifiedStatusUnlessCancelledOrRejectedBelowThreshold(EntityKey personId) throws DAOException ;    
    
	/**
     * Returns enrollment data for the immediately prior enrollment.  Only EnrollmentDetermination
     * data is populated for person.
	 * @param personId
	 * @return
	 * @throws DAOException
	 */
    public Person getPersonForPriorEnrollment(EntityKey personId) throws DAOException ;
	
	 /**
 	  * For ee calculation: returns the application date corresponding to the more recent
      * unverified enrollment
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public Date getAppDateForMostRecentUnverifiedEnrollment(EntityKey personId) throws DAOException;
	
	 /**
	  * For ee calculation: returns the effective date for the earliest enrollment
      * with not null effective date.

	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public Date getFirstNotNullEffDate(EntityKey personId) throws DAOException;

	// For Batch Process
	 /**
	  * Returns the enrollment status and application date for the earliest enrollment.
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public Object[] getStatusAndAppDateForEarliest(EntityKey personId) throws DAOException;
	
	 /**
	  * Returns the codes for priority group and priority subGroup for the most recent enrollment.
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public String[] getPriorityGrpsForMostRecent(EntityKey personId) throws DAOException;
	 
	 /**
   	 * Returns the Enrollment Date from the earliest enrollment with Verified Status
	 * UNLESS a Cancel Decline Status is encountered, in which case use
	 * subsequent (one more recent than cancel). 

	 * @param personId
	 * @return
	 * @throws DAOException
	 */
	 public Date getEnrollmentDateForEarliestVerifiedUnlessCancelled(EntityKey personId)
	 	throws DAOException;

	 /**
	  *	Get the enrollment determination for the Earliest Verified and not Canceled Record 
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public EnrollmentDetermination getEnrollmentDeterminationForEarliestVerifiedUnlessCancelledOrRejectedBelowEnrollmentGroupThreshold(EntityKey personId)
	 	throws DAOException;
	 
	 /**
	  * Returns true if there is ANY application date that is prior to EGT Setting effective date
	  */
	 public boolean isApplicationDatePriortoEGTEffectiveDate(EntityKey personId)
	 		throws DAOException;
	 
	 /**
	  * Returns true if Verified Enrollment exists in the Past with the given Eligibility Type Code
	  * @param personId
	  * @param eligiblityTypeCode
	  * @return
	  * @throws DAOException
	  */
	 public boolean isVerifiedEnrollmentExistsForEligibilityCode(EntityKey personId, 
			 String eligiblityTypeCode) throws DAOException;
	 
	 /**
	  * Returns earliest verified enrollment effective date with eligibility code AG and 
	  * AOE location DMZ
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public Date getEarliestEnrollmentEffectiveDatewithAOWatDMZ(EntityKey personId) 
	 		throws DAOException;
	 
	 /**
	  * Returns true if verified enrollment exists with Service Connection Award Percent >= given Value
	  * @param personId
	  * @param percent
	  * @return
	  * @throws DAOException
	  */
	 public boolean isVerifiedEnrollmentExistsforSvcConnPercent(EntityKey personId, 
			 Integer percent) throws DAOException;
	 
	 
     /**
      * Returns income year from MT linked to a verified enrollment
      *  
      * @param personId
      * @return
      * @throws DAOException
      */
	 /**
	     * @deprecated Fix for CR_8675 modified to return multiple incomeYears in the below method.
	 */
	 public Integer getIncomeYearForVerifiedMT(EntityKey personId) throws DAOException;      
     
     public List getIncomeYearsForVerifiedMT(EntityKey personId) throws DAOException;
     
     /**
      * CodeCR10023 for CR 9803 updates
      * 
      * @param personId
      * @return
      * @throws DAOException
      */
     public List getIncomeYearsForVerifiedP8(EntityKey personId) throws DAOException;
     
     /**
      * CodeCR10023 for CR 9803 updates
      * 
      * @param personId
      * @return
      * @throws DAOException
      */
	 public String[] getMostRecentVerifiedEnrollmentPriorityGroups(EntityKey personId) throws DAOException;
     

	 /**
	  * CodeCR10023 for CR 9803 updates
	  * 
	  * @param personId
	  * @param relaxPercent
	  * @return
	  * @throws DAOException
	  */
	 public boolean hasIncomeTestMeetingP8RecheckCriteria(EntityKey personId, float relaxPercent) throws DAOException;	 
	 

	 /**
	  * CodeCR10023 for CR 9803 updates
	  * 
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public List getHistoricalIncomeYearsForVerifiedMT(EntityKey personId) throws DAOException; 
	 
     
	 /**
	  * Returns true if verified enrollment exists for Combat Veteran eligibility code
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public boolean isVerifiedEnrollmentExistsforCombatVeteran(EntityKey personId) 
	 		throws DAOException;

	 /**
	  * Return most recent enrollment data for continuous enrollment 
	  * (Exclude the status "'6','15','16','17','18','19','20','21','23'"
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	public EEResultInfo getMostRecentEnrollmentforContinuousEnrollment(EntityKey personId) throws DAOException;

	 /**
	  * Returns true if verified enrollment exists for person
	  * @param personId
	  * @return
	  * @throws DAOException
	  */
	 public boolean isVerifiedEnrollmentExists(EntityKey personId) throws DAOException;

     /**
      * Returns the most recent non-null priority group
      * 
      * @param personId
      * @return
      * @throws DAOException
      */
     public String getMostRecentNonNullPriorityCode(EntityKey personId) throws DAOException;      
     
     public String getMostRecentNonNullPriorityLevelByDate(EntityKey personId, Date beforeDate) throws DAOException;

     /**
      * Return true if the person has ANY enrollment record before the specified date
      * @param personId
      * @param date
      * @return
      * @throws DAOException
      */
     public boolean hasAnyEnrollmentRecordBeforeDate(EntityKey personId, Date beforeDate) throws DAOException;
     /**
      * Returns the enrollment priority code and status code before PH was added. Used when letter 630D is triggered
      * @param personId
      * @return String[] with enrollment priority code and enrollment status code. Return null if not found.
      */
     public String[] getEnrollmentPriorityBeforePH(EntityKey personId) throws DAOException;
     
     /**
	  * Returns true if Verified Enrollment exists in the Past with Medicaid Eligibility
	  * @param personId
	  * @param eligiblityTypeCode
	  * @return
	  * @throws DAOException
	  */
	 public boolean isVerifiedEnrollmentExistsForMedicaidEligibility(EntityKey personId) throws DAOException;
	 
	 /**
	  * Returns the history of all enrollment determination records for this person
	  * @param personId 
	  * @return
	  * @throws DAOException
	  */
	 public List getPersonEnrollmentDeterminationHistory(EntityKey personId)
		throws DAOException;
	 
	 public boolean getPriorToEnrDetermHistory(EntityKey personId, Date beforeDate, Date recModifiedDate) throws DAOException;
}
