package gov.va.med.esr.common.model.person;

import gov.va.med.fw.model.AbstractKeyedEntity;

public class FullyQualifiedIdentity extends AbstractKeyedEntity {

/*
 * This is an object holding fully qualified identitier - ICN or IEN, but not both at the same time.
 * This class is added primarily for the IdM 1305 search by fully qualified identity:
 * ICN Should be in the form:
 *   <id extension="icn value^identifier type" root="2.16.840.1.113883.4.349"/>
 *   Example:
 *   <id extension="1030811953V001173^NI" root="2.16.840.1.113883.4.349"/>
 * IEN Should be in the form:
 *   <id extension="ien value^category type^assigning location/station number^assigning authority" root="2.16.840.1.113883.4.349"/>
 *   Example:
 *   <id extension="1305236^PI^500^USVHA" root="2.16.840.1.113883.4.349"/>
 * 
 */
	
	/* for the definitions for these properties, please see ICD "HL7 Table 0203 - Identifier Types" */
	
	/**
	 * 
	 */
	private static final long serialVersionUID = -662856570689099399L;
	private static final String NATIONAL_ICN = "NI";
	private static final String PATIENT_IDENTIFIER = "PI";
	private static final String US_VHA = "USVHA";
		
	/*
	 * idValue can be ICN (short vpid) or IEN (dfn)
	 */
	private String idValue = null;
	
	/*
	 * "NI" for ICN, "PI" for IEN
	 */
	
	private String identifierType = null;
	
	/*
	 * assigning location/station number, e.g. "200ESR"
	 */
	private String stationNumber = null; 
	
	/*
	 * e.g. "USVHA"
	 */
	private String assigningAuthority = null;

	
	private String idmExtension = null;
	/**
	 * set the ICN with a short vpid value
	 * @param icn - short vpid value
	 */
	
	
	public FullyQualifiedIdentity(String shortVPID)
	{
		this.idValue = shortVPID;
		this.identifierType = NATIONAL_ICN;
		this.idmExtension = shortVPID + "^NI";
	}

	/**
	 * set the IEN with a dfn value and station number
	 * @param ien - dfn value
	 * @param stationNumber - e.g. "200ESR"
	 */
	public FullyQualifiedIdentity(String dfn, String stationNumber)
	{
		this.idValue = dfn;
		this.identifierType = PATIENT_IDENTIFIER;
		this.stationNumber = stationNumber;
		this.assigningAuthority = US_VHA;
		this.idmExtension = dfn + "^PI^" + stationNumber + "^USVHA";
	}
	
	public FullyQualifiedIdentity(String idValue, String identifierType, String stationNumber, String assigningAuthority)
	{
		this.idValue = idValue;
		this.identifierType = identifierType;
		this.stationNumber = stationNumber;
		this.assigningAuthority = assigningAuthority;
		this.idmExtension = idValue +"^"+ identifierType +"^"+ stationNumber +"^"+ assigningAuthority;
	}

	public String getAssigningAuthority() {
		return assigningAuthority;
	}

	public String getIdentifierType() {
		return identifierType;
	}

	public String getIdValue() {
		return idValue;
	}

	public String getStationNumber() {
		return stationNumber;
	}

	public String getIdmExtension() {
		return this.idmExtension;
	
	}
	
	
}
