/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.model.lookup;

// Java classes

// Library classes
import gov.va.med.fw.model.lookup.AbstractCode;
import gov.va.med.fw.model.lookup.AbstractNamedActiveLookup;
import gov.va.med.fw.util.DateUtils;

import java.util.List;
import java.util.Arrays;
 
// ESR classes

/**
 * This is one of the classes of the Business Object Model (BOM).
 * This class represents a Period of Service.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public class ServicePeriod extends AbstractNamedActiveLookup
{
    private static final long serialVersionUID = -330943289461503827L;

    /* Korean Code */
    public static final Code CODE_KOREAN = new Code("0");

    /* World War I Code */
    public static final Code CODE_WORLD_WAR_I = new Code("1");

    /* World War II Code */
    public static final Code CODE_WORLD_WAR_II = new Code("2");

    /* Spanish American Code */
    public static final Code CODE_SPANISH_AMERICAN = new Code("3");

    /* Pre-Korean Code */
    public static final Code CODE_PRE_KOREAN = new Code("4");

    /* Post-Korean Code */
    public static final Code CODE_POST_KOREAN = new Code("5");

    /* Vietnam Era Code */
    public static final Code CODE_VIETNAM_ERA = new Code("7");

    /* Post-Vietnam Code */
    public static final Code CODE_POST_VIETNAM = new Code("8");

    /* Other or None Code */
    public static final Code CODE_OTHER_OR_NONE = new Code("9");

    /* Army Code */
    public static final Code CODE_ARMY = new Code("A");

    /* Navy Code */
    public static final Code CODE_NAVY = new Code("B");

    /* Air Force Code */
    public static final Code CODE_AIR_FORCE = new Code("C");

    /* Coast Guard Code */
    public static final Code CODE_COAST_GUARD = new Code("D");

    /* Other USPHS Beneficiaries Code */
    public static final Code CODE_OTHER_USPHS_BENEFICIARIES = new Code("H");

    /* Office of Workers Comp Code */
    public static final Code CODE_OFFICE_OF_WORKERS_COMP = new Code("J");

    /* Other Reimburs Non-Vet Code */
    public static final Code CODE_OTHER_REIMBURS_NON_VET = new Code("P");

    /* Retired Uniformed Forces Code */
    public static final Code CODE_RETIRED_UNIFORMED_FORCES_CODE = new Code("E");

    /* Medical Remedial Enlist Code */
    public static final Code CODE_MEDICAL_REMEDIAL_ENLIST = new Code("F");

    /* Merchant Seamen USPHS Code */
    public static final Code CODE_MERCHANT_SEAMEN_USPHS = new Code("G");

    /* Beneficiaries - Foreign Gov Code */
    public static final Code CODE_BENEFICIARIES_FOREIGN_GOV = new Code("M");

    /* Humanitarian (Non-Vet) Code */
    public static final Code CODE_HUMANITARIAN_NON_VET = new Code("N");

    /* Donors (Non-Vet) Code */
    public static final Code CODE_DONORS_NON_VET = new Code("R");

    /* Other (Non-Veterans) Code */
    public static final Code CODE_OTHER_NON_VET = new Code("T");

    /* Czechoslovakia/Poland SVC Code */
    public static final Code CODE_CZECHOSLOVAKIA_POLAND_SVC = new Code("W");

    /* CAV/NPS Code */
    public static final Code CODE_CAV_NPS = new Code("Y");

    /* Champva - Spouse, Child Code */
    public static final Code CODE_CHAMPVA_SPOUSE_CHILD = new Code("U");

    /* Special Studies (Non-Vet) Code */
    public static final Code CODE_SPECIAL_STUDIES_NON_VET = new Code("S");

    /* Job Corps/Peace Corps Code */
    public static final Code CODE_JOB_CORPS_PEACE_CORPS = new Code("K");

    /* Merchant Marine Code */
    public static final Code CODE_MERCHANT_MARINE = new Code("Z");

    /* Observation/Examination Code */
    public static final Code CODE_OBSERVATION_EXAMINATION = new Code("I");

    /* Railroad Retirement Code */
    public static final Code CODE_RAILROAD_RETIREMENT = new Code("L");

    /* Champus Restore Code */
    public static final Code CODE_CHAMPUS_RESTORE = new Code("O");

    /* Champus Code */
    public static final Code CODE_TRICARE_CHAMPUS = new Code("V");

    /* Other Federal - Dependent Code */
    public static final Code CODE_OTHER_FEDERAL_DEPENDENT = new Code("Q");

    /* Operation Desert Shield Code */
    public static final Code CODE_OPERATION_DESERT_SHIELD = new Code("6");

    /* Persian Gulf War Code */
    public static final Code CODE_PERSIAN_GULF_WAR = new Code("X");

	public static class Code extends AbstractCode {
		private static final long serialVersionUID = 2616656799203290724L;
        private Code(String code) {	super(code); }
		public static Code getByCode(String code) {
			return (Code) getCode(Code.class, code);
		}
	}	

//	5911[UC28.3.3.2] 
//		 The Period of Service (POS) the actor must enter is contingent on the specific Non-Veteran Eligibility Code they have entered.  The actor can only select one POS. The system will present the POS that are applicable as follows:
//
//	Allied Veteran
//	Allowable POS are:
//	CAV/NPS
//	CZECHOSLOVAKIA/POLAND SVC
//	OTHER NON-VETERANS
//	OTHER OR NONE
//	RAILROAD RETIREMENT
    static private String[] alliedVeteranCodes = new String[]
	{
    	CODE_CAV_NPS.getCode(),
		CODE_CZECHOSLOVAKIA_POLAND_SVC.getCode(),
		CODE_OTHER_NON_VET.getCode(),
		CODE_OTHER_OR_NONE.getCode(),
		CODE_RAILROAD_RETIREMENT.getCode()
	};
	
//	Other Federal Agency
//	Allowable POS are:
//	AIR FORCE--ACTIVE DUTY
//	ARMY--ACTIVE DUTY
//	BENEFICIARIES-FOREIGN GOV
//	COAST GUARD--ACTIVE DUTY
//	DONORS (NON-VET)
//	HUMANITARIAN (NON-VET)
//	JOB CORPS/PEACE CORPS
//	MEDICAL REMEDIAL ENLIST
//	MERCHANT SEAMEN--USPHS
//	NAVY, MARINE--ACTIVE DUTY
//	OPERATION DESERT SHIELD-ACTIVE DUTY FROM ODS
//	OTHER FEDERAL  DEPENDENT- DOD DEPENDENTS AT VA FAC
//	OTHER NON-VETERANS
//	OTHER OR NONE
//	OTHER USPHS BENEFICIARIES
//	RAILROAD RETIREMENT
//	RETIRED, UNIFORMED FORCES
//	WORLD WAR II
    static private String[] otherFederalAgencyCodes = new String[]
	{
    		CODE_AIR_FORCE.getCode(),
    		CODE_ARMY.getCode(),
			CODE_BENEFICIARIES_FOREIGN_GOV.getCode(),
			CODE_COAST_GUARD.getCode(),
			CODE_DONORS_NON_VET.getCode(),
			CODE_HUMANITARIAN_NON_VET.getCode(),
			CODE_JOB_CORPS_PEACE_CORPS.getCode(),
			CODE_MEDICAL_REMEDIAL_ENLIST.getCode(),
			CODE_MERCHANT_SEAMEN_USPHS.getCode(),
			CODE_NAVY.getCode(),
			CODE_OPERATION_DESERT_SHIELD.getCode(),
			CODE_OTHER_FEDERAL_DEPENDENT.getCode(),
			CODE_OTHER_NON_VET.getCode(),
			CODE_OTHER_OR_NONE.getCode(),
			CODE_OTHER_USPHS_BENEFICIARIES.getCode(),
			CODE_RAILROAD_RETIREMENT.getCode(),
			CODE_RETIRED_UNIFORMED_FORCES_CODE.getCode(),
			CODE_WORLD_WAR_II.getCode()
	};
    
//	Collateral of Vet
//	Allowable POS are:
//	CHAMPVA--SPOUSE, CHILD
//	OTHER NON-VETERANS
//	OTHER OR NONE
    static private String[] collateralOfVetCodes = new String[]
	{
    		CODE_CHAMPVA_SPOUSE_CHILD.getCode(),
			CODE_OTHER_NON_VET.getCode(),
			CODE_OTHER_OR_NONE.getCode()
	};
    
//    CHAMPVA
//	Allowable POS are:
//	CHAMPVA--SPOUSE, CHILD
//	KOREAN
//	MERCHANT MARINE
//	OTHER OR NONE
//	PERSIAN GULF WAR
//	POST-KOREAN
//	POST-VIETNAM
//	PRE-KOREAN
//	SPANISH AMERICAN
//	VIETNAM ERA
//	WORLD WAR I
//	WORLD WAR II
    static private String[] champvaCodes = new String[]
	{
    		CODE_CHAMPVA_SPOUSE_CHILD.getCode(),
			CODE_KOREAN.getCode(),
			CODE_MERCHANT_MARINE.getCode(),
			CODE_OTHER_OR_NONE.getCode(),
			CODE_PERSIAN_GULF_WAR.getCode(),
			CODE_POST_KOREAN.getCode(),
			CODE_POST_VIETNAM.getCode(),
			CODE_PRE_KOREAN.getCode(),
			CODE_SPANISH_AMERICAN.getCode(),
			CODE_VIETNAM_ERA.getCode(),
			CODE_WORLD_WAR_I.getCode(),
			CODE_WORLD_WAR_II.getCode()
	};
    
//    Employee
//	Allowable POS are:
//	OFFICE OF WORKERS COMP
//	OTHER NON-VETERANS
//	OTHER OR NONE
    static private String[] employeeCodes = new String[]
	{
    		CODE_OFFICE_OF_WORKERS_COMP.getCode(),
			CODE_OTHER_NON_VET.getCode(),
			CODE_OTHER_OR_NONE.getCode()
	};

//    Sharing Agreement
//	Allowable POS are:
//	AIR FORCE--ACTIVE DUTY
//	ARMY--ACTIVE DUTY
//	BENEFICIARIES-FOREIGN GOV
//	CAV/NPS
//	COAST GUARD--ACTIVE DUTY
//	CZECHOSLOVAKIA/POLAND SVC
//	JOB CORPS/PEACE CORPS
//	NAVY, MARINE--ACTIVE DUTY
//	OFFICE OF WORKERS COMP.
//	OTHER NON-VETERANS
//	OTHER OR NONE
//	OTHER REIMBURS. (NON-VET)
//	OTHER USPHS BENEFICIARIES
//	SPECIAL STUDIES (NON-VET)
//	WORLD WAR II
    static private String[] sharingAgreementCodes = new String[]
	{
    		CODE_AIR_FORCE.getCode(),
    		CODE_ARMY.getCode(),
			CODE_BENEFICIARIES_FOREIGN_GOV.getCode(),
			CODE_CAV_NPS.getCode(),
			CODE_COAST_GUARD.getCode(),
			CODE_CZECHOSLOVAKIA_POLAND_SVC.getCode(),			
			CODE_JOB_CORPS_PEACE_CORPS.getCode(),
			CODE_NAVY.getCode(),
			CODE_OFFICE_OF_WORKERS_COMP.getCode(),
			CODE_OTHER_NON_VET.getCode(),
			CODE_OTHER_OR_NONE.getCode(),
			CODE_OTHER_REIMBURS_NON_VET.getCode(),
			CODE_OTHER_USPHS_BENEFICIARIES.getCode(),
			CODE_SPECIAL_STUDIES_NON_VET.getCode(),
			CODE_WORLD_WAR_II.getCode()
	};

//    TRICARE
//	Allowable POS are:
//	AIR FORCE--ACTIVE DUTY
//	ARMY--ACTIVE DUTY
//	BENEFICIARIES-FOREIGN GOV
//	CAV/NPS
//	TRICARE #####
//	CHAMPUS PTS AT VA FACILITY
//	COAST GUARD--ACTIVE DUTY
//	NAVY, MARINE--ACTIVE DUTY
//	OTHER NON-VETERANS
//	WORLD WAR II  
    static private String[] tricareCodes = new String[]
	{
    		CODE_AIR_FORCE.getCode(),
    		CODE_ARMY.getCode(),
			CODE_BENEFICIARIES_FOREIGN_GOV.getCode(),
			CODE_CAV_NPS.getCode(),
			CODE_TRICARE_CHAMPUS.getCode(),
			CODE_COAST_GUARD.getCode(),
			CODE_NAVY.getCode(),
			CODE_OTHER_NON_VET.getCode(),
			CODE_WORLD_WAR_II.getCode()
	};

    /**
     * Default constructor.
     */
    public ServicePeriod()
    {
        super();
    }
    
    public boolean isSubSetForNonVeteranCode(String code)
    {
    	if (code == null)
    		return false;
    	
    	if (code.equals(EligibilityType.ALLIED_VETERAN.getCode()))
    	{
    		List list = Arrays.asList(alliedVeteranCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.OTHER_FEDERAL_AGENCY.getCode()))
    	{
    		List list = Arrays.asList(otherFederalAgencyCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.COLLATERAL_OF_VETERAN.getCode()))
    	{
    		List list = Arrays.asList(collateralOfVetCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.CHAMPVA.getCode()))
    	{
    		List list = Arrays.asList(champvaCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.EMPLOYEE.getCode()))
    	{
    		List list = Arrays.asList(employeeCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.SHARING_AGREEMENT.getCode()))
    	{
    		List list = Arrays.asList(sharingAgreementCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
    	else if (code.equals(EligibilityType.TRICARE_CHAMPUS.getCode()))
    	{
    		List list = Arrays.asList(tricareCodes);
    		if (list.contains(this.getCode()))
    			return true;
    		else
    			return false;
    	}
   	
    	return false;
    }
    
    public String getNameAndDateRange () {
        StringBuffer nameDate = new StringBuffer(this.getName());
        
        String start = (this.getStartDate() != null) ? DateUtils.format(this.getStartDate(),DateUtils.MMDDYYYY) : null;
        String end = (this.getStartDate() != null) ? DateUtils.format(this.getEndDate(),DateUtils.MMDDYYYY) : null;
        
        if(start != null) {
            if(end == null) end = "...";
            nameDate.append(" - (" + start + " to " + end + ")");
        }
        return nameDate.toString();
    }
}

