/*******************************************************************************
 * Copyright  2005-2006 VHA. All rights reserved
 ******************************************************************************/
package gov.va.med.esr.common.model.lookup;

// Java classes

// Library classes
import org.apache.commons.lang.builder.ToStringBuilder;

// Framework Classes
import gov.va.med.fw.model.lookup.AbstractLookup;
import gov.va.med.fw.model.lookup.AbstractCode;
import gov.va.med.fw.util.StringUtils;

// ESR classes

/**
 * This is one of the classes of the Business Object Model (BOM). This class represents a Country.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public class Country extends AbstractLookup
{
    // Note that the codes defined here use the "ALPHA3CODE" column and not the
    // "NUMERICCODE" column.

    /**
     * An instance of serialVersionUID
     */
    private static final long serialVersionUID = 7639817666922345899L;

    /* USA Code */
    public static final Code CODE_USA = new Code("USA");
    
    /* Cancda Code */
    public static final Code CODE_CAN = new Code("CAN");
    
    /* Mexico Code */
    public static final Code CODE_MEX = new Code("MEX");
    
    public static final Code CODE_PHILIPPINES = new Code("PHL");

    public static class Code extends AbstractCode
    {

        /**
         * An instance of serialVersionUID
         */
        private static final long serialVersionUID = -3147087186714279051L;

        private Code(String code)
        {
            super(code);
        }

        public static Code getByCode(String code)
        {
            return (Code)getCode(Code.class, code);
        }
    }

    private String shortName;

    private String numericCode;

    private String alpha3Code;

    private String fipsCode;

    private String postalName;

    /**
     * Default constructor.
     */
    public Country()
    {
        super();
    }

    /**
     * Returns whether the address is a US Address.  Null will return true.
     * @return True if the country is United States or false if not.
     */
    public static boolean isUSAddress(String countryCode)
    {
        // Assume that Country is USA when no country exists since most countries will be
        // US countries.  Although the UI will always set the country, Messaging might not.
        if ((StringUtils.isBlank(countryCode)) || (Country.CODE_USA.getName().equals(countryCode)))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Returns whether this Country is a US Address.  This method delegates to the
     * isUSAddress(countryCode) method.
     *
     * @return True if the country is United States or false if not.
     */
    public boolean isUSAddress()
    {
        return isUSAddress(getAlpha3Code());
    }

    /**
     * Returns the unique ALPHA3 code for this object.
     *
     * @return The unique ALPHA3 code for this object.
     */
    public String getAlpha3Code()
    {
        return this.alpha3Code;
    }

    /**
     * Sets the unique ALPHA3 code for this object.
     *
     * @param alpha3Code The unique ALPHA3 code for this object.
     */
    private void setAlpha3Code(String alpha3Code)
    {
        this.alpha3Code = alpha3Code;
    }

    /**
     * Returns the unique FIPS code for this object.
     *
     * @return The unique FIPS code for this object.
     */
    public String getFipsCode()
    {
        return this.fipsCode;
    }

    /**
     * Sets the unique FIPS code for this object.
     *
     * @param fipsCode The unique FIPS code for this object.
     */
    private void setFipsCode(String fipsCode)
    {
        this.fipsCode = fipsCode;
    }

    /**
     * Returns the unique numeric code for this object.
     *
     * @return The unique numeric code for this object.
     */
    public String getNumericCode()
    {
        return this.numericCode;
    }

    /**
     * Sets the unique numeric code for this object.
     *
     * @param numericCode The unique numeric code for this object.
     */
    private void setNumericCode(String numericCode)
    {
        this.numericCode = numericCode;
    }

    /**
     * Returns the unique USPS name for this object.
     *
     * @return The unique USPS name for this object.
     */
    public String getPostalName()
    {
        return this.postalName;
    }

    /**
     * Sets the unique USPS name for this object.
     *
     * @param postalName The unique USPS name for this object.
     */
    public void setPostalName(String postalName)
    {
        this.postalName = postalName;
    }

    /**
     * Returns the unique short name for this object.
     *
     * @return The unique short name for this object.
     */
    public String getShortName()
    {
        return this.shortName;
    }

    /**
     * Sets the unique short name for this object.
     *
     * @param shortName The unique short name for this object.
     */
    public void setShortName(String shortName)
    {
        if (shortName != null)
        {
            shortName = shortName.toUpperCase();
        }
        this.shortName = shortName;
    }

    /*
      * (non-Javadoc)
      *
      * @see gov.va.med.fw.model.AbstractEntity#buildToString(org.apache.commons.lang.builder.ToStringBuilder)
      */
    protected void buildToString(ToStringBuilder builder)
    {
        super.buildToString(builder.append("alpha3Code", this.alpha3Code).append(
            "fipsCode", this.fipsCode).append("numericCode", this.numericCode)
            .append("postalName", this.postalName).append("shortName",
            this.shortName));
    }

    /*
      * (non-Javadoc)
      *
      * @see java.lang.Object#finalize()
      */
    protected void finalize() throws Throwable
    {
        super.finalize();

        this.alpha3Code = null;
        this.fipsCode = null;
        this.numericCode = null;
        this.postalName = null;
        this.shortName = null;
    }

    /*
      * (non-Javadoc)
      *
      * @see gov.va.med.esr.common.model.lookup.LookupType#getLookupDescription()
      */
    public String getDescription()
    {
		return getShortName();
	}
}