/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.model.financials;

//----------------------------------------------------------------------------
//  Import java classes
//----------------------------------------------------------------------------
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.lang.Validate;
import org.apache.commons.lang.builder.ToStringBuilder;

import gov.va.med.fw.model.AbstractKeyedEntity;
import gov.va.med.fw.util.DateUtils;

import gov.va.med.esr.common.model.SiteYear;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;

/**
 * @author Commons Team
 * @version 1.0
 */
public class PatientVisitSummary
    extends AbstractKeyedEntity {
    /**
     * An instance of serialVersionUID
     */
    private static final long serialVersionUID = 2712584199371463303L;

    private Person            person           = null;
    private SiteYear          siteYear         = null;
    private VAFacility        facilityVisited  = null;
    private Integer           incomeYear       = null;

    /**
     * Date of last visit is defined as the date of the veteran's last visit.
     * This is captured by veteran, by site for an income yearFormat is a date
     */
    private Date              lastVisitDate    = null;

    /**
     * # Of inpatient days since last MT is defined as the Number of inpatient
     * stays by this veteran for this income year. This attribute is captured by
     * veteran, by income year, by siteFormat is a number
     */
    private Integer           inpatientDays    = null;

    /**
     * # Of outpatient visits since last MT is defined as the Number of
     * outpatient visits by this veteran for this income year. This attribute is
     * captured by veteran, by income year, by siteFormat is a number
     */
    private Integer           outpatientDays   = null;

    /**
     * Creates a new PatientVisitSummary object.
     */
    public PatientVisitSummary() {
        super();
    }

    //----------------------------------------------------------------------------
    // Public method(s)
    //----------------------------------------------------------------------------    
    public Person getPerson() {
        return this.person;
    }

    public void setPerson(Person person) {
        this.validateOwner(this.person, person);
        this.person = person;
    }

    private SiteYear getSiteYear() {
        return siteYear;
    }

    private void setSiteYear(SiteYear siteYear) {
        this.siteYear = siteYear;
    }

    public VAFacility getFacilityVisited() {
        return this.facilityVisited;
    }

    public void setFacilityVisited(VAFacility facilityVisited) {
        this.facilityVisited = facilityVisited;
    }

    public Integer getIncomeYear() {
        return this.incomeYear;
    }

    public void setIncomeYear(Integer incomeYear) {
        this.incomeYear = incomeYear;
    }

    public Date getLastVisitDate() {
        return this.lastVisitDate;
    }

    public void setLastVisitDate(Date lastVisitDate) {
        this.lastVisitDate = lastVisitDate;
    }

    public Integer getInpatientDays() {
        return this.inpatientDays;
    }

    public void setInpatientDays(Integer inpatientDays) {
        this.inpatientDays = inpatientDays;
    }

    public Integer getOutpatientDays() {
        return this.outpatientDays;
    }

    public void setOutpatientDays(Integer outpatientDays) {
        this.outpatientDays = outpatientDays;
    }

    /**
     * (non-Javadoc)
     * @see gov.va.med.fw.model.AbstractKeyedEntity#buildToString(org.apache.commons.lang.builder.ToStringBuilder)
     */
    protected void buildToString(ToStringBuilder builder) {
        super.buildToString(builder);
        builder.append("incomeYear", this.incomeYear);
        builder.append("lastVisitDate", this.lastVisitDate);
        builder.append("inpatientDays", this.inpatientDays);
        builder.append("outpatientDays", this.outpatientDays);
    }

    /**
     * Given the collection of all PatientVisitSummary Object, this method
     * returns a PatientVisitSummary that has the latest visit date.
     * @param visitSummaries Collection of PatientVisitSummary
     * @return PatientVisitSummary if there is any, null otherwise.
     */
    public static PatientVisitSummary getLastestVisitedSummary(Collection visitSummaries) {
        Validate.notNull(visitSummaries, "Collection of PatientVisitSummary can not be null");
        PatientVisitSummary latest = null;
        Date latestVisit = null;
        for (Iterator iter = visitSummaries.iterator(); iter.hasNext();) {
            PatientVisitSummary vs = (PatientVisitSummary) iter.next();
            if (DateUtils.isAfter(vs.getLastVisitDate(), latestVisit)) {
                latestVisit = vs.getLastVisitDate();
                latest = vs;
            }
        }
        return latest;
    }


    /**
     * Given the collection of all PatientVisitSummary Object, this method
     * returns a PatientVisitSummary that has the latest visit date in a given facility.
     * @param visitSummaries Collection of PatientVisitSummary
     * @param facilityVisited
     * @return PatientVisitSummary if there is any, null otherwise.
     */
    public static PatientVisitSummary getLastestVisitedSummary(Collection visitSummaries, VAFacility facilityVisited) {
        return getLastestVisitedSummary(getPatientVisitSummaries(visitSummaries, facilityVisited));
    }

    /**
     * Given the collection of all PatientVisitSummary Object, this method
     * returns the collection of PatientVisitSummary for a given income year 
     * @param visitSummaries Collection of PatientVisitSummary
     * @param incomeYear
     * @return unmofiable Set of PatientVisitSummary
     */
    public static Set getPatientVisitSummaries(Collection visitSummaries, Integer incomeYear) {
        Validate.notNull(visitSummaries, "Collection of PatientVisitSummary can not be null");
        Validate.notNull(incomeYear, "Income year can not be null");
        Set s = new HashSet();
        Iterator iter = visitSummaries.iterator();
        while (iter.hasNext()) {
            PatientVisitSummary v = (PatientVisitSummary) iter.next();
            if (incomeYear.equals(v.getIncomeYear()))
                s.add(v);
        }
        return Collections.unmodifiableSet(s);
    }

    /**
     * Given the collection of all PatientVisitSummary Object, this method
     * returns the collection of PatientVisitSummary for a given facility. 
     * @param visitSummaries Collection of PatientVisitSummary
     * @param facilityVisited
     * @return unmodifiable Set of PatientVisitSummary
     */
    public static Set getPatientVisitSummaries(Collection visitSummaries, VAFacility facilityVisited) {
        Validate.notNull(visitSummaries, "Collection of PatientVisitSummary can not be null");
        Validate.notNull(facilityVisited, "VAFacility can not be null");
        Set s = new HashSet();
        Iterator iter = visitSummaries.iterator();
        while (iter.hasNext()) {
            PatientVisitSummary v = (PatientVisitSummary) iter.next();
            if (facilityVisited.equals(v.getFacilityVisited()))
                s.add(v);
        }
        return Collections.unmodifiableSet(s);
    }

}