/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.common.model.ee;

// Java imports
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.lang.Validate;
import org.apache.commons.lang.builder.ToStringBuilder;

// ESR imports
import gov.va.med.esr.common.model.lookup.Indicator;
import gov.va.med.esr.common.model.registry.Registry;
import gov.va.med.fw.model.EntityKey;

/**
 * @author Muddaiah Ranga
 * @version 1.0
 */
public class PrisonerOfWar extends Registry {

	private static final long serialVersionUID = -7790636942564256641L;
	
    private Indicator       powIndicator        = null;
    private String          icn                 = null;
    private String          user                = null;
    private Date            dateEntered         = null;
    
    private Set             internalEpisodes    = null;
    
    /**
     * Creates a new PrisonerOfWar object.
     */
    public PrisonerOfWar() {
        super();
    }

    //----------------------------------------------------------------------------
    // Public method(s)
    //----------------------------------------------------------------------------
    
    public Indicator getPowIndicator() {
        return powIndicator;
    }

    public void setPowIndicator(Indicator powIndicator) {
        this.powIndicator = powIndicator;
    }

    public String getIcn() {
        return icn;
    }

    public void setIcn(String icn) {
        this.icn = icn;
    }

    public String getUser() {
        return user;
    }

    public void setUser(String user) {
        this.user = user;
    }

    public Date getDateEntered() {
        return dateEntered;
    }

    public void setDateEntered(Date dateEntered) {
        this.dateEntered = dateEntered;
    }
    
    public Set getEpisodes() {
        return Collections.unmodifiableSet(getInternalEpisodes());
    }

    public void addEpisode(POWEpisode episode) {
        Validate.notNull(episode, "episode cannot be null");
        getInternalEpisodes().add(episode);
        episode.setPrisonerOfWar(this);
    }

    public void addAllEpisodes(Set episodes) {
        Validate.notEmpty(episodes, "POW episodes cannot be null or empty");
        
        // Call addEpisode to handle bidirectional relationship
        for (Iterator i = episodes.iterator(); i.hasNext();) {
            Object episode = i.next();
            if (episode instanceof POWEpisode) {
                this.addEpisode((POWEpisode)episode);
            }
        }
    }
    
    public void removeEpisode(POWEpisode episode) {
        Validate.notNull(episode, "Episode cannot be null");
        getInternalEpisodes().remove(episode);
        episode.setPrisonerOfWar(null);
    }
    
    public POWEpisode getEpisodeByEntityKey(EntityKey identifier) {
        return (POWEpisode) getEntityByEntityKey(getEpisodes(),identifier);
    }

    public POWEpisode removeEpisodeByEntityKey(EntityKey identifier) {
        return (POWEpisode) removeEntityByEntityKey(getInternalEpisodes(),identifier);
    }

    public void removeAllEpisodes() {
        getInternalEpisodes().clear();
    }
    
    private Set getInternalEpisodes() {
        if (internalEpisodes == null) {
            this.internalEpisodes = new HashSet();
        }
        return this.internalEpisodes;
    }

    private void setInternalEpisodes(Set internalEpisodes) {
        this.internalEpisodes = internalEpisodes;
    }

    /*
     * (non-Javadoc)
     *
     * @see gov.va.med.fw.model.AbstractKeyedEntity#buildToString(org.apache.commons.lang.builder.ToStringBuilder)
     */
    protected void buildToString(ToStringBuilder builder) {
        super.buildToString(builder);

        builder.append("powIndicator", this.powIndicator);
        builder.append("legacyICN", this.icn);
        builder.append("legacyUserName", this.user);
        builder.append("legacyEnteredDate", this.dateEntered);
    }
}

