/*****************************************************************************************
 * Copyright  2004 VHA. All rights reserved
 ****************************************************************************************/
package gov.va.med.esr.common.model.ee;

// Libraries classes
import org.apache.commons.lang.builder.ToStringBuilder;

// Framework classes
import gov.va.med.fw.model.AbstractKeyedEntity;
import gov.va.med.fw.util.StringUtils;

// ESR Classes
import gov.va.med.esr.common.model.lookup.AACIndicator;
import gov.va.med.esr.common.model.lookup.EligibilityStatus;
import gov.va.med.esr.common.model.lookup.EligibilityStatusPendVerfiReason;
import gov.va.med.esr.common.model.lookup.EligibilityVerificationSource;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.infra.ImpreciseDate;

/**
 * @author Commons Team
 * @version 1.0
 */
public class EligibilityVerification extends AbstractKeyedEntity {
   /**
    * An instance of serialVersionUID
    */
   private static final long serialVersionUID = 2815027233801965458L;

   private Person person = null;

   /**
    * Eligibility Verification Method is defined as the method in which the eligibility
    * for this applicant was verified. It is Free text [between 2-50 characters] On HEC:
    * This field contains "free-text" description of "how" or "source" eligibility was
    * checked/verified. If the Eligibility is verified by the HEC this entry will always
    * have CEV appended to the beginning of the entry. (e.g. CEV/HINQ)
    */
   private String verificationMethod = null;

   /**
    * Eligibility Verification Source is defined as the location the eligibility status
    * was determined. Values Include: VistA & site number CEV (the HEC User Name- is
    * displayed on EDB if HEC user sets eligibility status and CEV is sent to sites to be
    * displayed on VISTA). VBA if data is from AAC.
    */
   private EligibilityVerificationSource verificationSource = null;

   private VAFacility verificationFacility = null;

   /**
    * Eligibility Status Date is defined as the effective date of the eligibility status.
    * Can be imprecise.
    */
   private ImpreciseDate eligibilityStatusDate = null;

   private EligibilityStatus eligibilityStatus = null;

   private EligibilityStatusPendVerfiReason eligStatusPendingVerficationReason  = null;
   
   private String eligStatusPendingVerfReasonExplain;

   private AACIndicator aacIndicator = null;

   /**
    * Creates a new EligibilityVerification object.
    */
   public EligibilityVerification() {
      super();
   }

   // ----------------------------------------------------------------------------
   // Public method(s)
   // ----------------------------------------------------------------------------
   public Person getPerson() {
      return this.person;
   }

   public void setPerson(Person person) {
      this.person = person;
   }

   public String getVerificationMethod() {
      return this.verificationMethod;
   }

   public void setVerificationMethod(String verificationMethod) {
      this.verificationMethod = verificationMethod;
   }

   public EligibilityVerificationSource getVerificationSource() {
      return this.verificationSource;
   }

   public void setVerificationSource(EligibilityVerificationSource verificationSource) {
      this.verificationSource = verificationSource;
   }

   public VAFacility getVerificationFacility() {
      return this.verificationFacility;
   }

   public void setVerificationFacility(VAFacility verificationFacility) {
      this.verificationFacility = verificationFacility;
   }

   /**
    * Function added for defect 845, 2177, 2176 with two reasons 
    * 
    * 1. To display Source and Site together for view and history page 
    * 2. Manipulate description for 'VISTA', change to 'VAMC'
    * 
    * @return description string contains Source and Site.
    */
   public String getVerificationSourceFacility() {

      // Defect 2233 and 3550 state that only display VAMC + station number
      // the old logics append a station number to all facility 
      
      String name = null;
      EligibilityVerificationSource evs = getVerificationSource();
      String evsCode = evs != null ? evs.getCode() : null;
      if( evsCode != null ) {
         name = EligibilityVerificationSource.CODE_VAMC.getName().equals( evsCode ) ? "VAMC" : evs.getDescription();

         if(  "VAMC".equals( name ) && getVerificationFacility() != null && StringUtils.isNotEmpty(getVerificationFacility().getStationNumber())) {
            name += " - " + getVerificationFacility().getStationNumber();
         }
      }
      return name;
   }

   public ImpreciseDate getEligibilityStatusDate() {
      return this.eligibilityStatusDate;
   }

   public void setEligibilityStatusDate(ImpreciseDate eligibilityStatusDate) {
      this.eligibilityStatusDate = eligibilityStatusDate;
   }

   public EligibilityStatus getEligibilityStatus() {
      return this.eligibilityStatus;
   }

   public void setEligibilityStatus(EligibilityStatus eligibilityStatus) {
      this.eligibilityStatus = eligibilityStatus;
   }

   public AACIndicator getAacIndicator() {
      return aacIndicator;
   }

   public void setAacIndicator(AACIndicator aacIndicator) {
      this.aacIndicator = aacIndicator;
   }

   /*
    * (non-Javadoc)
    * 
    * @see gov.va.med.fw.model.AbstractKeyedEntity#buildToString(org.apache.commons.lang.builder.ToStringBuilder)
    */
   public void buildToString(ToStringBuilder builder) {
      super.buildToString(builder);
      builder.append("verificationMethod", this.verificationMethod);
      builder.append("verificationSource", this.verificationSource);
      builder.append("verificationFacility", this.verificationFacility);
      builder.append("eligibilityStatus", this.eligibilityStatus);
      builder.append("eligStatusPendingVerficationReason", this.eligStatusPendingVerficationReason);
      builder.append("eligStatusPendingVerfReasonExplain", this.eligStatusPendingVerfReasonExplain);
      builder.append("aacIndicator", this.aacIndicator);
      builder.append("eligibilityStatusDate", this.eligibilityStatusDate);
   }

public EligibilityStatusPendVerfiReason getEligStatusPendingVerficationReason() {
	return eligStatusPendingVerficationReason;
}

public void setEligStatusPendingVerficationReason(EligibilityStatusPendVerfiReason eligStatusPendingVerficationReason) {
	this.eligStatusPendingVerficationReason = eligStatusPendingVerficationReason;
}

public String getEligStatusPendingVerfReasonExplain() {
	return eligStatusPendingVerfReasonExplain;
}

public void setEligStatusPendingVerfReasonExplain(String eligStatusPendingVerfReasonExplain) {
	this.eligStatusPendingVerfReasonExplain = eligStatusPendingVerfReasonExplain;
}
}