package gov.va.med.esr.common.batchprocess;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import gov.va.med.esr.common.model.lookup.AddressType;
import gov.va.med.esr.common.model.lookup.Country;
import gov.va.med.esr.service.CommsEmailBulletinService;
import gov.va.med.esr.service.trigger.BulletinTrigger;
import gov.va.med.fw.batchprocess.DataProcessCompletedHandler;
import gov.va.med.fw.batchprocess.DataProcessExecutionContext;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.service.AbstractComponent;

/**
 * This class persists the statistics at the end of the Send NCOA Address File
 * Verification process.
 * 
 * @author Rajiv Patnaik Created on Feb 27, 2006
 * @version 1.0
 * 
 * Copyright  2006 VHA. All rights reserved
 */
public class SendNCOAAddressFileProcessCompletedHandler extends AbstractComponent
        implements DataProcessCompletedHandler
{
    private DAOOperations dao;
    
    private CommsEmailBulletinService bulletinService;

    private static final String PARAM_NAME_ADDRESS_CODE = "addressCode";

    private static final String PARAM_NAME_COUNTRY_CODE = "countryCode";

    private String queryNameTotalScreenedCount;

    private String queryNameTotalDeceasedCount;

    /**
     * @return Returns the queryNameTotalDeceasedCount.
     */
    public String getQueryNameTotalDeceasedCount()
    {
        return queryNameTotalDeceasedCount;
    }

    /**
     * @param queryNameTotalDeceasedCount
     *            The queryNameTotalDeceasedCount to set.
     */
    public void setQueryNameTotalDeceasedCount(
            String queryNameTotalDeceasedCount)
    {
        this.queryNameTotalDeceasedCount = queryNameTotalDeceasedCount;
    }

    /**
     * @return Returns the queryNameTotalScreenedCount.
     */
    public String getQueryNameTotalScreenedCount()
    {
        return queryNameTotalScreenedCount;
    }

    /**
     * @param queryNameTotalScreenedCount
     *            The queryNameTotalScreenedCount to set.
     */
    public void setQueryNameTotalScreenedCount(
            String queryNameTotalScreenedCount)
    {
        this.queryNameTotalScreenedCount = queryNameTotalScreenedCount;
    }

    /*
     * (non-Javadoc)
     * 
     * @see gov.va.med.fw.batchprocess.DataProcessCompletedHandler#dataProcessingComplete(gov.va.med.fw.batchprocess.DataProcessExecutionContext)
     */
    public void dataProcessingComplete(DataProcessExecutionContext context)
    {
        //Persist the statistics data
        NCOAAddressVerificationFileProcessStatistics statistics = (NCOAAddressVerificationFileProcessStatistics)context.getProcessStatistics();

        NCOAAddressVerificationFileSummary summary = new NCOAAddressVerificationFileSummary();

        try
        {
            //Make additional queries to fetch other counts.
            Integer numberOfRecordsScreened = (Integer) (getDao()
                    .find(queryNameTotalScreenedCount)
                    .iterator().next());
            summary.setCountOfRecordsScreened(numberOfRecordsScreened
                    .intValue());

            String[] paramNames = { PARAM_NAME_COUNTRY_CODE };
            Object[] params = { Country.CODE_USA.getCode() };

            Integer numberOfRecordsDeceased = (Integer) (getDao()
                    .findByNamedQueryAndNamedParam(queryNameTotalDeceasedCount,
                            paramNames, params).iterator().next());
            summary.setCountOfRecordsDeceased(numberOfRecordsDeceased
                    .intValue());

            summary.setCountOfRecordsAfterScreening(statistics
                    .getNumberOfSuccessfulRecords());
            summary.setCountOfRecordsExcluded(numberOfRecordsScreened
                    .intValue()
                    - statistics.getNumberOfSuccessfulRecords());

            //Persist summary data
            getDao().saveObject(summary); 
            
            statistics.setNcoaAddressVerificationFileSummary(summary);
            
            // send bulletin
            if(bulletinService != null) {
	            Map bulletinData = new HashMap();
	            bulletinData.put("EndTime", new Date());
	            bulletinData.put("TotalCount", new Integer(summary.getCountOfRecordsScreened()));
	            bulletinData.put("ExcludedCount", new Integer(summary.getCountOfRecordsExcluded()));
	            bulletinData.put("DeceasedCount", new Integer(summary.getCountOfRecordsDeceased()));
	            bulletinData.put("SentToAACCount", new Integer(summary.getCountOfRecordsAfterScreening()));
	            bulletinData.put("SentToAACFileCount", new Integer(1));
	            bulletinService.sendEmailBulletin(BulletinTrigger.DataType.NCOA_PROCESS_ENDED,
	                    bulletinData, null);
            }            
           
        } catch (Exception e)
        {
            throw new RuntimeException(
                    "Exception while handling the completion of NCOA processing", e);
        }        
    }

    /**
     * @return Returns the dao.
     */
    public DAOOperations getDao()
    {
        return dao;
    }

    /**
     * @param dao
     *            The dao to set.
     */
    public void setDao(DAOOperations dao)
    {
        this.dao = dao;
    }

	/**
	 * @return Returns the bulletinService.
	 */
	public CommsEmailBulletinService getBulletinService() {
		return bulletinService;
	}

	/**
	 * @param bulletinService The bulletinService to set.
	 */
	public void setBulletinService(CommsEmailBulletinService bulletinService) {
		this.bulletinService = bulletinService;
	}
}