/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.esr.common.batchprocess;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.service.FinancialsService;
import gov.va.med.esr.service.PersonIdentityTraits;
import gov.va.med.esr.service.PersonService;
import gov.va.med.fw.batchprocess.AbstractDataFileSingleRowIncrementProcess;
import gov.va.med.fw.batchprocess.DataFileProcessExecutionContext;

/**
 * ESR Batch Process for the Receive Signature Information Process use case.
 * 
 * Created Jan 25, 2006 4:15:29 PM
 * 
 * @author DNS   BOHMEG
 */
public class ReceiveSignatureInformationFileProcess extends
	AbstractDataFileSingleRowIncrementProcess {
	private static String DELIMITER = "^";
	
	private PersonService personService;

	private FinancialsService financialsService;

	protected boolean processDataRecord(
            DataFileProcessExecutionContext context, Object bean)
    {
		SignatureFileData dataRecord = (SignatureFileData) bean;
		
		// requirements dictate to only include those "No Match" Persons in the exception file				
		boolean success = true;
		
		try {
			Person person = matchPerson(dataRecord);
			if (person != null) {
				person.addSignatureImage(dataRecord.getSignatureImage());
				financialsService.processPersonSignature(person);
			} else
				success = false;
		} catch (Exception e) {
			//throwIllegalStateException("Unable to persist Person Signature", e);
			if(logger.isErrorEnabled())
				logger.error("There was an exception trying to apply/persist the Signature information.  It is logged here for triage but" +
						", per requirements, this will not go into the exception file.", e);
		}
		return success;
	}

	/* overridden to strip out (last field) user name (not in ICD to put in exception .FSSN file) */
	protected Object createExceptionObject(String rawData) {
		return StringUtils.substringBeforeLast(rawData, DELIMITER);
	}
	

	private Person matchPerson(SignatureFileData data) {
		PersonIdentityTraits traits = new PersonIdentityTraits();
		traits.setSsn(data.getSSNWrapper());
		Person person = null;
		try {
			person = personService.find(traits);
		} catch (Exception e) {
			if(logger.isErrorEnabled())
				logger.error("Unable to find (exact match) Person for SignatureImage ssn: "
							//+ data.getSsn()
							, e);
		}
			
		if(person != null && logger.isInfoEnabled()) {
				logger.info("VPID [" + person.getVPIDEntityKey().getVPID() + "] found"); // for SSN [" + traits.getSsn().getFormattedSsnText() + "]");
		}
		
		return person;
	}

	/**
	 * @return Returns the financialsService.
	 */
	public FinancialsService getFinancialsService() {
		return financialsService;
	}

	/**
	 * @param financialsService
	 *            The financialsService to set.
	 */
	public void setFinancialsService(FinancialsService financialsService) {
		this.financialsService = financialsService;
	}

	/**
	 * @return Returns the personService.
	 */
	public PersonService getPersonService() {
		return personService;
	}

	/**
	 * @param personService
	 *            The personService to set.
	 */
	public void setPersonService(PersonService personService) {
		this.personService = personService;
	}

	public void afterPropertiesSet() {
		super.afterPropertiesSet();
		Validate.notNull(personService, "A PersonService is needed");
		Validate.notNull(financialsService, "A FinancialsService is needed");
	}
}
