/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.esr.common.batchprocess;

import java.math.BigDecimal;
import java.util.List;
import java.util.ArrayList;

import org.apache.commons.lang.Validate;

import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.id.PersonEntityKey;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKeyImpl;
import gov.va.med.esr.common.model.ee.EligibilityVerification;
import gov.va.med.esr.common.model.lookup.AACIndicator;
import gov.va.med.esr.service.GenerateQRYZ11MessagingService;
import gov.va.med.esr.service.LookupService;
import gov.va.med.esr.service.PersonService;
import gov.va.med.esr.service.CommsEmailBulletinService;
import gov.va.med.esr.service.impl.GenerateQRYZ11MessagingServiceImpl;
import gov.va.med.fw.batchprocess.AbstractDataQueryIncrementalProcess;
import gov.va.med.fw.batchprocess.BatchProcessService;
import gov.va.med.fw.batchprocess.DataProcessCompletedHandler;
import gov.va.med.fw.batchprocess.DataQueryProcessExecutionContext;
import gov.va.med.fw.batchprocess.ProcessStatisticsHandler;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.batchprocess.model.JobConfig;
import org.apache.commons.lang.StringUtils;
/**
 * Generates QRY-Z11 messages to send to VBA for people with a VBA Query Status
 * of VBA Query Initiated or VBA Queried - Pending Response.
 * 
 * Created Aug 18, 2008 2:39:36 PM
 * @author DNS   GALLAS
 */
public class GenerateQueryVBAProcess extends AbstractDataQueryIncrementalProcess {
	
    private GenerateQRYZ11MessagingService messagingService = null;
    private PersonService personService;
    private LookupService lookupService;  
	private ProcessStatisticsHandler processStatisticsHandler;

    protected void processData(DataQueryProcessExecutionContext context, List acquiredData) {
        if(logger.isDebugEnabled())
            logger.debug(/*Generate Query VBA:*/ getProcessName() + ": Query increment result Size="+(acquiredData == null ? 0 : acquiredData.size()));
        
        if (acquiredData == null)
            return;
        
        // Add check for isInterrupted so don't have to wait for entire batch to finish
        for (int i = 0; i < acquiredData.size() && !isInterrupted(context); i++) {
            // Object[] returned from get when scrolling (idempotent is false)
            Object[] row = (Object[]) acquiredData.get(i);
            BigDecimal personId = (BigDecimal) row[0];
            
            PersonEntityKey key = new PersonIdEntityKeyImpl(personId);
            
            if(logger.isDebugEnabled())
                logger.debug("Processing person # " + (i+1) + " with id = " + personId);
            
            try {
               processRecord(key, row);
               context.getProcessStatistics().incrementNumberOfSuccessfulRecords();
            } catch (Exception ex) {
                context.getProcessStatistics().incrementNumberOfErrorRecords();
                String errMsg = "Error while executing " + getProcessName() + " for person "
                                + personId;
                context.getExceptionData().add(errMsg+ " Reason: "+ ex.getMessage());
                if(logger.isWarnEnabled())
                    logger.warn(errMsg + " Reason: ", ex);
            }
            
            // Update statistics more frequently than once per batch (default size 500)
            if(context.getProcessStatistics().isTotalNumberMod(DEFAULT_JOB_RESULT_UPDATE_INTERVAL))
                this.updateJobResult(context);
        }
        
        if(logger.isDebugEnabled()) {
            logger.debug(/*Generate Query VBA:*/ getProcessName() + ": Processing of current batch complete.");
            logger.debug(/*Generate Query VBA1:*/ getProcessName() + ": Successful records count = "+context.getProcessStatistics().getNumberOfSuccessfulRecords());
            logger.debug(/*Generate Query VBA:*/ getProcessName() + ": Failure records count ="+context.getProcessStatistics().getNumberOfErrorRecords());
        }
        
        // send email notification - 
        // framework processStatistics() has sent email at the end of batch job if email address is configured
        // send email to PII                   if no email config
        this.processStatisticsHandler = this.getProcessStatisticsHandler();
        try{
        	if(this.processStatisticsHandler!= null) {
        		JobConfig config = this.getBatchProcessService().getJobConfig(getJobName(context),getJobGroupName(context));
        		if(config == null || StringUtils.isBlank(config.getEmailDistributionList()) ) {
        			List defaultEmailList = new ArrayList();
        			defaultEmailList.add("PII                  ");
        			processStatisticsHandler.processStatisticsWithDynamicRecipients(context.getProcessStatistics(), defaultEmailList);
        			logger.info("GenerateQueryVBAProcess batch job - email of total query count was sent to PII                  \n");
        		}
        	}
        }catch (Exception ex) {
        	logger.error("GenerateQueryVBAProcess - ERROR in sending email of VBA query count ", ex);
        }
    }	

    /**
     * Processes data for a particular person
     * @param key
     * @param VPID
     * @throws ServiceException 
     */
    protected void processRecord(PersonEntityKey key, Object[] dataRow) throws Exception 
    {
        String VPID = (String) dataRow[1];
       	messagingService.generateMessage(key, VPID);
       	
       	// CR9703 
       	// after QRYZ11 message was sent, set VBA Query Status = "VBA Queried, Pending Response"
       	// retrieve Person
		Person person = personService.getPerson(key);
		if (person == null)
			throw new RuntimeException("Unable to find Person for VPID: "+ VPID);
		
		// set person AAC indicator for VBA query status 
		 if (person.getEligibilityVerification() == null) {
	            EligibilityVerification eligibilityVerification = new EligibilityVerification();
	            person.setEligibilityVerification(eligibilityVerification);
	    }
		person.getEligibilityVerification().setAacIndicator(this.getLookupService().getAACIndicatorByCode(AACIndicator.CODE_AAC_QUERIED_PENDING.getCode()));
		
		personService.save(person);
    }
        
    /**
     * @param messagingService
     *            The messagingService to set.
     */
    public void setMessagingService(GenerateQRYZ11MessagingService messagingService) {
        this.messagingService = messagingService;
    }

    protected GenerateQRYZ11MessagingService getMessagingService() {
    	return messagingService;
    }

    public PersonService getPersonService() {
		return personService;
	}

    /**
     * @return Returns the lookupService.
     */
    public LookupService getLookupService() {
        return lookupService;
    }
    
    /**
     * @param lookupService The lookupService to set.
     */
    public void setLookupService(LookupService lookupService) {
        this.lookupService = lookupService;
    }    
	/**
	 * @param personService
	 *            The personService to set.
	 */
	public void setPersonService(PersonService personService) {
		this.personService = personService;
	}
	/**
	 * @return Returns the bulletinService.
	 */

    public void afterPropertiesSet() {
        super.afterPropertiesSet();
        Validate.notNull(personService);
        Validate.notNull(lookupService);
        Validate.notNull(messagingService, "A MessagingService is needed");
        Validate.isTrue(!(messagingService instanceof GenerateQRYZ11MessagingServiceImpl), 
                "A GenerateQRYZ11 Messaging Service is required");
    }
}
