package gov.va.ccd.service;

import gov.va.ccd.components.impl.AdvanceDirectiveComponent;
import gov.va.ccd.components.impl.AllergyComponent;
import gov.va.ccd.components.impl.ConditionProblemsComponent;
import gov.va.ccd.components.impl.EncounterComponent;
import gov.va.ccd.components.impl.ImmunizationComponent;
import gov.va.ccd.components.impl.MedicationsComponent;
import gov.va.ccd.components.impl.PayersComponent;
import gov.va.ccd.components.impl.PlanOfCare;
import gov.va.ccd.components.impl.ProcedureComponent;
import gov.va.ccd.components.impl.ResultsComponent;
import gov.va.ccd.components.impl.VitalSignsComponent;
import gov.va.ccd.consol.processor.ConsolProcessor;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.mail.internet.MimeUtility;

import org.openhealthtools.mdht.uml.cda.CDAPackage;
import org.openhealthtools.mdht.uml.cda.ClinicalDocument;
import org.openhealthtools.mdht.uml.cda.ccd.AdvanceDirectivesSection;
import org.openhealthtools.mdht.uml.cda.ccd.AlertsSection;
import org.openhealthtools.mdht.uml.cda.ccd.CCDPackage;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.EncountersSection;
import org.openhealthtools.mdht.uml.cda.ccd.ImmunizationsSection;
import org.openhealthtools.mdht.uml.cda.ccd.ProblemSection;
import org.openhealthtools.mdht.uml.cda.consol.ConsolPackage;
import org.openhealthtools.mdht.uml.cda.util.CDAUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardizeXML
{
	private static final Logger logger = LoggerFactory.getLogger(StandardizeXML.class);

	ClinicalDocument ccdDocument = null;
	// If an input is encoded coming in return it encoded.
	boolean isEncoded = false;

	public static void init()
	{
		CDAPackage.eINSTANCE.eClass();
		ConsolPackage.eINSTANCE.eClass();
		CCDPackage.eINSTANCE.eClass();
	}

	public StandardizeXML()
	{
		init();
	}

	public StandardizeXML(final byte[] docBytes) throws Exception
	{
		this();
		setDocBytes(docBytes);
	}

	public StandardizeXML(String fileName)
	{
		this();
		try
		{
			if(fileName != null && fileName.length() > 0)
			{
				ccdDocument = CDAUtil.load(new FileInputStream(fileName));
			}
			else
			{
				logger.error("Please provide valid file name!");
			}
		}
		catch(FileNotFoundException e)
		{
			logger.error("File Not found!");
		}
		catch(Exception e)
		{
			e.printStackTrace();
			logger.error("Error occurred", e);
		}

	}

	public void setDocBytes(final byte[] docBytes) throws Exception
	{
		if(docBytes != null)
		{
			InputStream is = null;
			try
			{
				is = new ByteArrayInputStream(docBytes);
				isEncoded = false;
				ccdDocument = CDAUtil.load(is);
				logger.debug("Processing as plain text.");
			}
			catch(Exception ex)
			{
				logger.error("Failed to load bytes.");
				logger.error("Error occurred", ex);
				ccdDocument = null;
			}

			if(ccdDocument == null)
			{
				try
				{
					is = new ByteArrayInputStream(decode(docBytes, "base64"));
					ccdDocument = CDAUtil.load(is);
					// isEncoded = true;
					logger.debug("Processing as encoded.");
				}
				catch(Exception ex)
				{
					logger.debug("Failed to decode incoming bytes.");
					throw ex;
				}
			}

			if(ccdDocument == null)
			{
				logger.debug("Not a Continuity of Care Document can not process.");
				throw new Exception("Not a Continuity of Care Document can not process.");
			}
		}
	}

	public byte[] processDocument() throws Exception
	{
		processSections();

		StringWriter sw = new StringWriter();
		CDAUtil.save(this.ccdDocument, sw);
		logger.debug("StyleSheet Printing Doc.");
		logger.debug(sw.getBuffer().toString());
		if(isEncoded)
		{
			logger.debug("Returning encoded.");
			return encode(sw.getBuffer().toString().getBytes(), "base64");
		}
		else
		{
			logger.debug("Returning string bytes.");
			sw.getBuffer().toString().getBytes("UTF-8");
			return sw.getBuffer().toString().getBytes();
		}
	}

	/**
	 * 
	 * @param b
	 * @param encodingType "base64", "quoted-printable", "7bit", "8bit" or
	 *            "binary"
	 * @return
	 * @throws Exception
	 */
	public static byte[] encode(final byte[] b, final String encodingType) throws Exception
	{
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		OutputStream b64os = MimeUtility.encode(baos, encodingType);
		b64os.write(b);
		b64os.close();
		return baos.toByteArray();
	}

	/**
	 * 
	 * @param b
	 * @param encodingType "base64", "quoted-printable", "7bit", "8bit" or
	 *            "binary"
	 * @return
	 * @throws Exception
	 */
	public static byte[] decode(final byte[] b, final String encodingType) throws Exception
	{
		ByteArrayInputStream bais = new ByteArrayInputStream(b);
		InputStream b64is = MimeUtility.decode(bais, encodingType);
		byte[] tmp = new byte[b.length];
		int n = b64is.read(tmp);
		byte[] res = new byte[n];
		System.arraycopy(tmp, 0, res, 0, n);
		return res;
	}

	public static void main(String arg[]) throws Exception
	{
		if(arg != null && arg.length > 0)
		{
			// Only looking for a single file.
			logger.debug("Filename entered: " + arg[0]);
			File f = new File(arg[0]);

			String filename = null;

			int i = f.getName().lastIndexOf('.');
			if(i > 0)
			{
				filename = f.getName().substring(0, i);
			}
			Double sTime = (double) System.currentTimeMillis();

			StandardizeXML obj = new StandardizeXML(arg[0]);
			if(obj != null && obj.ccdDocument != null)
			{
				obj.processSections();

				writeToFile(obj, filename);
			}
			Long stopTime = System.currentTimeMillis();
			sTime = stopTime - sTime;
			sTime = sTime / 1000.00;
			logger.debug("It took " + sTime + " seconds to process " + filename);
		}
		else
		{
			// Lets look at everything in resources.
			String dirName = "C:/SRA/Inc2R1/NHIN_Adapter/SpecProcessor/src/test/resources/input/CCDA";//
			// "./";
			File dir = new File(dirName);
			for(String fname : dir.list())
			{
				if(fname.endsWith(".xml"))
				{
					Long sTime = System.currentTimeMillis();
					logger.debug("Attempting to start processing: " + fname);
					StandardizeXML obj = new StandardizeXML(dirName + "/" + fname);
					if(obj.ccdDocument != null)
					{
						obj.processSections();
						writeToFile(obj, fname);
						obj = null;
					}
					else
					{
						logger.error("Failed to process file: " + fname);
					}
					Long stopTime = System.currentTimeMillis();
					sTime = stopTime - sTime;
					sTime = sTime / 1000;
					logger.debug("It took " + sTime + " seconds to process " + fname);
				}
			}

		}

	}

	private static void writeToFile(final StandardizeXML obj, final String fileName) throws Exception
	{
		SimpleDateFormat dateformatMMDDYYYY = new SimpleDateFormat("MMddyyyyHHmmss");
		String s = dateformatMMDDYYYY.format(new Date());
		String fileSeparator = System.getProperties().getProperty("file.separator");
		File outfile = new File("c:" + fileSeparator + "test" + fileSeparator + "newstylesheets" + fileSeparator + "nhin" + fileSeparator + fileName + "_output_" + s + ".xml");
		logger.debug("outputfile: " + outfile);
		StringWriter sw = new StringWriter();
		// FileWriter fw = new FileWriter(outfile.getAbsoluteFile());
		// fix for fortify issue - Unreleased Resource: Streams RTC ticket #
		// 163008
		FileOutputStream outs = null;
		try
		{
			outs = new FileOutputStream(outfile);
		}
		catch(IOException e)
		{

		}
		finally
		{
			if(outs != null)
			{
				outs.close();
			}
		}

		OutputStreamWriter ows = new OutputStreamWriter(outs, "UTF8");
		CDAUtil.save(obj.ccdDocument, sw);
		int pos = sw.getBuffer().indexOf("\n");
		sw.getBuffer().insert(pos, "\n<?xml-stylesheet type=\"text/xsl\" href=\"cda2detail.xsl\"?>\n");
		ows.write(sw.getBuffer().toString());
		ows.flush();
		ows.close();
	}

	private void processSections()
	{
		// Need to split out to a factory. This will get very difficult to
		// maintain.
		if(ccdDocument == null)
		{
			// Need to see the best place to throw an error.
			return;
		}

		if(ccdDocument instanceof ContinuityOfCareDocument)
		{
			logger.debug("Processing document as HITSP Doc.");
			processHitspSections();
		}
		else if(ConsolProcessor.isConsolDoc(ccdDocument))
		{
			// !!!! This code has been removed per Business request during evaluation of Increment 7 R3 NVW.
			// Need to split out to avoid namespace issues.
			//logger.debug("Processing document as Consolidated.");
			//ConsolProcessor cp = new ConsolProcessor(ccdDocument);
			//cp.processDocument();
		}
		else
		{
			logger.error("Failed to determine type of document.");
		}
	}

	private void processHitspSections()
	{
		// Document should never be null at this point!
		assert (ccdDocument != null);

		ContinuityOfCareDocument doc = (ContinuityOfCareDocument) ccdDocument;
		if(doc == null)
		{
			return;
		}

		try
		{
			AlertsSection alertSection = doc.getAlertsSection();
			if(alertSection != null)
			{
				AllergyComponent allergy = new AllergyComponent(doc);
				allergy.execute();
			}
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Alerts", exception);
		}

		try
		{
			AdvanceDirectivesSection ad = doc.getAdvanceDirectivesSection();
			if(ad != null)
			{
				AdvanceDirectiveComponent adc = new AdvanceDirectiveComponent(doc);
				adc.execute();
			}
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Advanced Directive", exception);
		}

		try
		{
			EncountersSection encounterSection = doc.getEncountersSection();
			if(encounterSection != null)
			{
				EncounterComponent encounter = new EncounterComponent(doc);
				encounter.execute();
			}
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Encounters", exception);
		}

		try
		{
			ImmunizationsSection immunSection = doc.getImmunizationsSection();
			if(immunSection != null)
			{
				ImmunizationComponent immun = new ImmunizationComponent(doc);
				immun.execute();
			}
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Immunizations", exception);
		}

		try
		{
			ProblemSection problemSection = doc.getProblemSection();
			if(problemSection != null)
			{
				ConditionProblemsComponent problem = new ConditionProblemsComponent(doc);
				problem.execute();
			}

		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Problems", exception);
		}

		/** Vital Signs **/
		try
		{
			final VitalSignsComponent vitalSigns = new VitalSignsComponent(doc);
			vitalSigns.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Vitals", exception);
		}

		/** Medications **/
		try
		{
			final MedicationsComponent medicationsComponent = new MedicationsComponent(doc);
			medicationsComponent.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Medications", exception);
		}

		/** Procedure **/
		try
		{
			final ProcedureComponent procedureComponent = new ProcedureComponent(doc);
			procedureComponent.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Procedures", exception);
		}

		/** PayersSection **/
		try
		{
			final PayersComponent payersComponent = new PayersComponent(doc);
			payersComponent.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Payers", exception);
		}

		/** Plan of Care **/
		try
		{
			final PlanOfCare plansComponent = new PlanOfCare(doc);
			plansComponent.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Plan of Care", exception);
		}

		/** Results **/
		try
		{
			final ResultsComponent results = new ResultsComponent(doc);
			results.execute();
		}
		catch(Throwable exception)
		{
			logger.error("Failed to process Results", exception);
		}
	}
}
