package gov.va.ccd.components.impl;

import gov.va.ccd.service.util.Utils;
import gov.va.ccd.service.value.object.UnitValue;
import gov.va.ccd.service.value.objects.impl.VitalSignsVO;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.Component4;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.VitalSignsSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;

import com.va.ccd.service.enums.VitialSignCodes;

/**
 * Vital Signs Component
 * 
 * @author Brad Messerle
 * @since 1.0
 * 
 * @param <S> VitalSignsSection
 */
public class VitalSignsComponent extends AbstractComponent<VitalSignsSection>
{
	/**
	 * Constructor
	 * 
	 * @param vitalSignsSection
	 * @param ccdDocument
	 */
	public VitalSignsComponent(final ContinuityOfCareDocument ccdDocument)
	{
		super(ccdDocument.getVitalSignsSection(), ccdDocument);
	}

	/**
	 * Process Section
	 */
	@Override
	public void execute()
	{
		if(section != null)
		{
			final List<VitalSignsVO> rows = this.createVitalSignHelperObjects();
			appendNarrative(this.generateXml(rows));
		}
	}

	private String generateXml(List<VitalSignsVO> rows)
	{
		StringBuffer sb = new StringBuffer();
		if(CollectionUtils.isNotEmpty(rows))
		{
			sb.append("<table ID='_nbVitalSignsComments' border='1' width='100%'>");
			sb.append("<thead>");
			sb.append("<tr><th>Comments</th></tr>");
			sb.append("</thead>");
			sb.append("<tbody>");

			for(String comment : this.getComments())
			{
				sb.append("<tr width='100%'>");
				sb.append("<td width='100%'>");
				sb.append("<p>");
				sb.append(comment);
				sb.append("</p>");
				sb.append("</td></tr>");
			}

			sb.append("</tbody>");
			sb.append("</table>");

			sb.append("<table ID='_nbVitalSigns' border='1' width='100%'>");

			sb.append("<thead>");
			sb.append("<tr>");
			sb.append("<th>Date</th>");
			sb.append("<th>TEMP</th>");
			sb.append("<th>Heart Rate</th>");
			sb.append("<th>Resp Rate</th>");
			sb.append("<th>BP</th>");
			sb.append("<th>Height</th>");
			sb.append("<th>Height-Lying</th>");
			sb.append("<th>Body Mass Index</th>");
			sb.append("<th>Weight</th>");
			sb.append("<th>O2 Sat</th>");
			sb.append("<th>Head Circumf</th>");
			sb.append("<th>Body Surface Area</th>");
			sb.append("<th>Source</th>");
			sb.append("</tr>\n");
			sb.append("</thead>");
			sb.append("<tbody>");

			for(VitalSignsVO helper : rows)
			{
				sb.append("<tr>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getDate()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getTemp()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getHeartRate()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getRespRate()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getBp()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getHeight()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getHeightLying()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getBodyMassIndex()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getWeight()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getO2Sat()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getHeadCircumf()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getBodySurfaceArea()));
				sb.append("</td>");
				sb.append("<td>");
				sb.append(Utils.valueOrDefault(helper.getSource()));
				sb.append("</td>");

				sb.append("</tr>");
			}

			sb.append("</tbody>");
			sb.append("</table>");
		}
		return sb.toString();
	}

	private List<VitalSignsVO> createVitalSignHelperObjects()
	{
		List<VitalSignsVO> rows = new ArrayList<VitalSignsVO>();
		if(section != null)
		{
			for(Entry entry : section.getEntries())
			{
				if(entry.getOrganizer() != null)
				{
					final VitalSignsVO helper = new VitalSignsVO();
					this.getDateColumn(entry, helper);
					helper.getTemp().addAll(getValueAndUnit(entry, VitialSignCodes.BODY_TEMPERATURE.getCode()));
					helper.getHeartRate().addAll(getValueAndUnit(entry, VitialSignCodes.HEART_RATE.getCode()));
					helper.getRespRate().addAll(getValueAndUnit(entry, VitialSignCodes.RESPIRATION_RATE.getCode()));
					this.getBPColumn(entry, helper);
					helper.getHeight().addAll(getValueAndUnit(entry, VitialSignCodes.HEIGHT.getCode()));
					helper.getHeightLying().addAll(getValueAndUnit(entry, VitialSignCodes.HEIGHT_LYING.getCode()));
					helper.getWeight().addAll(getValueAndUnit(entry, VitialSignCodes.WEIGHT_MEASURED.getCode()));
					helper.getBodyMassIndex().addAll(getValueAndUnit(entry, VitialSignCodes.BODY_MASS_INDEX.getCode()));
					helper.getBodySurfaceArea().addAll(getValueAndUnit(entry, VitialSignCodes.BODY_SURFACE_AREA.getCode()));
					helper.getO2Sat().addAll(getValueAndUnit(entry, VitialSignCodes.OXYGEN_SATURATION.getCode()));
					helper.getHeadCircumf().addAll(getValueAndUnit(entry, VitialSignCodes.HEAD_CIRCUMFERENCE.getCode()));
					this.getSourceColumn(entry, helper);

					rows.add(helper);
				}
			}
		}

		return rows;

	}

	private void getDateColumn(final Entry entry, final VitalSignsVO helper)
	{
		if(entry != null && entry.getOrganizer() != null && entry.getOrganizer().getEffectiveTime() != null)
		{

			if(StringUtils.isNotBlank(entry.getOrganizer().getEffectiveTime().getValue()))
			{
				helper.setDate(entry.getOrganizer().getEffectiveTime().getValue());
			}

			if(entry.getOrganizer().getEffectiveTime().getLow() != null)
			{
				helper.setDate(entry.getOrganizer().getEffectiveTime().getLow().getValue());
			}
		}

		if(StringUtils.isBlank(helper.getDate()))
		{
			if(entry != null && entry.getOrganizer() != null)
			{
				helper.setDate(getDateFromObservation(entry.getOrganizer().getObservations()));
			}
		}
	}

	private String getDateFromObservation(final EList<Observation> obsList)
	{
		for(Observation obs : obsList)
		{
			if(obs.getEffectiveTime() != null)
			{
				if(StringUtils.isNotBlank(obs.getEffectiveTime().getValue()))
				{
					return obs.getEffectiveTime().getValue();
				}
				else if(obs.getEffectiveTime().getLow() != null && StringUtils.isNotBlank(obs.getEffectiveTime().getLow().getValue()))
				{
					return obs.getEffectiveTime().getLow().getValue();
				}
			}
		}
		return null;
	}

	private void getBPColumn(final Entry entry, final VitalSignsVO helper)
	{
		List<UnitValue> systolic = getValueAndUnit(entry, VitialSignCodes.BP_SYSTOLIC.getCode());
		List<UnitValue> diastolic = getValueAndUnit(entry, VitialSignCodes.BP_DIASTOLIC.getCode());
		// check Observation directly under entry
		systolic.addAll(getValueUnitFromObs(entry.getObservation(), VitialSignCodes.BP_SYSTOLIC.getCode()));
		diastolic.addAll(getValueUnitFromObs(entry.getObservation(), VitialSignCodes.BP_DIASTOLIC.getCode()));

		List<UnitValue> ret = new ArrayList<UnitValue>();

		for(Iterator<UnitValue> iter = systolic.iterator(); iter.hasNext();)
		{
			UnitValue sys = iter.next();
			for(UnitValue dia : diastolic)
			{
				if(sys.getTime().equals(dia.getTime()))
				{
					ret.add(new UnitValue(sys.getValue() + "/" + dia.getValue(), sys.getUnit(), sys.getTime()));
					iter.remove();
					diastolic.remove(dia);
					break;
				}
			}
		}
		helper.getBp().addAll(ret);
	}

	private void getSourceColumn(final Entry entry, final VitalSignsVO helper)
	{
		if(entry != null && entry.getOrganizer() != null)
		{
			helper.setSource(Utils.getSourceFromOrganizer(entry.getOrganizer()));
		}
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected String generateXml()
	{
		return null;
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected void createObjects()
	{

	}

	private String loopForValues(final EList<ANY> values)
	{

		final StringBuilder valueString = new StringBuilder();

		for(ANY value : values)
		{

			if(value instanceof PQ)
			{
				final PQ pqValue = (PQ) value;
				if(StringUtils.isNotBlank(pqValue.getValue().toString()))
				{
					valueString.append(pqValue.getValue().toString());
				}
				if(StringUtils.isNotBlank(pqValue.getUnit()) && !"1".equals(pqValue.getUnit()))
				{
					valueString.append(pqValue.getUnit());
				}
				valueString.append("<br/>");
			}
		}

		return valueString.toString();
	}

	private List<UnitValue> loopForUnitValues(final EList<ANY> values, final String time)
	{
		List<UnitValue> ret = new ArrayList<UnitValue>();
		for(ANY value : values)
		{

			if(value instanceof PQ)
			{
				final PQ pqValue = (PQ) value;
				if(StringUtils.isNotBlank(pqValue.getValue().toString()))
				{
					ret.add(new UnitValue(pqValue.getValue().toString(), pqValue.getUnit(), time));
				}
			}
		}

		return ret;
	}

	protected String getValuesFromComponentsForVitalCode(final Entry entry, final String code)
	{
		final StringBuilder resp = new StringBuilder();

		// Check to see if there are any components
		if(entry != null && entry.getOrganizer() != null && entry.getOrganizer().getComponents() != null && !entry.getOrganizer().getComponents().isEmpty())
		{
			for(Component4 component : entry.getOrganizer().getComponents())
			{
				if(component.getObservation() != null && component.getObservation().getCode() != null && component.getObservation().getCode().getCode().equals(code))
				{
					resp.append(this.getValuesFromObservation(component.getObservation(), code));
				}
			}
		}

		return resp.toString();
	}

	protected List<UnitValue> getValueAndUnit(final Entry entry, final String code)
	{
		List<UnitValue> ret = new ArrayList<UnitValue>();
		if(entry != null && entry.getOrganizer() != null && StringUtils.isNotBlank(code))
		{
			ret.addAll(getValueUnitFromObs(entry.getObservation(), code));

			if(CollectionUtils.isNotEmpty(entry.getOrganizer().getObservations()))
			{
				for(Observation obs : entry.getOrganizer().getObservations())
				{
					ret.addAll(getValueUnitFromObs(obs, code));
				}
			}
		}
		return ret;
	}

	protected List<UnitValue> getValueUnitFromObs(final Observation obs, final String code)
	{
		List<UnitValue> ret = new ArrayList<UnitValue>();
		if(obs != null)
		{
			if(obs.getCode() != null && code.equalsIgnoreCase(obs.getCode().getCode()))
			{
				// Add time for multiples. And BP comparison.
				String time;
				if(obs.getEffectiveTime() != null && StringUtils.isNotBlank(obs.getEffectiveTime().getValue()))
				{
					time = obs.getEffectiveTime().getValue();
				}
				else
				{
					time = StringUtils.EMPTY;
				}

				ret.addAll(loopForUnitValues(obs.getValues(), time));
			}
		}

		return ret;
	}

	protected String getValuesFromObservation(final Observation observation, final String code)
	{
		if(observation != null && observation.getCode() != null && StringUtils.isNotBlank(code) && code.equals(observation.getCode().getCode()))
		{
			return this.loopForValues(observation.getValues());
		}

		return StringUtils.EMPTY;
	}
}
