package gov.va.ccd.components.impl;

import gov.va.ccd.service.util.Utils;
import gov.va.ccd.service.value.object.SectionValueObject;
import gov.va.ccd.service.value.objects.impl.ItemObject;
import gov.va.ccd.service.value.objects.impl.MedicationsVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.AssignedEntity;
import org.openhealthtools.mdht.uml.cda.Author;
import org.openhealthtools.mdht.uml.cda.Consumable;
import org.openhealthtools.mdht.uml.cda.Criterion;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.ManufacturedProduct;
import org.openhealthtools.mdht.uml.cda.Material;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.Organization;
import org.openhealthtools.mdht.uml.cda.Participant2;
import org.openhealthtools.mdht.uml.cda.ParticipantRole;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.Person;
import org.openhealthtools.mdht.uml.cda.PlayingEntity;
import org.openhealthtools.mdht.uml.cda.Precondition;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.SubstanceAdministration;
import org.openhealthtools.mdht.uml.cda.Supply;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.MedicationsSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.AD;
import org.openhealthtools.mdht.uml.hl7.datatypes.ADXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.CE;
import org.openhealthtools.mdht.uml.hl7.datatypes.ED;
import org.openhealthtools.mdht.uml.hl7.datatypes.EIVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.ENXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.II;
import org.openhealthtools.mdht.uml.hl7.datatypes.INT;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.ON;
import org.openhealthtools.mdht.uml.hl7.datatypes.PIVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.PN;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.RTO_PQ_PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.SXCM_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.TS;
import org.openhealthtools.mdht.uml.hl7.vocab.SetOperator;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActRelationshipEntryRelationship;
import org.openhealthtools.mdht.uml.hl7.vocab.x_DocumentSubstanceMood;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Mediations Component Section
 * 
 * 
 * FullFillment History is an entire section. Currently only the last one will
 * be populated.
 * 
 * @author Brad Messerle
 * @since 1.0
 * 
 * @param <S> MedicationsComponent
 */
public class MedicationsComponent extends AbstractComponent<MedicationsSection>
{
	private static final Logger logger = LoggerFactory.getLogger(MedicationsComponent.class);
	
	/**
	 * Constructor
	 * 
	 * @param medicationSection
	 * @param ccdDocument
	 */
	public MedicationsComponent(final ContinuityOfCareDocument ccdDocument)
	{
		super(ccdDocument.getMedicationsSection(), ccdDocument);
	}

	@Override
	public void execute()
	{

		// Set the narrative text for the section.
		if(section != null)
		{
			// Create Section Data
			this.createMedicationHelperObjects();
			super.execute();
		}

	}

	@Override
	protected String generateXml()
	{
		StringBuffer sb = new StringBuffer();
		if(CollectionUtils.isNotEmpty(getSectionData()))
		{
			if(CollectionUtils.isNotEmpty(getComments()))
			{
				sb.append("<table ID='_nbMedicationsComments' border='1' width='100%'>");
				sb.append("<thead>");
				sb.append("<tr><th>Comments</th></tr>");
				sb.append("</thead>");
				sb.append("<tbody>");

				for(String comment : this.getComments())
				{
					sb.append("<tr width='100%'>");
					sb.append("<td width='100%'>");
					sb.append("<p>");
					sb.append(comment);
					sb.append("</p>");
					sb.append("</td></tr>");
				}

				sb.append("</tbody>");
				sb.append("</table>");
			}
			sb.append("<table ID='_nbMedications' border='1' width='100%'>\n");

			sb.append("\t<thead>\n");
			sb.append("<tr>");
			sb.append("<th>Medications</th>");
			sb.append("<th>Medication Brand Name</th>");
			sb.append("<th>Medication Status</th>");
			sb.append("<th>Order Date</th>");
			sb.append("<th>Order Expiration</th>");
			sb.append("<th>Provider (Ordering)</th>");
			sb.append("<th>Prescription #</th>");
			sb.append("<th>Fill Status</th>");
			sb.append("<th>Quantity Dispensed</th>");
			sb.append("<th>Dispense Date</th>");
			sb.append("<th>Sig</th>");
			sb.append("<th>Source</th>");
			sb.append("<th>Reaction</th>");

			// Additional fields
			sb.append("<th>Indicate Medication Stopped</th>");
			sb.append("<th>Administrative Timing</th>");
			sb.append("<th>Route Code</th>");
			sb.append("<th>Body Site</th>");
			sb.append("<th>Dose Quanity</th>");
			sb.append("<th>Dose Restriction</th>");
			sb.append("<th>Product Form</th>");
			sb.append("<th>Delivery Method</th>");
			sb.append("<th>Drug Manufacture</th>");
			sb.append("<th>Type of Medication</th>");
			sb.append("<th>Indication</th>");
			sb.append("<th>Vehicle</th>");
			sb.append("<th>Dose Indicator</th>");
			sb.append("<th>Order Number</th>");
			sb.append("<th>Total Fills</th>");
			sb.append("<th>FulFillment Instructions</th>");
			sb.append("<th>Dispensing Provider</th>");
			sb.append("<th>Pharmacy Location</th>");
			sb.append("<th>Fill Number</th>");
			sb.append("</tr>\n");
			sb.append("</thead>\n");
			sb.append("<tbody>\n");

			for(SectionValueObject valueObejct : this.getSectionData())
			{

				final MedicationsVO medicationValueObject = (MedicationsVO) valueObejct;
				sb.append("<tr>");
				sb.append("<td>");
				sb.append(medicationValueObject.getProductName().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getBrandName().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getMedicationStatus().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getOrderDate().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getOrderExpiration().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getOrderingProvider().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getPrescriptionNumber().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getFillStatus().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getQuantityDispensed().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDispenseDate().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				if(!medicationValueObject.getSig().isEmpty())
				{
					sb.append(medicationValueObject.getSig().getValueWithRefDefault());
				}
				else if(!medicationValueObject.getPatientInstructions().isEmpty())
				{
					sb.append(medicationValueObject.getPatientInstructions().getValueWithRefDefault());
				}
				else
				{
					sb.append("--");
				}
				sb.append("</td>");
				sb.append("<td>").append(medicationValueObject.getSource().getValueWithRefDefault()).append("</td>");

				sb.append("<td>");
				sb.append(medicationValueObject.getReaction().getValueWithRefDefault());
				sb.append("</td>");

				// Additional Columns
				sb.append("<td>");
				sb.append(medicationValueObject.getIndicateMedicationStopped().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getEffectiveTime().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getRouteCode().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getSiteCode().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDoseQuanity().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDoseRestriction().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getProductForm().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDeliveryMethod().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDrugManufacture().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getTypeOfMedication().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getIndication().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getVehicle().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getDoseIndicator().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getOrderNumber().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getFills().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getFulfillmentInstructions().getValueWithRefDefault());
				sb.append("</td>");

				sb.append("<td>").append(medicationValueObject.getProvider().getValueWithRefDefault()).append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getPharmacyLocation().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("<td>");
				sb.append(medicationValueObject.getFillNumber().getValueWithRefDefault());
				sb.append("</td>");
				sb.append("</tr>\n");
			}

			sb.append("</tbody>");
			sb.append("</table>");
		}
		return sb.toString();
	}

	private void createMedicationHelperObjects()
	{
		// Loop through all of the entries in the section.
		// This section is only for parsing of the values. All BR's need to be
		// processed elsewhere.
		for(Entry entry : this.section.getEntries())
		{
			// All data should be stored in substance admin.
			SubstanceAdministration subAdmin = entry.getSubstanceAdministration();
			// Make sure there is one.
			if(subAdmin != null)
			{
				// Item to add the values to.
				final MedicationsVO valueObject = new MedicationsVO();
				// Add all items that are direct descendants of substance admin.
				// Those are sections 8.01-8.12
				// 8.01 Free Text Sig.
				this.getSig(subAdmin, valueObject);

				// 8.02 - 8.06
				// This is for processing of medication frequency.
				this.getEffectiveTime(subAdmin, valueObject);

				// 8.07 Route.
				this.getRoute(subAdmin, valueObject);

				// 8.08 Dose
				this.getDoseQuanity(subAdmin, valueObject);

				// 8.09 Site
				this.getSiteCode(subAdmin, valueObject);

				// 8.10 Dose Restriction
				this.getDoseRestriction(subAdmin, valueObject);

				// 8.11 Product Form.
				this.getProductForm(subAdmin, valueObject);

				// 8.12 Delivery Method
				this.getDeliveryMethod(subAdmin, valueObject);

				// Start Consumables 8.13-8.17
				this.processConsumables(subAdmin, valueObject);

				// VA Specific Items. Source et al.
				this.getSourceColumn(subAdmin, valueObject);

				// 8.20
				this.processVehicle(subAdmin, valueObject);
				for(Precondition pre : subAdmin.getPreconditions())
				{
					Criterion crit = pre.getCriterion();
					if(crit != null)
					{
						if(crit.getText() != null)
							valueObject.setDoseIndicator(crit.getText().getText());

					}
				}
				// 8.19 - 8.40 All the rest of the data is contained in
				// entryRelationships.
				this.processEntryRels(subAdmin, valueObject);
				if(valueObject.getMedicationStatus() == null || StringUtils.isBlank(valueObject.getMedicationStatus().getValue()))
				{
					if(subAdmin.getStatusCode() != null && StringUtils.isNotBlank(subAdmin.getStatusCode().getCode()))
					{
						valueObject.setMedicationStatus(new ItemObject(subAdmin.getStatusCode().getCode()));
					}
				}
				// Add the newly created object to the list.
				this.getSectionData().add(valueObject);
			}
			else
			{
				logger.debug("Failed to find substance administration element in Medication entry.  Need to determine if there needs to be extra processing here.");
			}
		}
	}

	private void processVehicle(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		StringBuilder sb = new StringBuilder();
		for(Participant2 part : subAdmin.getParticipants())
		{
			ParticipantRole role = part.getParticipantRole();
			if(role != null)
			{
				PlayingEntity ent = role.getPlayingEntity();
				if(ent != null)
				{
					for(PN name : ent.getNames())
					{
						if(StringUtils.isNotBlank(name.getText()))
						{
							sb.append(name.getText()).append(",");
						}
					}
				}
			}
		}
		if(sb.length() > 0)
		{
			valueObject.setVehicle(sb.substring(0, sb.length() - 1));
		}
	}

	private void processConsumables(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		Consumable consume = subAdmin.getConsumable();
		if(consume != null)
		{
			// Start Manufactured Product.
			ManufacturedProduct prod = consume.getManufacturedProduct();
			if(prod != null)
			{
				// Manufactured Material
				Material material = prod.getManufacturedMaterial();
				if(material != null)
				{
					this.getMaterialInformation(material, valueObject);
				}
				// Manufacturer name. 8.17
				Organization org = prod.getManufacturerOrganization();
				if(org != null)
				{
					StringBuilder sb = new StringBuilder();
					for(ON name : org.getNames())
					{
						if(StringUtils.isNotBlank(name.getText()))
						{
							sb.append(name.getText()).append(",");
						}
					}
					if(sb.length() > 0)
					{
						valueObject.setDrugManufacture(sb.substring(0, sb.length() - 1));
					}
				}
			}
		}
	}

	private void processEntryRels(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		for(EntryRelationship rel : subAdmin.getEntryRelationships())
		{
			processObservations(rel, valueObject);
			Act act = rel.getAct();
			Supply supply = rel.getSupply();
			if(x_ActRelationshipEntryRelationship.RSON.equals(rel.getTypeCode()))
			{
				Observation obs = rel.getObservation();
				if(obs != null)
				{
					if(Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.28"))
					{
						// First look to see if it was placed in a code
						// block
						// 8.21
						CD code = obs.getCode();
						ItemObject item = Utils.getFromCode(code, narrativeText);
						if(item != null && item.hasValue())
						{
							valueObject.setIndication(item);
						}
					}
				}
			}

			if(supply != null)
			{
				// Check for proper moodCode for OrderInformation
				if(x_DocumentSubstanceMood.INT.equals(supply.getMoodCode()))
				{
					// Start order information 8.26-8.31
					// 8.26
					getOrderNumber(supply, valueObject);

					// 8.27
					getFills(supply, valueObject);

					// 8.28 This is currently a noop.
					getQuantityOrdered(supply, valueObject);

					// 8.29
					getOrderExperation(supply, valueObject);

					// 8.30
					getOrderDate(supply, valueObject);

					// 8.31
					getOrderingProvider(supply, valueObject);
				}

				if(x_DocumentSubstanceMood.EVN.equals(supply.getMoodCode()))
				{
					// 8.34 Prescription Number
					getPerscriptionNumber(supply, valueObject);

					// 8.35 & 8.36
					getAssignedPharmInfo(supply, valueObject);

					// 8.37
					getDispenseDate(supply, valueObject);

					// 8.38
					getQuantityDispensed(supply, valueObject);

					// 8.39 - 8.40
					getFillNumberAndStatus(supply, valueObject);

				}
			}
			// Start processing items that do not have an explicit
			// EntryRelationship/typeCode.
			if(act != null)
			{
				ED text = act.getText();
				if(text != null)
				{
					// 8.32 2.16.840.1.113883.10.20.1.43
					if(Utils.containsTemplateId(act.getTemplateIds(), "2.16.840.1.113883.10.20.1.43"))
					{
						if(StringUtils.isNotBlank(text.getText()))
						{
							valueObject.setFulfillmentInstructions(text.getText());
						}
					}
					else if(text.getReference() != null && StringUtils.isNotBlank(text.getReference().getValue()))
					{
						// 8.22 Patient Instructions Specifies a templateID that
						// Kaiser does not use.
						// Skip that test for now.

						// There is a reference value not actual.
						String refval = Utils.getReferenceValue(narrativeText, text.getReference().getValue());
						if(StringUtils.isNotBlank(refval))
						{
							// Found a reference and a value. Store it.
							valueObject.setPatientInstructions(new ItemObject(text.getReference().getValue(), refval));
						}
					}
					else if(StringUtils.isNotBlank(text.getText()))
					{
						// Check to see if freetext is actually in original
						// text.
						valueObject.setPatientInstructions(new ItemObject(text.getText()));
					}
				}
			}
		}
	}

	private void processObservations(final EntryRelationship rel, final MedicationsVO valueObject)
	{
		Observation obs = rel.getObservation();
		if(obs == null)
			return;

		switch(rel.getTypeCode())
		{
		case SUBJ:
			// 8.19 Type of medication
			if(Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.3.88.11.83.8.1"))
			{
				// First look to see if it was placed in a code
				// block
				CD code = obs.getCode();
				valueObject.setTypeOfMedication(Utils.getFromCode(code, narrativeText));
			}
			break;
		case REFR:
			// 8.20 Status of Medication
			if(Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.47"))
			{

				if(CollectionUtils.isNotEmpty(obs.getValues()))
				{
					StringBuilder sb = new StringBuilder();
					for(ANY val : obs.getValues())
					{
						if(val instanceof CD)
						{
							// CD is one of the main types that
							// contains a display name.
							CD cd = (CD) val;
							ItemObject item = Utils.getFromCode(cd, narrativeText);
							if(item != null && item.hasValue())
							{
								sb.append(item.getValue()).append(",");
							}
						}
					}
					if(sb.length() > 0)
					{
						valueObject.setMedicationStatus(sb.substring(0, sb.length() - 1));
					}
				}

			}
			break;
		case CAUS:
			// 8.23 Reaction
			if(Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.1.54"))
			{
				ED text = obs.getText();
				if(text != null)
				{
					if(text.getReference() != null && StringUtils.isNotBlank(text.getReference().getValue()))
					{
						// There is a reference value not actual.
						String refval = Utils.getReferenceValue(narrativeText, text.getReference().getValue());
						if(StringUtils.isNotBlank(refval))
						{
							// Found a reference and a value. Store it.
							valueObject.setReaction(new ItemObject(refval, text.getReference().getValue()));
						}
					}
					else if(StringUtils.isNotBlank(text.getText()))
					{
						// Check to see if freetext is actually in
						// original
						// text.
						valueObject.setReaction(text.getText());
					}
				}
			}
			break;
		default:
			logger.debug("Found an entry with an observation that is not able to be processed.");
			break;
		}
	}

	private void getOrderDate(final Supply supply, final MedicationsVO valueObject)
	{

		if(CollectionUtils.isNotEmpty(supply.getAuthors()))
		{
			for(Author author : supply.getAuthors())
			{
				final TS timeValue = author.getTime();
				if(timeValue != null && StringUtils.isNotBlank(timeValue.getValue()))
				{
					valueObject.setOrderDate(timeValue.getValue());
				}
			}
		}

		if(valueObject.getOrderDate().isEmpty())
		{
			valueObject.setOrderDate(EMPTY_FIELD);
		}
	}

	private void getOrderExperation(final Supply supply, final MedicationsVO valueObject)
	{
		if(CollectionUtils.isNotEmpty(supply.getEffectiveTimes()))
		{
			for(SXCM_TS time : supply.getEffectiveTimes())
			{
				if(StringUtils.isNotBlank(time.getValue()))
				{
					valueObject.setOrderExpiration(time.getValue());
				}
			}
		}

		if(valueObject.getOrderExpiration().isEmpty())
		{
			valueObject.setOrderExpiration(EMPTY_FIELD);
		}
	}

	private void getPerscriptionNumber(final Supply supply, final MedicationsVO valueObject)
	{
		if(supply != null && supply.getIds() != null)
		{
			for(II id : supply.getIds())
			{
				if(id.getExtension() != null && !id.getExtension().equalsIgnoreCase("nullflavor"))
				{
					valueObject.setPrescriptionNumber(id.getExtension());
				}
			}
		}

		// still not found put the -- string
		if(valueObject.getPrescriptionNumber().isEmpty())
		{
			valueObject.setPrescriptionNumber(EMPTY_FIELD);
		}

	}

	private void getQuantityDispensed(final Supply supply, final MedicationsVO valueObject)
	{
		final PQ quantity = supply.getQuantity();
		if(quantity != null)
		{
			valueObject.setQuantityDispensed(quantity.getValue().toString());
		}

		// TODO: Move to business rules.
		// if (valueObject.getQuantityDispensed() == null
		// || (valueObject.getQuantityDispensed() != null &&
		// valueObject.getQuantityDispensed().isEmpty())) {
		// final IVL_PQ doseQnty =
		// supply.getSubstanceAdministration().getDoseQuantity();
		//
		// if (doseQnty != null && doseQnty.getValue() != null) {
		// valueObject.setQuantityDispensed(doseQnty.getValue().toString());
		// }
		// }

		// still not found put the -- string
		if(valueObject.getQuantityDispensed().isEmpty())
		{
			valueObject.setQuantityDispensed(EMPTY_FIELD);
		}

	}

	private void getDispenseDate(final Supply supply, final MedicationsVO valueObject)
	{
		if(CollectionUtils.isNotEmpty(supply.getEffectiveTimes()))
		{
			for(SXCM_TS time : supply.getEffectiveTimes())
			{
				if(StringUtils.isNotBlank(time.getValue()))
				{
					valueObject.setDispenseDate(time.getValue());
				}
			}
		}

		// still not found put the -- string
		if(valueObject.getDispenseDate().isEmpty())
		{
			valueObject.setDispenseDate(EMPTY_FIELD);
		}

	}

	private void getSig(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{

		final ED text = subAdmin.getText();

		if(text != null && text.getReference() != null)
		{

			final String refId = text.getReference().getValue();

			if(refId != null)
			{

				final String refValue = Utils.getReferenceValue(narrativeText, refId);
				valueObject.setSig(new ItemObject(refValue, refId));
			}
		}
		else if(text != null && StringUtils.isNotBlank(text.getText()))
		{
			valueObject.setSig(new ItemObject(text.getText()));
		}

		// // Check 8.22
		// if (valueObject.getSig() == null || valueObject.getSig().isEmpty()) {
		// if (subAdmin.getAct() != null &&
		// !subAdmin.getAct().getTemplateIds().isEmpty()) {
		// final II tempId = subAdmin.getAct().getTemplateIds().get(0);
		// valueObject.setSig(tempId.getRoot());
		// }
		// }

	}

	private void getSourceColumn(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		for(Author author : subAdmin.getAuthors())
		{
			if(author.getAssignedAuthor().getRepresentedOrganization() != null && CollectionUtils.isNotEmpty(author.getAssignedAuthor().getRepresentedOrganization().getNames()))
			{
				// TODO: Update this to loop through orgs and concatenate names.
				valueObject.setSource(author.getAssignedAuthor().getRepresentedOrganization().getNames().get(0).getText());
			}
		}

		// still not found put the -- string
		if(valueObject.getSource().isEmpty())
		{
			valueObject.setSource(EMPTY_FIELD);
		}

	}

	private void getEffectiveTime(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		StringBuilder ret = new StringBuilder();

		for(SXCM_TS time : subAdmin.getEffectiveTimes())
		{
			if(time instanceof PIVL_TS)
			{
				ret.append(processPIVL((PIVL_TS) time)).append(" ");
			}
			else if(time instanceof EIVL_TS)
			{
				ret.append(processEIVL((EIVL_TS) time)).append(" ");
			}
			else if(time instanceof IVL_TS)
			{
				ret.append(processIVL((IVL_TS) time, valueObject)).append(" ");
			}
			else if(StringUtils.isNotBlank(time.getValue()))
			{
				// Fall through test
				ret.append("Once on " + time.getValue());
			}
		}

		if(ret.length() > 0)
		{
			valueObject.setEffectiveTime(ret.toString());
		}
		else
		{
			valueObject.setEffectiveTime(EMPTY_FIELD);
		}
	}

	/**
	 * Will set Indicate med stopped only if time.getHigh != null and
	 * time.getLow == null
	 * 
	 * Otherwise this will return the time string for indicating timing.
	 * 
	 * @param time
	 * @param vo
	 * @return
	 */
	private String processIVL(final IVL_TS time, final MedicationsVO vo)
	{
		if(time == null)
		{
			return StringUtils.EMPTY;
		}
		String timeLow;
		String timeHigh;

		if(time.getLow() != null)
		{
			timeLow = time.getLow().getValue();
		}
		else
		{
			timeLow = StringUtils.EMPTY;
		}

		if(time.getHigh() != null)
		{
			timeHigh = time.getHigh().getValue();
		}
		else
		{
			timeHigh = StringUtils.EMPTY;
		}

		if(StringUtils.isNotBlank(timeHigh) && StringUtils.isBlank(timeLow))
		{
			// Indicate MedStopped
			vo.setIndicateMedicationStopped(timeHigh);
			return StringUtils.EMPTY;
		}

		if(StringUtils.isNotBlank(timeLow) && StringUtils.isNotBlank(timeHigh))
		{
			return "From: " + timeLow + " To: " + timeHigh;
		}

		if(StringUtils.isNotBlank(time.getValue()))
		{
			return "Once on " + time.getValue();
		}

		if(StringUtils.isNotBlank(timeLow))
		{
			return "From: " + timeLow;
		}

		return StringUtils.EMPTY;
	}

	private String processPIVL(final PIVL_TS time)
	{
		if(time == null)
		{
			return null;
		}

		StringBuilder effectiveTime = new StringBuilder();

		if(time.getOperator() != null && time.getOperator().equals(SetOperator.A) && time.getPeriod() != null && time.getPeriod().getValue() != null)
		{
			if(time.getInstitutionSpecified() && "h".equalsIgnoreCase(time.getPeriod().getUnit()))
			{
				Double value = time.getPeriod().getValue().doubleValue();
				Double period = 0.0;
				if(value != null)
				{
					if(value < 1.0)
					{
						period = 24.0 * value;
					}
					else
					{
						period = 24.0 / value;
					}
				}

				effectiveTime.append(period);
				effectiveTime.append(" ");
				effectiveTime.append("time(s) a day");
			}
			else
			{
				effectiveTime.append("every ");
				effectiveTime.append(time.getPeriod().getValue());
				effectiveTime.append(" ");
				effectiveTime.append(time.getPeriod().getUnit()).append(" ");
				String phase = processPhase(time.getPhase());
				if(StringUtils.isNotBlank(phase))
				{
					effectiveTime.append("for ").append(phase).append(" ");
				}

			}
		}
		else if(time.getPhase() != null)
		{
			effectiveTime.append(" ");
			effectiveTime.append("for ");
			effectiveTime.append(processPhase(time.getPhase()));
		}
		return effectiveTime.toString().trim();
	}

	private String processPhase(final IVL_TS phase)
	{
		StringBuilder sb = new StringBuilder();

		if(phase != null)
		{
			sb.append(phase.getWidth().getValue());
			sb.append(" ");
			sb.append(phase.getWidth().getUnit());

			if(phase.getLow() != null)
			{
				String time = phase.getLow().getValue();
				if(StringUtils.isNotBlank(time))
				{
					sb.append(" at ");
					sb.append(time.substring(time.length() - 4));
				}
			}
		}
		return sb.toString().trim();
	}

	private String processEIVL(final EIVL_TS time)
	{
		if(time == null)
		{
			return null;
		}

		StringBuilder ret = new StringBuilder();

		if(SetOperator.A.equals(time.getOperator()))
		{
			if(time.getEvent() != null && StringUtils.isNotBlank(time.getEvent().getCode()))
			{
				// Cycle through all of the possible code values to get the
				// proper return.
				if("AC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before meal");
				}
				else if("ACD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before lunch");
				}
				else if("ACM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before breakfast");
				}
				else if("ACV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("before dinner");
				}
				else if("HS".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("hour of sleep");
				}
				else if("IC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between meals");
				}
				else if("ICD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between lunch and dinner");
				}
				else if("ICM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between breakfast and lunch");
				}
				else if("ICV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("between dinner and hour of sleep");
				}
				else if("PC".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after meal");
				}
				else if("PCD".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after lunch");
				}
				else if("PCM".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after breakfast");
				}
				else if("PCV".equalsIgnoreCase(time.getEvent().getCode()))
				{
					ret.append("after dinner");
				}
			}
			if(ret.length() > 1)
			{
				return ret.toString();
			}
		}
		return null;
	}

	private void getRoute(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		final CE routeCode = subAdmin.getRouteCode();
		if(routeCode != null)
		{
			valueObject.setRouteCode(Utils.getFromCode(routeCode, narrativeText));
		}

		// still not found put the -- string
		if(valueObject.getRouteCode().isEmpty())
		{
			valueObject.setRouteCode(EMPTY_FIELD);
		}

	}

	// TODO: Update to deal with translations.
	private void getDoseQuanity(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		final IVL_PQ doseQuantity = subAdmin.getDoseQuantity();
		final StringBuilder sb = new StringBuilder();
		if(doseQuantity != null)
		{
			if(doseQuantity.getValue() != null)
			{
				sb.append(doseQuantity.getValue().toString());
			}

			if(StringUtils.isNotBlank(doseQuantity.getUnit()) && !"1".equals(doseQuantity.getUnit()))
			{
				sb.append(" ").append(doseQuantity.getUnit());
			}
		}
		valueObject.setDoseQuanity(sb.toString());
		// still not found put the -- string
		if(valueObject.getDoseQuanity().isEmpty())
		{
			valueObject.setDoseQuanity(EMPTY_FIELD);
		}
	}

	private void getSiteCode(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		if(CollectionUtils.isNotEmpty(subAdmin.getApproachSiteCodes()))
		{
			StringBuilder sb = new StringBuilder();
			for(CD siteCode : subAdmin.getApproachSiteCodes())
			{
				ItemObject item = Utils.getFromCode(siteCode, narrativeText);
				if(item != null)
				{
					if(subAdmin.getApproachSiteCodes().size() < 2)
					{
						valueObject.setSiteCode(item);
						break;
					}
					sb.append(item.getValue()).append(",");
					if(item.hasReference())
					{
						logger.error("Going to lose a reference for " + item);
					}
				}
			}
			if(sb.length() > 0)
			{
				valueObject.setSiteCode(sb.substring(0, sb.length() - 1));
			}
		}

		// still not found put the -- string
		if(valueObject.getSiteCode().isEmpty())
		{
			valueObject.setSiteCode(EMPTY_FIELD);
		}

	}

	// TODO: Update with check for translations.
	private void getDoseRestriction(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		final RTO_PQ_PQ maxDoseQnty = subAdmin.getMaxDoseQuantity();

		final StringBuilder maxDoseString = new StringBuilder();

		if(maxDoseQnty != null)
		{

			if(maxDoseQnty.getNumerator() != null)
			{
				PQ num = maxDoseQnty.getNumerator();
				if(num.getValue() != null)
				{
					maxDoseString.append(num.getValue().toString());
				}
				if(StringUtils.isNotBlank(num.getUnit()))
				{
					maxDoseString.append(num.getUnit());
				}
			}

			if(maxDoseQnty.getDenominator() != null)
			{
				maxDoseString.append(" / ");
				if(maxDoseQnty.getDenominator().getValue() != null)
				{
					maxDoseString.append(maxDoseQnty.getDenominator().getValue().toString());
				}
				if(StringUtils.isNotBlank(maxDoseQnty.getDenominator().getUnit()))
				{
					maxDoseString.append(maxDoseQnty.getDenominator().getUnit());
				}
			}
		}

		valueObject.setDoseRestriction(maxDoseString.toString());
		// still not found put the -- string
		if(valueObject.getDoseRestriction().isEmpty())
		{
			valueObject.setDoseRestriction(EMPTY_FIELD);
		}

	}

	private void getProductForm(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		final CE productForm = subAdmin.getAdministrationUnitCode();
		if(productForm != null)
		{
			valueObject.setProductForm(Utils.getFromCode(productForm, narrativeText));
		}

		// still not found put the -- string
		if(valueObject.getProductForm().isEmpty())
		{
			valueObject.setProductForm(EMPTY_FIELD);
		}

	}

	private void getDeliveryMethod(final SubstanceAdministration subAdmin, final MedicationsVO valueObject)
	{
		final CD deliveryMethod = subAdmin.getCode();

		if(deliveryMethod != null)
		{
			valueObject.setDeliveryMethod(Utils.getFromCode(deliveryMethod, narrativeText));
		}

		// still not found put the -- string
		if(valueObject.getDeliveryMethod().isEmpty())
		{
			valueObject.setDeliveryMethod(EMPTY_FIELD);
		}

	}

	/**
	 * Will set the coded product name, product Name, coded brand name, and
	 * brand name
	 * 
	 * @param material
	 * @param valueObject
	 */
	private void getMaterialInformation(final Material material, final MedicationsVO valueObject)
	{
		CE code = material.getCode();
		if(code != null)
		{
			// Look for free text product Name 8.15
			ED oText = code.getOriginalText();
			if(oText != null)
			{
				if(oText.getReference() != null && StringUtils.isNotBlank(oText.getReference().getValue()))
				{
					// There is a reference value not actual.
					String refval = Utils.getReferenceValue(narrativeText, oText.getReference().getValue());
					if(StringUtils.isNotBlank(refval))
					{
						// Found a reference and a value. Store it.
						valueObject.setProductName(new ItemObject(refval, oText.getReference().getValue()));
					}
				}
				else if(StringUtils.isNotBlank(oText.getText()))
				{
					// Check to see if freetext is actually in original text.
					valueObject.setProductName(oText.getText());
				}
			}

			// Look for freetext brand name 8.16
			if(material.getName() != null)
			{
				if(StringUtils.isNotBlank(material.getName().getText()))
				{
					valueObject.setBrandName(material.getName().getText());
				}
			}
		}
	}

	private void getOrderNumber(final Supply supply, final MedicationsVO valueObject)
	{
		for(II id : supply.getIds())
		{
			if(StringUtils.isNotBlank(id.getExtension()))
			{
				valueObject.setOrderNumber(id.getExtension());
			}
		}

		// still not found put the -- string
		if(valueObject.getOrderNumber().isEmpty())
		{
			valueObject.setOrderNumber(EMPTY_FIELD);
		}

	}

	private void getFills(final Supply supply, final MedicationsVO valueObject)
	{

		if(supply.getRepeatNumber() != null && supply.getRepeatNumber().getValue() != null)
		{
			valueObject.setFills(supply.getRepeatNumber().getValue().toString());
		}

		// still not found put the -- string
		if(valueObject.getFills().isEmpty())
		{
			valueObject.setFills(EMPTY_FIELD);
		}

	}

	/**
	 * Currently a noop. The customer has requested this not be displayed.
	 * 
	 * @param supply
	 * @param valueObject
	 */
	private void getQuantityOrdered(final Supply supply, final MedicationsVO valueObject)
	{
	}

	private void getOrderingProvider(final Supply supply, final MedicationsVO valueObject)
	{

		final StringBuilder nameString = new StringBuilder();

		for(Author author : supply.getAuthors())
		{

			if(author.getAssignedAuthor() != null && author.getAssignedAuthor().getAssignedPerson() != null)
			{

				for(PN name : author.getAssignedAuthor().getAssignedPerson().getNames())
				{
					nameString.append(this.createName(name));
				}
			}
		}

		valueObject.setOrderingProvider(nameString.toString());

		// still not found put the -- string
		if(valueObject.getOrderingProvider().isEmpty())
		{
			valueObject.setOrderingProvider(EMPTY_FIELD);
		}

	}

	// private void getFulfillmentInstructions(final Entry entry, final
	// MedicationsVO valueObject) {
	//
	// for (EntryRelationship er :
	// entry.getSubstanceAdministration().getEntryRelationships()) {
	// if (er.getAct() != null && er.getAct().getText() != null) {
	// valueObject.setFulfillmentInstructions(er.getAct().getText().getText());
	// }
	// }
	//
	// // still not found put the -- string
	// if (StringUtils.isBlank(valueObject.getFulfillmentInstructions())) {
	// valueObject.setFulfillmentInstructions(EMPTY_FIELD);
	// }
	//
	// }
	//
	// private void getFulfillmentHistory(final Entry entry, final MedicationsVO
	// valueObject) {
	//
	// for (Supply supply : entry.getSubstanceAdministration().getSupplies()) {
	// if (supply.getMoodCode().equals(x_DocumentSubstanceMood.EVN)) {
	// if (supply.getText() != null) {
	// valueObject.setFulfillmentHistory(supply.getText().getText());
	// }
	// }
	// }
	//
	// // still not found put the -- string
	// if (StringUtils.isBlank(valueObject.getFulfillmentHistory())) {
	// valueObject.setFulfillmentHistory(EMPTY_FIELD);
	// }
	//
	// }
	//

	/**
	 * Will process the supply to find 8.35 & 8.36 Pharmacy Location and
	 * dispensing provider.
	 * 
	 * @param supply
	 * @param valueObject
	 */
	private void getAssignedPharmInfo(final Supply supply, final MedicationsVO valueObject)
	{

		final StringBuilder pharmacyLocation = new StringBuilder();

		for(Performer2 performer : supply.getPerformers())
		{
			AssignedEntity assigned = performer.getAssignedEntity();
			if(assigned != null)
			{
				Person person = assigned.getAssignedPerson();
				// 8.35 Get the name of the person dispensing.
				if(person != null)
				{
					StringBuilder sb = new StringBuilder();
					for(PN name : person.getNames())
					{
						String val = createName(name);
						if(StringUtils.isNotBlank(val))
						{
							sb.append(val).append(",");
						}
					}
					if(sb.length() > 0)
					{
						valueObject.setProvider(new ItemObject(sb.substring(0, sb.length() - 1)));
					}
				}

				// 8.36 Get the parmacy location.
				for(AD address : assigned.getAddrs())
				{
					pharmacyLocation.append(this.createAddress(address));
				}
			}
		}

		valueObject.setPharmacyLocation(pharmacyLocation.toString());

		// still not found put the -- string
		if(!valueObject.getPharmacyLocation().hasValue())
		{
			valueObject.setPharmacyLocation(EMPTY_FIELD);
		}

	}

	private void getFillNumberAndStatus(final Supply supply, final MedicationsVO valueObject)
	{
		for(EntryRelationship rel : supply.getEntryRelationships())
		{
			if(x_ActRelationshipEntryRelationship.COMP.equals(rel.getTypeCode()))
			{
				INT seq = rel.getSequenceNumber();
				if(seq != null && seq.getValue() != null)
				{
					valueObject.setFillNumber(rel.getSequenceNumber().getValue().toString());
				}
				CD code = supply.getStatusCode();
				valueObject.setFillStatus(Utils.getFromCode(code, narrativeText));
			}
		}

		// still not found put the -- string
		if(!valueObject.getFillNumber().hasValue())
		{
			valueObject.setFillNumber(EMPTY_FIELD);
		}

		// still not found put the -- string
		if(!valueObject.getFillStatus().hasValue())
		{
			valueObject.setFillStatus(EMPTY_FIELD);
		}

	}

	/**
	 * @return the narrativeText
	 */
	public final StrucDocText getNarrativeText()
	{
		return narrativeText;
	}

	/**
	 * @param narrativeText the narrativeText to set
	 */
	public final void setNarrativeText(StrucDocText narrativeText)
	{
		this.narrativeText = narrativeText;
	}

	@Override
	protected void createObjects()
	{

	}

	private String createName(final PN name)
	{
		final StringBuilder sb = new StringBuilder();

		if(StringUtils.isNotBlank(name.getText()))
		{
			return name.getText();
		}

		if(StringUtils.isNotBlank(loopForName(name.getPrefixes())))
		{
			sb.append(loopForName(name.getPrefixes())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getGivens())))
		{
			sb.append(loopForName(name.getGivens())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getFamilies())))
		{
			sb.append(loopForName(name.getFamilies())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getSuffixes())))
		{
			sb.append(loopForName(name.getSuffixes())).append(" ");
		}
		return sb.toString();
	}

	private String loopForName(final EList<ENXP> names)
	{
		for(ENXP name : names)
		{
			if(StringUtils.isNotBlank(name.getText()))
			{
				return name.getText();
			}
		}

		return null;
	}

	private String createAddress(final AD address)
	{
		StringBuilder sb = new StringBuilder();
		for(ADXP street : address.getStreetAddressLines())
		{
			if(StringUtils.isNotBlank(street.getText()))
			{
				sb.append(street.getText()).append(" ");
			}
		}

		for(ADXP city : address.getCities())
		{
			if(StringUtils.isNotBlank(city.getText()))
			{
				sb.append(city.getText()).append(", ");
			}
		}

		for(ADXP state : address.getStates())
		{
			if(StringUtils.isNotBlank(state.getText()))
			{
				sb.append(state.getText()).append(" ");
			}
		}

		for(ADXP zip : address.getPostalCodes())
		{
			if(StringUtils.isNotBlank(zip.getText()))
			{
				sb.append(zip.getText());
			}
		}

		return sb.toString();
	}
}
