package gov.va.ccd.components.impl;

import gov.va.ccd.components.transform.Transformer;
import gov.va.ccd.service.util.EffTimeUtils;
import gov.va.ccd.service.util.Utils;
import gov.va.ccd.service.value.objects.impl.ConditionProblemsVO;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.joda.time.DateTime;
import org.joda.time.Years;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.AssignedEntity;
import org.openhealthtools.mdht.uml.cda.DocumentationOf;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.Performer1;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.RecordTarget;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.ProblemSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.EN;
import org.openhealthtools.mdht.uml.hl7.datatypes.II;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActRelationshipEntryRelationship;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to generate the narrative block with help of Conditions
 * and Problems with helper.
 * 
 * @author vacokandlm
 * 
 * @param <S> ProblemSection
 */
public class ConditionProblemsComponent extends AbstractComponent<ProblemSection>
{
	private static final Logger logger = LoggerFactory.getLogger(ConditionProblemsComponent.class);

	private DateTime dateOfBirth = null;
	Map<String, String> providerExtNames = new HashMap<String, String>();

	/**
	 * Constructor
	 * 
	 * @param problemSection
	 * @param ccdDocument
	 */
	public ConditionProblemsComponent(final ContinuityOfCareDocument ccdDocument)
	{
		super(ccdDocument.getProblemSection(), ccdDocument);
		// Set the date of birth
		final RecordTarget target = ccdDocument.getRecordTargets().get(0);
		String dob = null;
		if(target != null && target.getPatientRole() != null && target.getPatientRole().getPatient() != null && target.getPatientRole().getPatient().getBirthTime() != null && StringUtils.isNotBlank(target.getPatientRole().getPatient().getBirthTime().getValue()))
		{
			dob = target.getPatientRole().getPatient().getBirthTime().getValue();
		}

		providerExtNames.putAll(getDocumentationOfPerfMap());

		if(StringUtils.isNotBlank(dob))
		{
			dateOfBirth = EffTimeUtils.convertDate(dob);
		}
	}

	@Override
	public void execute()
	{
		if(section != null)
		{

			List<ConditionProblemsVO> rows = this.getProblemRows();
			appendNarrative(this.generateXml(rows));
		}

	}

	private String generateXml(List<ConditionProblemsVO> rows)
	{
		StringBuffer sb = new StringBuffer();
		if(CollectionUtils.isNotEmpty(rows))
		{
			if(CollectionUtils.isNotEmpty(this.getComments()))
			{
				sb.append("<table ID='_nbConditionProblemsComments' border='1' width='100%'>");
				sb.append("<thead>");
				sb.append("<tr><th>Comments</th></tr>");
				sb.append("</thead>");
				sb.append("<tbody>");

				for(String comment : this.getComments())
				{
					sb.append("<tr width='100%'>");
					sb.append("<td width='100%'>");
					sb.append("<p>");
					sb.append(comment);
					sb.append("</p>");
					sb.append("</td></tr>");
				}

				sb.append("</tbody>");
				sb.append("</table>");
			}
			sb.append("<table ID='_nbConditionProblems' border='1' width='100%'>");
			sb.append("<thead>");
			sb.append("<tr>");
			sb.append("<th>Problem</th>");
			sb.append("<th>Status</th>");
			sb.append("<th>Problem Code</th>");
			sb.append("<th>Problem Type</th>");
			sb.append("<th>Date of Onset</th>");
			sb.append("<th>Date Resolved</th>");
			sb.append("<th>Cause of Death</th>");
			sb.append("<th>Age at Onset</th>");
			sb.append("<th>Age at Death</th>");
			sb.append("<th>Time of Death</th>");
			sb.append("<th>Provider</th>");
			sb.append("<th>Source</th>");
			sb.append("</tr>\n");
			sb.append("</thead>");
			sb.append("<tbody>");
			for(ConditionProblemsVO helper : rows)
			{
				sb.append("<tr>");
				sb.append("<td ID=\"" + helper.getProblemRef() + "\">" + Utils.getNonNullValue(helper.getProblem(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getStatus(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getProblemCode(), helper.getProblemCodeTC()) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getProblemTypeDN(), helper.getProblemTypeTCDN(), helper.getProblemTypeOT()) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getDateOnset(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getDateResolved(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getCauseOfDeath(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getAgeOnset(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getAgeAtDeath(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getTimeOfDeath(), null) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getProviderExt(), helper.getProviderName()) + "</td>");
				sb.append("<td>" + Utils.getNonNullValue(helper.getSource(), null) + "</td>");

				sb.append("</tr>");
			}
			sb.append("</tbody>");
			sb.append("</table>");
		}
		return sb.toString();
	}

	private List<ConditionProblemsVO> getProblemRows()
	{
		List<ConditionProblemsVO> list = new ArrayList<ConditionProblemsVO>();
		ConditionProblemsVO helper = null;

		StrucDocText narrativeText = section.getText();
		if(CollectionUtils.isNotEmpty(section.getEntries()))
		{
			for(Entry entry : section.getEntries())
			{
				Act act = entry.getAct();
				if(act != null)
				{
					helper = new ConditionProblemsVO();

					// ProviderExt/Name
					Map<String, String> probPerfs = new HashMap<String, String>();
					if(CollectionUtils.isNotEmpty(act.getPerformers()))
					{
						// For providers without an ID.
						int emptyCount = 0;
						for(Performer2 performer : act.getPerformers())
						{
							if(performer.getAssignedEntity() != null)
							{
								AssignedEntity assignedEntity = performer.getAssignedEntity();
								String id;
								if(CollectionUtils.isNotEmpty(assignedEntity.getIds()))
								{
									id = assignedEntity.getIds().get(0).getExtension();
								}
								else
								{
									id = String.valueOf(emptyCount++);
								}

								String names;
								if(assignedEntity.getAssignedPerson() != null && CollectionUtils.isNotEmpty(assignedEntity.getAssignedPerson().getNames()))
								{
									names = Transformer.createNames(assignedEntity.getAssignedPerson().getNames());
								}
								else
								{
									names = StringUtils.EMPTY;
								}
								if(StringUtils.isNotBlank(id))
								{
									probPerfs.put(id, names);
								}
							}
						}
					}

					// Date Resolved
					if(act.getEffectiveTime() != null && act.getEffectiveTime().getHigh() != null)
					{
						helper.setDateResolved(Utils.getNonNullValue(act.getEffectiveTime().getHigh().getValue(), "--"));
					}
					else
					{
						helper.setDateResolved("--");
					}

					// Date on set
					if(act.getEffectiveTime() != null && act.getEffectiveTime().getLow() != null)
					{
						helper.setDateOnset(act.getEffectiveTime().getLow().getValue());
					}

					// Cause of Death
					for(EntryRelationship er : act.getEntryRelationships())
					{
						if(er.getTypeCode().equals(x_ActRelationshipEntryRelationship.CAUS))
						{

							if(er.getObservation() != null && er.getObservation().getCode() != null)
							{
								final CD code = er.getObservation().getCode();
								if(code.getCode().equals("419620001"))
								{
									if(er.getObservation().getText() != null)
									{
										helper.setCauseOfDeath(er.getObservation().getText().getText());
									}
								}
							}
						}

					}

					final StringBuilder ageOfDeathString = new StringBuilder();
					final StringBuilder ageAtOnset = new StringBuilder();

					for(EntryRelationship er : act.getEntryRelationships())
					{
						// CC Added. For getting Age at Death and Time of Death.
						// And Cause.
						if(x_ActRelationshipEntryRelationship.CAUS.equals(er.getTypeCode()))
						{
							if(er.getObservation() != null)
							{
								Observation obs = er.getObservation();
								if(obs.getEffectiveTime() != null)
								{
									if(StringUtils.isNotBlank(obs.getEffectiveTime().getValue()))
									{
										helper.setTimeOfDeath(obs.getEffectiveTime().getValue());
									}
								}
								for(EntryRelationship innerEr : obs.getEntryRelationships())
								{
									PQ age = Utils.getAge(innerEr);
									if(age != null && ageOfDeathString.length() < 1 && age.getValue() != null)
									{
										ageOfDeathString.append(age.getValue()).append(" ");
										ageOfDeathString.append(Utils.convertPQUnit(age));
										if(ageOfDeathString.length() > 1)
										{
											helper.setAgeAtDeath(ageOfDeathString.toString());
										}
									}
								}
								// Cause of death
								// Run through the usual Suspects
								if(CollectionUtils.isNotEmpty(obs.getValues()))
								{
									for(ANY any : obs.getValues())
									{
										if(any instanceof CD)
										{
											CD code = (CD) any;
											if(StringUtils.isNotBlank(code.getDisplayName()))
											{
												helper.setCauseOfDeath(code.getDisplayName());
											}
										}
									}
								}
								if(StringUtils.isBlank(helper.getCauseOfDeath()) && er.getObservation().getText() != null)
								{
									helper.setCauseOfDeath(er.getObservation().getText().getText());
								}

							}
						}
						else
						{
							// For getting age at onset
							PQ age = Utils.getAge(er);
							if(age != null && ageAtOnset.length() < 1 && age.getValue() != null)
							{
								ageAtOnset.append(age.getValue()).append(" ");
								ageAtOnset.append(Utils.convertPQUnit(age));
								if(ageAtOnset.length() > 1)
								{
									helper.setAgeOnset(ageAtOnset.toString());
								}
							}
						}
					}

					if(StringUtils.isBlank(helper.getAgeOnset()) && dateOfBirth != null && StringUtils.isNotBlank(helper.getDateOnset()))
					{
						// Need to calculate the age.
						final DateTime dateOfOnset = EffTimeUtils.convertDate(helper.getDateOnset());
						int years = Years.yearsBetween(dateOfBirth, dateOfOnset).getYears();
						helper.setAgeOnset(Integer.toString(years));
					}

					if(StringUtils.isBlank(helper.getAgeAtDeath()) && dateOfBirth != null && StringUtils.isNotBlank(helper.getTimeOfDeath()))
					{
						final DateTime dateOfDeath = EffTimeUtils.convertDate(helper.getTimeOfDeath());
						int years = Years.yearsBetween(dateOfBirth, dateOfDeath).getYears();
						helper.setAgeAtDeath(Integer.toString(years));
					}

					String source;
					source = Utils.getSourceFromAct(act);

					helper.setSource(source);

					for(EntryRelationship entryRelationship : act.getEntryRelationships())
					{

						Observation problemObservation = entryRelationship.getObservation();

						if(problemObservation != null)
						{
							// Double check to see if we can get source from the
							// observation.
							if(StringUtils.isBlank(source))
							{
								source = Utils.getSourceFromObservation(problemObservation);
								helper.setSource(source);
							}

							// problem
							if(problemObservation.getText() != null && problemObservation.getText().getReference() != null)
							{
								String problemRef = problemObservation.getText().getReference().getValue();
								if(StringUtils.isNotBlank(problemRef))
								{
									helper.setProblem(Utils.getReferenceValue(narrativeText, problemRef));
									helper.setProblemRef(problemRef);
								}
							}

							// problem code
							if(CollectionUtils.isNotEmpty(problemObservation.getValues()))
							{
								if(problemObservation.getValues().get(0) instanceof CD)
								{

									CD problemCode = (CD) problemObservation.getValues().get(0);

									// CCR 177986
									logger.debug(" observation code values code: {}, code system: {}, code system name: {}, display name: {} ", problemCode.getCode(), problemCode.getCodeSystem(), problemCode.getCodeSystemName(), problemCode.getDisplayName());

									if(problemCode.getCode() != null)
									{
										helper.setProblemCode(problemCode.getCode());
										if("V22.2".equals(problemCode.getCode()))
										{
											// Dealing with pregnancy need to
											// change
											// the
											// problename.
											helper.setProblem("Pregnancy (" + helper.getDateResolved() + ")");
										}
									}
									else
									{
										String problemCodeTC = null;
										if(CollectionUtils.isNotEmpty(problemCode.getTranslations()))
										{
											problemCodeTC = problemCode.getTranslations().get(0).getCode();
											helper.setProblemCodeTC(problemCodeTC);
											if("V22.2".equals(problemCodeTC))
											{
												// Dealing with pregnancy need
												// to
												// change
												// the problename.
												helper.setProblem("Pregnancy ( " + helper.getDateResolved() + " )");
											}
										}
									}
								}

								// Problem Type
								CD problemType = problemObservation.getCode();
								if(problemType != null)
								{
									if(problemType.getDisplayName() != null)
									{
										helper.setProblemTypeDN(problemType.getDisplayName());
									}
									else
									{
										String problemTypeTCDN = null;
										if(CollectionUtils.isNotEmpty(problemType.getTranslations()))
										{

											problemTypeTCDN = problemType.getTranslations().get(0).getDisplayName();
											if(problemTypeTCDN != null)
											{
												helper.setProblemTypeTCDN(problemTypeTCDN);
											}
										}
									}
									// Display Problem Type code originalText
									// Verified by VistA CCD Data Mapping Work
									// Group
									// on 12/11/2013 and Dr. Donahue on
									// 1/22/2014
									if(problemType.getOriginalText() != null)
									{
										// First check for a reference.
										if(problemType.getOriginalText().getReference() != null)
										{
											String ref = problemType.getOriginalText().getReference().getValue();
											if(StringUtils.isNotBlank(ref))
											{
												String val = Utils.getAllRefernceValue(narrativeText.getMixed(), ref);
												if(StringUtils.isNotBlank(val))
												{
													helper.setProblemTypeOT(val);
												}
											}
										}
										else
										{
											if(StringUtils.isNotBlank(problemType.getOriginalText().getText()))
											{
												helper.setProblemTypeOT(problemType.getOriginalText().getText());
											}
										}

									}

								}

							}

							// /ClinicalDocument/component/structuredBody/component/section/entry/act/
							// entryRelationship[typeCode=SUBJ]/observation/entryRelationship[typeCode=REFR]/observation[templateId@root
							// =
							// '2.16.840.1.113883.10.20.1.50']/value/@displayName

							// Status
							for(EntryRelationship entryRelationship1 : problemObservation.getEntryRelationships())
							{

								if(entryRelationship1.getTypeCode() != null && entryRelationship1.getTypeCode().toString().trim().equals("REFR"))
								{
									if(entryRelationship1.getObservation() != null && !entryRelationship1.getObservation().getValues().isEmpty() && entryRelationship1.getObservation().getValues().get(0) instanceof CD)
									{
										CD statusCode = (CD) entryRelationship1.getObservation().getValues().get(0);
										// CCR 177986
										logger.debug(" status code value {}, {}, {}, {} : " + statusCode.getCode() + ", " + statusCode.getCodeSystem() + ", " + statusCode.getCodeSystemName() + ", " + statusCode.getDisplayName());
										String status = statusCode.getDisplayName();
										helper.setStatus(status);
									}

								} // for

							} // problem code

						} // problem observation
					}

					if(!probPerfs.isEmpty())
					{
						// Lets try to link up some names.
						StringBuilder sb = new StringBuilder();

						for(Map.Entry<String, String> entry2 : probPerfs.entrySet())
						{
							// First if there is a match use the Documentation
							// of name.
							if(providerExtNames.containsKey(entry2.getKey()) && StringUtils.isNotBlank(providerExtNames.get(entry2.getValue())))
							{
								if(sb.length() > 0)
								{
									sb.append(", ");
								}
								sb.append(providerExtNames.get(entry2.getValue()));
							}
							else
							{
								if(StringUtils.isNotBlank(entry2.getValue()))
								{
									if(sb.length() > 0)
									{
										sb.append(", ");
									}
									sb.append(entry2.getValue());
								}
							}
						}
						if(sb.length() > 0)
						{
							helper.setProviderName(sb.toString());
						}
					}

					list.add(helper);
				}
			}
		}
		return list;
	}

	private Map<String, String> getDocumentationOfPerfMap()
	{
		Map<String, String> ret = new HashMap<String, String>();

		if(CollectionUtils.isNotEmpty(ccdDocument.getDocumentationOfs()))
		{
			for(DocumentationOf docOf : ccdDocument.getDocumentationOfs())
			{
				if(docOf.getServiceEvent() != null && CollectionUtils.isNotEmpty(docOf.getServiceEvent().getPerformers()))
				{
					for(Performer1 performer : docOf.getServiceEvent().getPerformers())
					{
						AssignedEntity ent = performer.getAssignedEntity();
						if(ent != null)
						{
							String id;
							if(CollectionUtils.isNotEmpty(ent.getIds()))
							{
								id = ent.getIds().get(0).getExtension();
							}
							else
							{
								id = StringUtils.EMPTY;
							}
							if(ent.getAssignedPerson() != null && CollectionUtils.isNotEmpty(ent.getAssignedPerson().getNames()))
							{
								String names = Transformer.createNames(ent.getAssignedPerson().getNames());
								if(StringUtils.isNotBlank(names) && StringUtils.isNotBlank(id))
								{
									ret.put(id, names);
								}
							}
						}
					}
				}
			}
		}
		return ret;
	}

	/**
	 * provider Name and extension from conditions section.
	 * 
	 * @param list
	 * @return
	 */

	protected Map<String, String> extractProviderExtOrName2(EList<Performer2> list)
	{
		Map<String, String> providerExtNames1 = new HashMap<String, String>();
		String providerExt = null, providerName = null;
		for(Performer2 performer : list)
		{

			providerExt = null;
			providerName = null;

			AssignedEntity assignedEntity = performer.getAssignedEntity();
			EList<II> ids = assignedEntity.getIds();
			for(II id : ids)
			{
				if(id.getExtension() != null)
				{
					providerExt = id.getExtension();

				}
			}

			if(assignedEntity.getAssignedPerson() != null)
			{
				if(assignedEntity.getAssignedPerson().getNames() != null && !assignedEntity.getAssignedPerson().getNames().isEmpty())
				{
					if(CollectionUtils.isNotEmpty(assignedEntity.getAssignedPerson().getNames()))
					{
						StringBuilder sb = new StringBuilder();
						EN en = assignedEntity.getAssignedPerson().getNames().get(0);
						if(CollectionUtils.isNotEmpty(en.getPrefixes()))
							sb.append(en.getPrefixes().get(0).getText()).append(" ");
						if(CollectionUtils.isNotEmpty(en.getGivens()))
							sb.append(en.getGivens().get(0).getText()).append(" ");
						if(CollectionUtils.isNotEmpty(en.getFamilies()))
							sb.append(en.getFamilies().get(0).getText()).append(" ");
						providerName = sb.toString();
					}
				}

			}
		}
		providerExtNames1.put(providerExt, providerName);

		return providerExtNames1;
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected String generateXml()
	{
		return null;
	}

	/**
	 * No-op handled above.
	 */
	@Override
	protected void createObjects()
	{

	}

}