package gov.va.med.nhin.adapter.xmlutils;

import java.util.Comparator;
import java.util.List;
import java.util.TreeMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author David Vazquez
 */
public class XMLMerge
{
	private static final Logger logger = LoggerFactory.getLogger(XMLMerge.class.getName());
	private final Document map;
	private boolean strict = true;
	private XMLNamespaceContext namespaceContext;

	private interface Id
	{
		public String getIdXPath();

		public String getId(Node node) throws Exception;
	}

	private class DefaultId implements Id
	{
		private long id = 0;

		public String getIdXPath()
		{
			return "DEFAULT";
		}

		public String getId(Node node) throws Exception
		{
			return Long.toString(++id);
		}
	}

	private class NodeId implements Id
	{
		private XPathExpression exprId;
		private String idXPath;

		public NodeId(String idXPath, XPathExpression exprId)
		{
			this.idXPath = idXPath;
			this.exprId = exprId;
		}

		public String getIdXPath()
		{
			return idXPath;
		}

		public String getId(Node node) throws Exception
		{
			Node docId = (Node) exprId.evaluate(node, XPathConstants.NODE);
			return docId != null ? docId.getTextContent() : null;
		}
	}

	private interface Sort
	{
		public String getSortXPath();

		public String getSort(Node node) throws Exception;

		public String getSortOrder();
	}

	private class DefaultSort implements Sort
	{
		public String getSortXPath()
		{
			return "";
		}

		public String getSort(Node node) throws Exception
		{
			return "";
		}

		public String getSortOrder()
		{
			return "";
		}
	}

	private class NodeSort implements Sort
	{
		private String sortXPath;
		private XPathExpression exprSort;
		private String sortOrder;

		public NodeSort(String sortXPath, XPathExpression exprSort, String sortOrder)
		{
			this.sortXPath = sortXPath;
			this.exprSort = exprSort;
			this.sortOrder = sortOrder;
		}

		public String getSortXPath()
		{
			return sortXPath;
		}

		public String getSort(Node node) throws Exception
		{
			Node docSort = (Node) exprSort.evaluate(node, XPathConstants.NODE);
			return docSort != null ? docSort.getTextContent() : null;
		}

		public String getSortOrder()
		{
			return sortOrder;
		}
	}

	private class NodeMapKey implements Comparable
	{
		public String id;
		public String sortValue;

		public NodeMapKey(String id, String sortValue)
		{
			this.id = id;
			this.sortValue = sortValue;
		}

		@Override
		public boolean equals(Object obj)
		{
			if(!(obj instanceof NodeMapKey))
			{
				return false;
			}

			NodeMapKey key = (NodeMapKey) obj;
			return id.equalsIgnoreCase(key.id);
		}

		@Override
		public int hashCode()
		{
			return id.hashCode();
		}

		public int compareTo(Object obj)
		{
			if(!(obj instanceof NodeMapKey))
			{
				return -1;
			}

			NodeMapKey key = (NodeMapKey) obj;
			int ret = id.compareToIgnoreCase(key.id);
			if(ret != 0)
			{
				int ret2 = sortValue.compareToIgnoreCase(key.sortValue);
				ret = ret2 != 0 ? ret2 : ret;
			}

			return ret;
		}
	}

	private class NodeMapKeyComparator implements Comparator<NodeMapKey>
	{
		public String sortOrder;

		public NodeMapKeyComparator(String sortOrder)
		{
			this.sortOrder = sortOrder;
		}

		public int compare(NodeMapKey left, NodeMapKey right)
		{
			int ret = left.compareTo(right);
			return (sortOrder.equalsIgnoreCase("desc") ? -1 : 1) * ret;
		}
	}

	public boolean isStrict()
	{
		return strict;
	}

	public void setStrict(boolean strict)
	{
		this.strict = strict;
	}

	/**
	 * Construct an XMLMerge object. An XMLUtilsException will be thrown if the
	 * mapping file cannot be parsed.
	 *
	 * @param mapFilename path of the file that contains the mapping
	 *            configuration.
	 * @throws XMLUtilsException if the mapping file cannot be parsed.
	 */
	public XMLMerge(String mapFilename) throws XMLUtilsException
	{
		try
		{
			XPathFactory xpathFactory = XPathFactory.newInstance();
			XPath xpathMap = xpathFactory.newXPath();
			DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
			documentBuilderFactory.setNamespaceAware(true);

			DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();

			map = documentBuilder.parse(mapFilename);

			NodeList namespaceMappings = (NodeList) xpathMap.evaluate("/template/sourceFile/namespaceMappings/namespaceMapping", map, XPathConstants.NODESET);
			if(namespaceMappings != null)
			{
				namespaceContext = new XMLNamespaceContext();
				for(int i = 0; i < namespaceMappings.getLength(); ++i)
				{
					String prefix = (String) xpathMap.evaluate("prefix", namespaceMappings.item(i), XPathConstants.STRING);
					String uri = (String) xpathMap.evaluate("uri", namespaceMappings.item(i), XPathConstants.STRING);
					namespaceContext.addMapping(prefix, uri);
				}
			}
		}
		catch(Throwable t)
		{
			throw new XMLUtilsException("An error occurred when constructing XMLTemplate.", t);
		}
	}

	/**
	 * Merges a list of XML documents as specified in the mapping file. Mappings
	 * whose property elements are tagged as collection and merge are merged
	 * inside of their parent elements. A mapping in the collection's
	 * subtemplate can be tagged is id to specify which xpath provides the
	 * unique id to remove duuplicates. Similarly, a mapping can be tagged as
	 * sort to control how the elements are sorted in the merge. The first
	 * document in the list serves as the source document into which everything
	 * is merged.
	 *
	 * @param docs List of XML documents that will be merged. The list must not
	 *            be empty. A copy of the first document in the list serves as
	 *            the source document.
	 * @return The merging of all of the documents.
	 * @throws xmlutils.XMLUtilsException if docs is null or empty or some other
	 *             problem occurs during the merge process.
	 */
	public Document merge(List<Document> docs) throws XMLUtilsException
	{
		Document ret = null;
		XPathFactory xpathFactory = XPathFactory.newInstance();
		String propertyString = null;
		String xpathString = null;

		try
		{
			if(docs == null || docs.isEmpty())
			{
				if(isStrict())
				{
					throw new XMLUtilsException("docs cannot be empty.");
				}
				else
				{
					logger.error("docs cannot be empty.");
					return DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
				}
			}

			// the first document is the source and hence provides the starting
			// point for the merge document.
			ret = (Document) docs.get(0).cloneNode(true);

			// construct the XPath that will be used to find stuff in
			// our mapping file and pre-compile commonly used expressions.
			XPath xpathMap = xpathFactory.newXPath();
			XPathExpression exprProperty = xpathMap.compile("property");
			XPathExpression exprXPath = xpathMap.compile("xpath");
			XPathExpression exprXPathParent = xpathMap.compile("xpathParent");
			XPathExpression exprXPathEntry = xpathMap.compile("xpathEntry");
			XPathExpression exprXPathMergeCleanup = xpathMap.compile("xpathMergeCleanup");
			XPathExpression exprId = xpathMap.compile("mappings/mapping/xpath[@id='true']");
			XPathExpression exprSort = xpathMap.compile("mappings/mapping/xpath[@sort='true' or @sort='asc' or @sort='desc']");

			// construct XPath that will be used to find stuff in list of docs
			// and give it the proper namespace context.
			XPath xpathDoc = xpathFactory.newXPath();
			xpathDoc.setNamespaceContext(namespaceContext);

			// retrieve mappings in the mapping file whose property element is a
			// collection
			// and that should be considered in the merge.
			NodeList mappings = (NodeList) xpathMap.evaluate("mappings/mapping[property/@collection='true' and property/@merge='true']", map.getDocumentElement(), XPathConstants.NODESET);
			if(mappings != null)
			{
				for(int i = 0; i < mappings.getLength(); ++i)
				{
					propertyString = (String) exprProperty.evaluate(mappings.item(i), XPathConstants.STRING);
					xpathString = (String) exprXPath.evaluate(mappings.item(i), XPathConstants.STRING);
					String xpathParentString = (String) exprXPathParent.evaluate(mappings.item(i), XPathConstants.STRING);
					String xpathEntryString = (String) exprXPathEntry.evaluate(mappings.item(i), XPathConstants.STRING);
					NodeList xpathMergeCleanup = (NodeList) exprXPathMergeCleanup.evaluate(mappings.item(i), XPathConstants.NODESET);

					String xpathIdString = (String) exprId.evaluate(mappings.item(i), XPathConstants.STRING);
					Id id;
					if(!NullChecker.isNullOrEmpty(xpathIdString))
					{
						id = new NodeId(xpathIdString, xpathDoc.compile(xpathIdString));
					}
					else
					{
						id = new DefaultId();
					}

					Node xpathSort = (Node) exprSort.evaluate(mappings.item(i), XPathConstants.NODE);
					Sort sort;
					if(xpathSort != null)
					{
						String xpathSortString = xpathSort.getTextContent();
						String xpathSortOrder = xpathSort.getAttributes().getNamedItem("sort").getTextContent();
						sort = new NodeSort(xpathSortString, xpathDoc.compile(xpathSortString), xpathSortOrder);
					}
					else
					{
						sort = new DefaultSort();
					}

					// pre-compile the expressions.
					XPathExpression exprDocXPath = xpathDoc.compile(xpathString);
					XPathExpression exprDocXPathEntry = xpathDoc.compile(xpathEntryString);

					// Get the entity that contains the collection that will be
					// merged.
					Node collectionNode = (Node) exprDocXPath.evaluate(ret, XPathConstants.NODE);
					if(collectionNode == null)
					{
						Node parentNode = (Node) xpathDoc.evaluate(xpathParentString, ret, XPathConstants.NODE);
						if(parentNode == null)
						{
							XMLUtilsException xue = new XMLUtilsException("xpathParent did not resolve to a node in first document.", propertyString, xpathString);
							xue.setMappingXPathParent(xpathParentString);
							if(isStrict())
							{
								throw xue;
							}
							else
							{
								// CCR 177986
								logger.error("Errror {}:", xue);
								continue;
							}
						}

						for(Document doc : docs.subList(1, docs.size()))
						{
							collectionNode = (Node) exprDocXPath.evaluate(doc, XPathConstants.NODE);
							if(collectionNode != null)
							{
								collectionNode = ret.importNode(collectionNode, true);
								parentNode.appendChild(collectionNode);
								break;
							}
						}
					}

					if(collectionNode != null)
					{
						// remove any elements that match the mapping's
						// xpathEntry from the
						// merge document.
						Node entryParentNode = null;
						NodeList nodes = (NodeList) exprDocXPathEntry.evaluate(collectionNode, XPathConstants.NODESET);
						if(nodes != null)
						{
							for(int j = 0; j < nodes.getLength(); ++j)
							{
								entryParentNode = nodes.item(j).getParentNode();
								entryParentNode.removeChild(nodes.item(j));
							}
						}

						// if the mapping specifies an xpathMergeCleanup path,
						// find it
						// and clear its contents.
						if(xpathMergeCleanup != null)
						{
							for(int j = 0; j < xpathMergeCleanup.getLength(); ++j)
							{
								String xpathMergeCleanupString = xpathMergeCleanup.item(j).getTextContent();
								Node cleanupNode = (Node) xpathDoc.evaluate(xpathMergeCleanupString, collectionNode, XPathConstants.NODE);
								if(cleanupNode != null)
								{
									cleanupNode.setTextContent("");
								}
							}
						}

						// collect all of the nodes into a HashMap keyed by the
						// values
						// obtained from the id xpath. This will ensure that any
						// dupplicates are removed.
						TreeMap<NodeMapKey, Node> nodeMap = new TreeMap<NodeMapKey, Node>(new NodeMapKeyComparator(sort.getSortOrder()));
						for(int docNumber = 0; docNumber < docs.size(); ++docNumber)
						{
							Document doc = docs.get(docNumber);
							Node node = (Node) exprDocXPath.evaluate(doc, XPathConstants.NODE);
							if(node != null)
							{
								nodes = (NodeList) exprDocXPathEntry.evaluate(node, XPathConstants.NODESET);
								if(nodes != null)
								{
									for(int j = 0; j < nodes.getLength(); ++j)
									{
										String idValue = id.getId(nodes.item(j));
										if(idValue == null)
										{
											XMLUtilsException xue = new XMLUtilsException("XPath given for id did not resolve to a node in document " + docNumber + ".", propertyString, xpathString);
											xue.setMappingIdString(id.getIdXPath());
											if(isStrict())
											{
												throw xue;
											}
											else
											{
												logger.error("Errror {}:", xue);
												continue;
											}
										}

										String sortValue = sort.getSort(nodes.item(j));
										if(sortValue == null)
										{
											XMLUtilsException xue = new XMLUtilsException("XPath given for sort did not resolve to a node in document " + docNumber + ".", propertyString, xpathString);
											xue.setMappingSortString(sort.getSortXPath());
											if(isStrict())
											{
												throw xue;
											}
											else
											{
												logger.error("Errror {}:", xue);
												continue;
											}
										}

										NodeMapKey key = new NodeMapKey(idValue, sortValue);
										// check if item wtih id is already in
										// map so we keep first occurrence.
										if(!nodeMap.containsKey(key))
										{
											nodeMap.put(key, nodes.item(j));
										}
									}
								}
							}
						}

						// output the nodes to the merge document.
						for(Node n : nodeMap.values())
						{
							Node newNode = ret.importNode(n, true);
							if(entryParentNode != null)
							{
								entryParentNode.appendChild(newNode);
							}
						}
					}
					else
					{
						XMLUtilsException xue = new XMLUtilsException("xpath did not resolve to a node in any of the documents.", propertyString, xpathString);
						logger.debug(xue.toString());
					}
				}
			}
		}
		catch(XMLUtilsException xte)
		{
			throw xte;
		}
		catch(Throwable t)
		{
			throw new XMLUtilsException("An error occurred when merging.", t, propertyString, xpathString);
		}

		return ret;
	}
}
