package gov.va.med.nhin.adapter.datamanager.adapters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.parser.GenericParser;
import ca.uhn.hl7v2.util.Terser;
import ca.uhn.hl7v2.validation.impl.NoValidation;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ParseErrorException;
import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author David Vazquez
 */
public abstract class HL7DataAdapter implements DataAdapter
{
	private static final Logger logger = LoggerFactory.getLogger(HL7DataAdapter.class.getName());

	public List getData(DataQuery dataQuery)
	{
		// CCR 177986
		logger.debug("getData() invoked");

		try
		{
			String message = buildMessage(dataQuery);
			// CCR 177986
			logger.trace("Request:\n {} ", addLFs(message));
			String response = sendMessage(dataQuery, message);
			logger.trace("Response:\n {}", addLFs(response));
			return processResponse(dataQuery, response);
		}
		catch(DataManagerException dme)
		{
			throw dme;
		}
		// CCR 179231 - propagate up in the call stack
		catch(Exception e)
		{
			logger.error("error in getData() ", e);
			throw new DataManagerException("Error occurred in processing response ", e);
		}
	}

	String buildMessage(DataQuery dataQuery) throws Exception
	{
		String template = dataQuery.getProperty("template").trim();

		GenericParser parser = new GenericParser();
		parser.setValidationContext(new NoValidation());
		Message message = parser.parse(template);
		Terser terser = new Terser(message);

		for(String p : dataQuery.getParameterBySourceNames())
		{
			if(!NullChecker.isNullOrEmpty(dataQuery.getParameterBySource(p)))
			{
				terser.set(p, dataQuery.getParameterBySource(p).toString());
			}
		}

		return parser.encode(message);
	}

	List processResponse(DataQuery dataQuery, String response) throws Exception
	{
		List ret = new ArrayList<HashMap>();
		String rowCountLocation = dataQuery.getProperty("rowCountLocation");
		GenericParser parser = new GenericParser();
		Message message = parser.parse(response);
		Terser terser = new Terser(message);
		int rowCount;

		checkResponseStatus(dataQuery, terser);

		if(!NullChecker.isNullOrEmpty(rowCountLocation))
		{
			rowCount = Integer.parseInt(terser.get(rowCountLocation));
		}
		else
		{
			rowCount = 1;
		}

		for(int i = 0; i < rowCount; ++i)
		{
			HashMap result = new HashMap();
			for(String source : dataQuery.getResultBySourceNames())
			{
				String dataLocation = source.replaceAll("\\$index", Integer.toString(i));
				String value = terser.get(dataLocation);
				result.put(source, value);
			}
			ret.add(result);
		}

		return ret;
	}

	private void checkResponseStatus(DataQuery dataQuery, Terser terser) throws Exception
	{
		String statusLocation = dataQuery.getProperty("statusLocation");
		String statusAllowedValues = dataQuery.getProperty("statusAllowedValues");
		String statusErrorTextLocation = dataQuery.getProperty("statusErrorTextLocation");

		if(!NullChecker.isNullOrEmpty(statusLocation) && !NullChecker.isNullOrEmpty(statusAllowedValues))
		{
			String status = terser.get(statusLocation);
			boolean foundIt = false;
			for(String statusAllowedValue : statusAllowedValues.split(","))
			{
				if(status.equals(statusAllowedValue))
				{
					foundIt = true;
					break;
				}
			}
			if(!foundIt)
			{
				String message = status;
				if(!NullChecker.isNullOrEmpty(statusErrorTextLocation))
				{
					message += " - " + terser.get(statusErrorTextLocation);
				}
				throw new ParseErrorException(message);
			}
		}
	}

	abstract String sendMessage(DataQuery dataQuery, String message) throws Exception;

	private String addLFs(String str)
	{
		return str.replaceAll("\r", "\r\n");
	}
}
