package gov.va.med.nhin.adapter.datamanager.adapters;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPHeader;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.PortInfo;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import javax.xml.ws.soap.SOAPFaultException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.esr.webservices.jaxws.schemas.DemographicInfo;
import gov.va.med.esr.webservices.jaxws.schemas.EeSummary;
import gov.va.med.esr.webservices.jaxws.schemas.EeSummaryPort;
import gov.va.med.esr.webservices.jaxws.schemas.EeSummaryPortService;
import gov.va.med.esr.webservices.jaxws.schemas.GetEESummaryRequest;
import gov.va.med.esr.webservices.jaxws.schemas.GetEESummaryResponse;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;

/**
 *
 * @author VHAISBVAZQUD
 */
public class ESRDataAdapter implements DataAdapter
{
	private static Logger logger = LoggerFactory.getLogger(CompositeDataAdapter.class);

	private class HeaderHandlerResolver implements HandlerResolver
	{
		private String username;
		private String password;

		public HeaderHandlerResolver(String username, String password)
		{
			this.username = username;
			this.password = password;
		}

		public List<Handler> getHandlerChain(PortInfo portInfo)
		{
			List<Handler> ret = new ArrayList<Handler>();
			ret.add(new HeaderHandler(username, password));
			return ret;
		}
	}

	private class HeaderHandler implements SOAPHandler<SOAPMessageContext>
	{
		private String un;
		private String pw;

		public HeaderHandler(String username, String password)
		{
			this.un = username;
			this.pw = password;
		}

		public Set<QName> getHeaders()
		{
			return null;
		}

		public void close(MessageContext context)
		{
		}

		public boolean handleFault(SOAPMessageContext context)
		{
			return true;
		}

		public boolean handleMessage(SOAPMessageContext smc)
		{
			Boolean outboundProperty = (Boolean) smc.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);

			if(outboundProperty.booleanValue())
			{
				try
				{
					SOAPEnvelope envelope = smc.getMessage().getSOAPPart().getEnvelope();
					SOAPHeader header = envelope.getHeader();

					if(header == null)
					{
						header = envelope.addHeader();
					}

					SOAPElement security = header.addChildElement("Security", "wsse", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");

					SOAPElement usernameToken = security.addChildElement("UsernameToken", "wsse");
					usernameToken.addAttribute(new QName("xmlns:wsu"), "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");

					SOAPElement username = usernameToken.addChildElement("Username", "wsse");
					username.addTextNode(un);

					SOAPElement password = usernameToken.addChildElement("Password", "wsse");
					password.setAttribute("Type", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
					password.addTextNode(pw);
				}
				// CCR 179231 -Exception handling
				catch(Exception e)
				{
					// CCR 179231
					throw new DataAdapterException("Exception occurred while trying to process SOAP message.", e);
				}
			}

			return outboundProperty;
		}
	}

	private EeSummaryPortService eeSummaryService = null;

	public List getData(DataQuery dataQuery)
	{
		ArrayList ret = new ArrayList();
		String icn = (String) dataQuery.getParameter("icn");
		String connectionFilename = dataQuery.getProperty("connectionFilename");
		String connection = dataQuery.getProperty("connection");

		GetEESummaryRequest eeSummaryRequest = new GetEESummaryRequest();
		eeSummaryRequest.setKey(icn);
		eeSummaryRequest.setKeyType("VPID");
		eeSummaryRequest.setRequestName("preferredFacility");

		GetEESummaryResponse eeSummaryResponse = null;
		try
		{
			Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
			Properties connectionProperties = (Properties) propertiesCollection.get(connection);
			String wsdlURL = connectionProperties.getProperty("wsdlURL");
			String username = connectionProperties.getProperty("username");
			String password = connectionProperties.getProperty("password");

			eeSummaryResponse = getEESummaryPort(wsdlURL, username, password).getEESummary(eeSummaryRequest);
			if(eeSummaryResponse.getSummary() == null || eeSummaryResponse.getSummary().getDemographics() == null || NullChecker.isNullOrEmpty(eeSummaryResponse.getSummary().getDemographics().getPreferredFacility()))
			{
				eeSummaryResponse = createDefaultResponse("<UNKNOWN>", "<UNKNOWN>");
			}
		}
		catch(SOAPFaultException sfe)
		{
			if(sfe.getMessage().equals("PERSON_NOT_FOUND"))
			{
				logger.error("ESR does not know patient with ICN {} ", icn, sfe);
				eeSummaryResponse = createDefaultResponse("<UNKNOWN>", "<UNKNOWN>");
			}
			else
			{
				logger.error("There was an error getting data from ESR {} ", sfe);
				eeSummaryResponse = createDefaultResponse("<UNAVAILABLE>", "<UNAVAILABLE>");
			}
		}
		// CCR 177986- improved log related code
		catch(Exception e)
		{
			logger.error("There was an error getting data from ESR.  Message = {}", e);
			eeSummaryResponse = createDefaultResponse("<UNAVAILABLE>", "<UNAVAILABLE>");
		}

		ret.add(eeSummaryResponse);

		return ret;
	}

	private synchronized EeSummaryPort getEESummaryPort(String eeSummaryServiceWSDL, String username, String password)
	{
		if(eeSummaryService == null)
		{
			try
			{
				eeSummaryService = new EeSummaryPortService(new URL(eeSummaryServiceWSDL), new QName("http://jaxws.webservices.esr.domain.ext/schemas", "eeSummaryPortService"));

				if(!NullChecker.isNullOrEmpty(username) && !NullChecker.isNullOrEmpty(password))
				{
					eeSummaryService.setHandlerResolver(new HeaderHandlerResolver(username, password));
				}
			}
			catch(Throwable t)
			{
				throw new RuntimeException("Error getting EESummaryPort.", t);
			}
		}
		return eeSummaryService.getEeSummaryPortSoap11();
	}

	private GetEESummaryResponse createDefaultResponse(String facilityNumber, String facilityName)
	{
		GetEESummaryResponse eeSummaryResponse = new GetEESummaryResponse();

		EeSummary eeSummary = new EeSummary();
		eeSummaryResponse.setSummary(eeSummary);

		DemographicInfo demographicInfo = new DemographicInfo();
		eeSummary.setDemographics(demographicInfo);

		// CCR 177986-logging updates
		logger.debug("facilityNumber: {}", facilityNumber);
		logger.debug("facilityName: {}", facilityName);

		demographicInfo.setPreferredFacility(facilityNumber + " - " + facilityName);

		return eeSummaryResponse;
	}
}
