package gov.va.med.nhin.adapter.patientdiscovery;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.activation.DataHandler;
import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.jws.WebService;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.util.JAXBSource;
import javax.xml.ws.BindingType;

import org.apache.commons.io.IOUtils;
import org.hl7.v3.AddPatientCorrelationRequestType;
import org.hl7.v3.CE;
import org.hl7.v3.COCTMT090300UV01AssignedDevice;
import org.hl7.v3.CS;
import org.hl7.v3.CommunicationFunctionType;
import org.hl7.v3.EntityClassDevice;
import org.hl7.v3.II;
import org.hl7.v3.MCCIMT000100UV01Agent;
import org.hl7.v3.MCCIMT000100UV01Organization;
import org.hl7.v3.MCCIMT000300UV01Agent;
import org.hl7.v3.MCCIMT000300UV01Device;
import org.hl7.v3.MCCIMT000300UV01Organization;
import org.hl7.v3.MCCIMT000300UV01Receiver;
import org.hl7.v3.MCCIMT000300UV01Sender;
import org.hl7.v3.MFMIMT700711UV01AuthorOrPerformer;
import org.hl7.v3.MFMIMT700711UV01Custodian;
import org.hl7.v3.PRPAIN201301UV02;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01ControlActProcess;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject1;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject2;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectId;
import org.hl7.v3.PRPAMT201310UV02Patient;
import org.hl7.v3.PRPAMT201310UV02Person;
import org.hl7.v3.PRPAMT201310UV02QueryMatchObservation;
import org.hl7.v3.PRPAMT201310UV02Subject;
import org.hl7.v3.ParticipationTargetSubject;
import org.hl7.v3.RespondingGatewayPRPAIN201305UV02RequestType;
import org.hl7.v3.XParticipationAuthorPerformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.adapterpatientdiscovery.AdapterPatientDiscoveryPortType;
import gov.hhs.fha.nhinc.adapterpolicyengine.AdapterPolicyEnginePortType;
import gov.hhs.fha.nhinc.common.eventcommon.PatDiscReqEventType;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.CeType;
import gov.hhs.fha.nhinc.common.nhinccommon.HomeCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunitiesType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.common.nhinccommon.SamlAuthzDecisionStatementType;
import gov.hhs.fha.nhinc.common.nhinccommon.SamlIssuerType;
import gov.hhs.fha.nhinc.common.nhinccommon.UserType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.hhs.fha.nhinc.entitydocquery.EntityDocQueryPortType;
import gov.hhs.fha.nhinc.entitydocretrieve.EntityDocRetrievePortType;
import gov.hhs.fha.nhinc.nhinclib.NhincConstants;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.transform.policy.PolicyEngineTransformer;
import gov.hhs.fha.nhinc.transform.subdisc.HL7Constants;
import gov.va.med.nhin.adapter.adaptergateway.docquery.EntityDocQueryPortTypeLocal;
import gov.va.med.nhin.adapter.adaptergateway.docretrieve.EntityDocRetrievePortTypeLocal;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.documentrepository.Document;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepository;
import gov.va.med.nhin.adapter.documentrepository.DocumentRepositoryLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.OperationOnOff;
import gov.va.med.nhin.adapter.logging.AdapterPDError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.mpi.hl7parsers.HL7DbParser201306;
import gov.va.med.nhin.adapter.mvi.AdapterMviPortTypeLocal;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201305;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.mvi.hl7transforms.AdapterHL7PRPA201301Transforms;
import gov.va.med.nhin.adapter.patientcorrelation.PatientCorrelationPortTypeLocal;
import gov.va.med.nhin.adapter.policyengine.AdapterPolicyEnginePortTypeLocal;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.das.DasDAO;
import gov.va.med.nhin.adapter.utils.das.DasDAOHttpImpl;
import gov.va.med.nhin.adapter.utils.das.DasException;
import gov.va.med.nhin.adapter.utils.das.DasOperationSendACPDocumentRequest;
import gov.va.med.nhin.adapter.utils.das.DasOperationSendACPDocumentResponse;
import gov.va.nvap.privacy.ConsentDirectiveAuthorizationRequestType;
import gov.va.nvap.privacy.ConsentDirectiveAuthorizationResponseType;
import gov.va.nvap.privacy.ConsentDirectiveOptOutReasonType;
import gov.va.nvap.privacy.ConsentDirectiveQueryParamType;
import gov.va.nvap.privacy.ConsentDirectiveQueryRequestType;
import gov.va.nvap.privacy.ConsentDirectiveQueryResponseType;
import gov.va.nvap.privacy.ConsentDirectiveReferenceType;
import gov.va.nvap.privacy.ConsentDirectiveRevocationRequestType;
import gov.va.nvap.privacy.ConsentDirectiveRevocationResponseType;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.ServiceConsumer;
import gov.va.nvap.privacy.ServiceConsumerContextType;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType.DocumentRequest;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType.DocumentResponse;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryRequest;
import oasis.names.tc.ebxml_regrep.xsd.query._3.AdhocQueryResponse;
import oasis.names.tc.ebxml_regrep.xsd.query._3.ResponseOptionType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.AdhocQueryType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ExternalIdentifierType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ExtrinsicObjectType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.IdentifiableType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.RegistryObjectListType;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.SlotType1;
import oasis.names.tc.ebxml_regrep.xsd.rim._3.ValueListType;
import oasis.names.tc.xacml._2_0.context.schema.os.DecisionType;
import privacy.service.nvap.domain.ConsentManagementPortType;
import privacy.service.nvap.domain.ConsentManagementServiceFaultMessage;

/**
 *
 * @author VHAISBVAZQUD
 * @author Zack Peterson
 */
@WebService(serviceName = "AdapterPatientDiscovery", wsdlLocation = "META-INF/wsdl/AdapterPatientDiscovery.wsdl", portName = "AdapterPatientDiscoveryPortSoap", endpointInterface = "gov.hhs.fha.nhinc.adapterpatientdiscovery.AdapterPatientDiscoveryPortType", targetNamespace = "urn:gov:hhs:fha:nhinc:adapterpatientdiscovery")
@BindingType(value = javax.xml.ws.soap.SOAPBinding.SOAP12HTTP_BINDING)
@Stateless
public class AdapterPatientDiscovery implements AdapterPatientDiscoveryPortType
{
	private static final Logger logger = LoggerFactory.getLogger(AdapterPatientDiscovery.class.getName());

	private static final long MILLIS_IN_YEAR = 1000L * 60 * 60 * 24 * 365;

	private AdapterMviPortTypeLocal adapterMvi;
	private PatientCorrelationPortTypeLocal adapterPatientCorrelation;
	private AdapterPolicyEnginePortType adapterPolicyEngine;
	private ConsentManagementPortType vapConsentManagement;
	private FacilityManager facilityManager;
	private PropertyLookup propertyLookup;

	private EntityDocQueryPortType entityDocQuery;
	private EntityDocRetrievePortType entityDocRetrieve;

	// audit logging
	private AuditManager auditManager;
	private DocumentRepository documentRepository;
        
        private static JAXBContext context;
        static
        {
            try
            {
                context = JAXBContext.newInstance(PRPAIN201306UV02.class);
            }
            catch (JAXBException e)
            {
                logger.debug("JAXBException occurred at initialization: ", e);
            }
        }
        
	@EJB(beanInterface = AdapterMviPortTypeLocal.class, beanName = "AdapterMVI")
	public void setAdapterMvi(AdapterMviPortTypeLocal adapterMvi)
	{
		this.adapterMvi = adapterMvi;
	}

	@EJB(beanInterface = PatientCorrelationPortTypeLocal.class, beanName = "AdapterPatientCorrelation")
	public void setAdapterPatientCorrelation(PatientCorrelationPortTypeLocal adapterPatientCorrelation)
	{
		this.adapterPatientCorrelation = adapterPatientCorrelation;
	}

	@EJB(beanInterface = AdapterPolicyEnginePortTypeLocal.class, beanName = "AdapterPolicyEngine")
	public void setAdapterPolicyEngine(AdapterPolicyEnginePortType adapterPolicyEngine)
	{
		this.adapterPolicyEngine = adapterPolicyEngine;
	}

	@EJB(beanInterface = VapConsentManagementPortTypeLocal.class, beanName = "VapConsentManagement")
	public void setVapConsentManagement(ConsentManagementPortType vapConsentManagement)
	{
		this.vapConsentManagement = vapConsentManagement;
	}

	@EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
	public void setFacilityManager(FacilityManager facilityManager)
	{
		this.facilityManager = facilityManager;
	}

	@EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyLookup")
	public void setPropertyLookup(PropertyLookup propertyLookup)
	{
		this.propertyLookup = propertyLookup;
	}

	@EJB(beanInterface = EntityDocQueryPortTypeLocal.class, beanName = "EntityDocQueryOrch")
	public void setEntityDocQuery(EntityDocQueryPortType entityDocQuery)
	{
		this.entityDocQuery = entityDocQuery;
	}

	@EJB(beanInterface = EntityDocRetrievePortTypeLocal.class, beanName = "AdapterGatewayDocRetrieve")
	public void setEntityDocRetrieve(EntityDocRetrievePortType entityDocRetrieve)
	{
		this.entityDocRetrieve = entityDocRetrieve;
	}

	@EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
	public void setAuditManager(AuditManager auditManager)
	{
		this.auditManager = auditManager;
	}

	@EJB(beanInterface = DocumentRepositoryLocal.class, beanName = "DocumentRepository")
	public void setDocumentRepository(DocumentRepository documentRepository)
	{
		this.documentRepository = documentRepository;
	}

	@Override
	public PRPAIN201306UV02 respondingGatewayPRPAIN201305UV02(RespondingGatewayPRPAIN201305UV02RequestType request)
	{
		String partnerCommunityId = request.getAssertion().getHomeCommunity().getHomeCommunityId();
		logger.debug("Starting  Patient Discovery and Home Community ID is {} ", partnerCommunityId);
		Map<Object, Object> searchResultsMap = null;
		PRPAIN201306UV02 ret;

		try
		{
			/*
			 * NOTE: the MVI RPC code (pre MVI web service migration - SR345)
			 * would flip the sender/receiver it got in the request when setting
			 * the 1306 response. Also noted I believe the request -> receiver
			 * OID sent by partner to adapter is always the VA's OID (i.e.
			 * getHomeCommunityId()) but decided to keep as close to original
			 * code.
			 */
			String sendingHCID = getSenderOID(request.getPRPAIN201305UV02());
			String sendingFacilityNumber = getFacilityNumber(sendingHCID);
			String vaHCID = getHomeCommunityId();
			String roid = sendingHCID;
			String soid = getReceiverOID(request.getPRPAIN201305UV02());

			PRPAIN201306UV02 mviResults = null;
			if(facilityManager.isPartnerAllowed(partnerCommunityId, OperationOnOff.ONBOARD))
			{
				if(facilityManager.isPartnerAllowed(partnerCommunityId, OperationOnOff.IN_PD))
				{
					if(checkPolicyNHINIn(request) && (searchResultsMap = findCandidates(request)) != null && (mviResults = (PRPAIN201306UV02) searchResultsMap.get("PRPAIN201306UV02")) != null && isMatch(request, mviResults) && checkPolicyPatientDiscoveryIn(roid, soid, request, mviResults))
					{
						// Take MVI 1306 response and use to copy/update for the
						// Adapter's 1306 response.
						ret = processMVIResults(mviResults, vaHCID, sendingHCID);

						II remotePatientIdRequest = extractPatientIdFrom201305(request.getPRPAIN201305UV02());

						if(remotePatientIdRequest == null)
						{
							logger.warn("No Remote Patient ID was sent in PD request so not calling addPatientCorrelation.");
						}
						else
						{
							boolean correlationAlreadyExists = HL7Parser201306.extractMatchingId(mviResults, remotePatientIdRequest.getExtension(), "NI", sendingFacilityNumber, remotePatientIdRequest.getRoot()) != null;

							if(correlationAlreadyExists)
							{
								logger.debug("Remote Patient ID: ''{}'' " + "is already correlated in MVI for HCID: ''{}''.", remotePatientIdRequest, sendingHCID);
							}
							// even if the remote Patient ID is already
							// correlated we will call addPatientCorrelation to
							// enter Audit log entry.
							Patient searchResultPatient = (Patient) searchResultsMap.get("SEARCH_RESULT_PATIENT");

							// Indicating the correlation already exists will
							// skip adding the correlation for SSA
                            if (facilityManager.getFacilityByFacilityNumber("200NSS") == null || request.getAssertion().getHomeCommunity().getHomeCommunityId().contains(facilityManager.getFacilityByFacilityNumber("200NSS").getHomeCommunityId()))
							{
								correlationAlreadyExists = false;
							}

							addPatientCorrelation(request, mviResults, searchResultPatient, correlationAlreadyExists);
						}
					}
					else
					{
						if(!checkPolicyNHINIn(request))
						{
							logger.debug("PD Failed checkPolicy");
							AdapterPDError.queryError(request, ErrorMessage.IN_PD_FAILED_POLICY_CHECK);
						}
						else if(findCandidates(request) == null)
						{
							logger.debug("PD Failed to find candidates.");
							AdapterPDError.queryError(request, ErrorMessage.IN_PD_NO_MATCH);
						}
						else if(searchResultsMap != null && searchResultsMap.get("PRPAIN201306UV02") == null)
						{
							logger.debug("PD Failed to find 1306 response.");
							AdapterPDError.queryError(request, ErrorMessage.IN_PD_NO_RESPONSE);
						}
						else if(!isMatch(request, mviResults))
						{
							logger.debug("PD Failed to find match.");
							AdapterPDError.queryError(request, ErrorMessage.IN_PD_NO_MATCH);
						}
						else if(!checkPolicyPatientDiscoveryIn(roid, soid, request, mviResults))
						{
							logger.debug("PD Failed check policy PD");
							AdapterPDError.queryError(request, ErrorMessage.IN_PD_FAILED_POLICY_CHECK);
						}
						ret = createPatientNotFoundResponse(request, vaHCID, getAssigningAuthorityId());
					}
				}
				else
				{
					AdapterPDError.queryError(request, ErrorMessage.IN_PD_DISABLED);
					ret = createErrorResponse(request, vaHCID, getAssigningAuthorityId());
				}
			}
			else
			{
				AdapterPDError.queryError(request, ErrorMessage.IN_PD_NOT_A_PARTNER);
				ret = createErrorResponse(request, vaHCID, getAssigningAuthorityId());
			}
		}
		catch(Throwable t)
		{
			String hcid;
			String hcidName;
			if(request.getAssertion() != null && request.getAssertion().getHomeCommunity() != null)
			{
				hcid = request.getAssertion().getHomeCommunity().getHomeCommunityId();
				hcidName = request.getAssertion().getHomeCommunity().getName();
			}
			else
			{
				hcid = "N/A";
				hcidName = "Not Available";
			}
			
			logger.error("Error processing PatientDiscovery from {} {} - {}", hcid, hcidName, t.getMessage());
	        
			ret = createErrorResponse(request, getHomeCommunityId(), getAssigningAuthorityId());
			
			AdapterPDError.queryError(request, ErrorMessage.IN_PD_UNKNOWN);
		}

		logger.debug("respondingGatewayPRPAIN201305UV02() exited");

		return ret;
	}

	/**
	 * Takes the MVI 1305 response/results and make copy and update as needed
	 * for use as the Adapter's PD response to Partner.
	 *
	 * @param mviResult
	 * @return
	 */
	private PRPAIN201306UV02 processMVIResults(PRPAIN201306UV02 mvi1306Result, String vaHomeFacilityID, String sendingFacilityOID)
	{
		PRPAIN201306UV02 adapter1306Response = null;
		try
		{
			adapter1306Response = (PRPAIN201306UV02) context.createUnmarshaller().unmarshal(new JAXBSource(context, mvi1306Result));
		}
		catch(JAXBException e)
		{
			logger.error("JAXBException occurred while making copy of MVI PRPAIN201306UV02 response message.", e);
            throw new RuntimeException(e.getMessage());
		}

		setSenderId(adapter1306Response, vaHomeFacilityID);
		setReceiver(adapter1306Response, sendingFacilityOID);
		setSubjectPatientId(adapter1306Response, vaHomeFacilityID);

		setRequiredElements(adapter1306Response, vaHomeFacilityID);
		return adapter1306Response;
	}

	/**
	 * Takes the MVI 1305 response/results and adds missing required elements in
	 * the response
	 *
	 * @param mviResult
	 * @return
	 */
	private void setRequiredElements(PRPAIN201306UV02 mvi1306Result, String vaHomeFacilityID)
	{
		PRPAIN201306UV02MFMIMT700711UV01ControlActProcess controlAckProcess = mvi1306Result.getControlActProcess();

		if(controlAckProcess == null)
		{
			return;
		}
		// create author or performer
		if(controlAckProcess.getAuthorOrPerformer() == null || controlAckProcess.getAuthorOrPerformer().size() < 1)
		{
			controlAckProcess.getAuthorOrPerformer().add(createAuthorOrPerformer(vaHomeFacilityID));
		}

		List<PRPAIN201306UV02MFMIMT700711UV01Subject1> subjectList = controlAckProcess.getSubject();
		for(PRPAIN201306UV02MFMIMT700711UV01Subject1 subject : subjectList)
		{
			PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent regEvent = subject.getRegistrationEvent();
			if(regEvent != null)
			{
				PRPAIN201306UV02MFMIMT700711UV01Subject2 subject1 = regEvent.getSubject1();
				PRPAMT201310UV02Patient patient = subject1.getPatient();
				if(patient != null)
				{
					JAXBElement<PRPAMT201310UV02Person> patPerson = patient.getPatientPerson();
					// add determiner code to patient person
					if(patPerson != null && patPerson.getValue() != null)
					{
						patPerson.getValue().setDeterminerCode("INSTANCE");

						if(patPerson.getValue().getClassCode().size() > 0)
						{
							patPerson.getValue().getClassCode().clear();
						}

						patPerson.getValue().getClassCode().add("PSN");
					}

					List<PRPAMT201310UV02Subject> subjectOf1List = patient.getSubjectOf1();
					for(PRPAMT201310UV02Subject subjectOf1 : subjectOf1List)
					{
						// set type code on subject of 1
						subjectOf1.setTypeCode(ParticipationTargetSubject.SBJ);

						// set class code on query match observation
						PRPAMT201310UV02QueryMatchObservation queryMatchObservation = subjectOf1.getQueryMatchObservation();

						if(queryMatchObservation != null)
						{
							if(queryMatchObservation.getClassCode().size() > 0)
							{
								queryMatchObservation.getClassCode().clear();
							}
							queryMatchObservation.getClassCode().add("OBS");
						}
					}
				}
				// set Code on assigned Entity
				MFMIMT700711UV01Custodian custodian = regEvent.getCustodian();
				if(custodian != null && custodian.getAssignedEntity() != null)
				{
					CE ce = new CE();
					ce.setCode("NotHealthDataLocator");
					ce.setCodeSystem("1.3.6.1.4.1.19376.1.2.27.2");
					custodian.getAssignedEntity().setCode(ce);
				}
			}
		}
	}

	/**
	 * Takes the assigning authority and created author or performer object on
	 * the xml
	 *
	 * @param assigningAuthority
	 * @return
	 */
	private MFMIMT700711UV01AuthorOrPerformer createAuthorOrPerformer(String assigningAuthority)
	{
		MFMIMT700711UV01AuthorOrPerformer authorOrPerformer = new MFMIMT700711UV01AuthorOrPerformer();
		authorOrPerformer.setTypeCode(XParticipationAuthorPerformer.AUT);

		COCTMT090300UV01AssignedDevice assignedDevice = new COCTMT090300UV01AssignedDevice();
		II id = new II();
		id.setRoot(assigningAuthority);
		assignedDevice.setClassCode(HL7Constants.ASSIGNED_DEVICE_CLASS_CODE);
		assignedDevice.getId().add(id);

		javax.xml.namespace.QName xmlqname = new javax.xml.namespace.QName("urn:hl7-org:v3", "assignedDevice");
		JAXBElement<COCTMT090300UV01AssignedDevice> assignedDeviceJAXBElement = new JAXBElement<>(xmlqname, COCTMT090300UV01AssignedDevice.class, assignedDevice);

		authorOrPerformer.setAssignedDevice(assignedDeviceJAXBElement);

		return authorOrPerformer;
	}

	/**
	 * Grab the ICN and then clear out all the other correlated IDs since the
	 * external partner only cares about the ICN. Strip the ICN down to just the
	 * value (e.g. remove ID type of "NI", etc.).
	 *
	 * @param adapter1306Response
	 * @param vaHomeFacility
	 */
	private void setSubjectPatientId(PRPAIN201306UV02 adapter1306Response, String vaHomeFacilityID)
	{

		PRPAMT201310UV02Patient patient = HL7Parser201306.extractSubjectPatient(adapter1306Response);
		if(patient != null && patient.getId() != null && patient.getId().size() > 0)
		{
			II newICN = new II();
			II icn = HL7Parser201306.extractICNId(patient.getId());
			newICN.setExtension(HL7Parser201306.extractICNValue(icn.getExtension()));
			newICN.setRoot(vaHomeFacilityID);

			patient.getId().clear();
			patient.getId().add(newICN);
		}
	}

	private void setReceiver(PRPAIN201306UV02 adapter1306Response, String sendingFacilityOID)
	{
		if(adapter1306Response.getReceiver() != null && adapter1306Response.getReceiver().size() > 0)
		{
			adapter1306Response.getReceiver().clear();
		}

		// Set the receiver
		MCCIMT000300UV01Receiver receiver = new MCCIMT000300UV01Receiver();
		receiver.setTypeCode(CommunicationFunctionType.RCV);

		MCCIMT000300UV01Device receiverDevice = new MCCIMT000300UV01Device();
		receiverDevice.setDeterminerCode("INSTANCE"); // HL7Constants.RECEIVER_DETERMINER_CODE
		receiverDevice.setClassCode(EntityClassDevice.DEV);

		II receiverId = new II();
		receiverId.setRoot(sendingFacilityOID);

		receiverDevice.getId().add(receiverId);
		receiver.setDevice(receiverDevice);
		adapter1306Response.getReceiver().add(receiver);

		// adapter1306Response.getReceiver().get(0).getDevice().getAsAgent().getValue().
		// Setup Receiver -> Device -> Agent
		MCCIMT000300UV01Agent receiverDeviceAgent = new MCCIMT000300UV01Agent();
		receiverDeviceAgent.getClassCode().add("AGNT");
		MCCIMT000300UV01Organization representedOrg = new MCCIMT000300UV01Organization();
		representedOrg.setClassCode("ORG");
		representedOrg.setDeterminerCode("INSTANCE");

		II representedOrgId = new II();
		representedOrgId.setRoot(sendingFacilityOID);
		// representedOrgId.setExtension(facilityNumber);
		representedOrg.getId().add(representedOrgId);

		org.hl7.v3.ObjectFactory objFactory = new org.hl7.v3.ObjectFactory();
		JAXBElement<MCCIMT000300UV01Agent> jaxbElementAgent = objFactory.createMCCIMT000300UV01DeviceAsAgent(receiverDeviceAgent);
		JAXBElement<MCCIMT000300UV01Organization> jaxbElementRepresentedOrg = objFactory.createMCCIMT000300UV01AgentRepresentedOrganization(representedOrg);

		receiverDeviceAgent.setRepresentedOrganization(jaxbElementRepresentedOrg);
		receiverDevice.setAsAgent(jaxbElementAgent);
	}

	private void setSenderId(PRPAIN201306UV02 adapter1306Response, String vaHomeFacilityID)
	{
		if(adapter1306Response != null && adapter1306Response.getSender() != null && adapter1306Response.getSender().getDevice() != null && adapter1306Response.getSender().getDevice().getId() != null && adapter1306Response.getSender().getDevice().getId().size() > 0 && adapter1306Response.getSender().getDevice().getId().get(0) != null)
		{
			MCCIMT000300UV01Sender sender = adapter1306Response.getSender();
			MCCIMT000300UV01Device senderDevice = sender.getDevice();
			II senderDeviceId = senderDevice.getId().get(0);
			senderDeviceId.setRoot(vaHomeFacilityID);

			// Setup Sender -> Device -> Agent
			MCCIMT000100UV01Agent senderDeviceAgent = new MCCIMT000100UV01Agent();
			senderDeviceAgent.getClassCode().add("AGNT");
			MCCIMT000100UV01Organization representedOrg = new MCCIMT000100UV01Organization();
			representedOrg.setClassCode("ORG");
			representedOrg.setDeterminerCode("INSTANCE");

			II representedOrgId = new II();
			representedOrgId.setRoot(vaHomeFacilityID);
			// representedOrgId.setExtension(facilityNumber);
			representedOrg.getId().add(representedOrgId);

			org.hl7.v3.ObjectFactory objFactory = new org.hl7.v3.ObjectFactory();
			JAXBElement jaxbElementAgent = objFactory.createMCCIMT000100UV01DeviceAsAgent(senderDeviceAgent);
			JAXBElement<MCCIMT000100UV01Organization> jaxbElementRepresentedOrg = objFactory.createMCCIMT000100UV01AgentRepresentedOrganization(representedOrg);

			senderDeviceAgent.setRepresentedOrganization(jaxbElementRepresentedOrg);
			senderDevice.setAsAgent(jaxbElementAgent);
			sender.setDevice(senderDevice);
		}
	}

	/**
	 * TODO
	 * <ol>
	 * <li>The sender OID exists.</li>
	 * <li>The sender OID is not the same as the HomeCommunityId.</li>
	 * <li>The sender OID is known to the facility manager.</li>
	 * </ol>
	 * 
	 * @param request The web service request.
	 * @return true if TODO
	 */
	private boolean checkPolicyNHINIn(RespondingGatewayPRPAIN201305UV02RequestType request)
	{
		String senderOID = getSenderOID(request.getPRPAIN201305UV02());

		//@formatter:off
		return !NullChecker.isNullOrEmpty(senderOID) && 
				!senderOID.equals(propertyLookup.getProperty("HomeCommunityId")) &&
				facilityManager.getFacilityByHomeCommunityId(senderOID) != null;
		//@formatter:on
	}

	/**
	 * TODO: What are candidates?
	 * 
	 * @param request The web service request.
	 * @return a map containing candidates. TODO: What are keys and values?
	 */
	@SuppressWarnings("unchecked")
	private Map<Object, Object> findCandidates(RespondingGatewayPRPAIN201305UV02RequestType request)
	{
		return adapterMvi.findCandidatesMap(request);
	}

	/**
	 * There is a match if the response contains a value for the
	 * ControlActProcess's subject. TODO: what is a ControlActProcess?
	 * 
	 * @param request The web service request.
	 * @param response TODO: this is extracted from the findCandidates method.
	 * @return true if there is match, false otherwise.
	 */
	private boolean isMatch(RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response)
	{
		return response != null && response.getControlActProcess() != null && !NullChecker.isNullOrEmpty(response.getControlActProcess().getSubject());
	}

	/**
	 * 
	 * @param roid
	 * @param soid
	 * @param request
	 * @param response
	 * @return
	 */
	private boolean checkPolicyPatientDiscoveryIn(String roid, String soid, RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response)
	{
		PatDiscReqEventType policyCheckReq = new PatDiscReqEventType();
		policyCheckReq.setDirection(NhincConstants.POLICYENGINE_INBOUND_DIRECTION);

		policyCheckReq.setPRPAIN201306UV02(response);
		policyCheckReq.setAssertion(request.getAssertion());
		HomeCommunityType senderHC = new HomeCommunityType();
		senderHC.setHomeCommunityId(roid);
		policyCheckReq.setSendingHomeCommunity(senderHC);
		HomeCommunityType receiverHC = new HomeCommunityType();
		receiverHC.setHomeCommunityId(soid);
		policyCheckReq.setReceivingHomeCommunity(receiverHC);

		PolicyEngineTransformer policyEngineTransformer = new PolicyEngineTransformer();
		CheckPolicyRequestType checkPolicyRequest = policyEngineTransformer.transformPatDiscReqToCheckPolicy(policyCheckReq);

		logger.info("About to enter checkACPDocument...");
		// Do ACP document query/retrieve
		boolean isPermitted = checkACPDocument(request, response, senderHC);

		// If valid eAuth cannot be obtained, call VAP for check policy
		if(!isPermitted)
		{
			CheckPolicyResponseType checkPolicyResponse = adapterPolicyEngine.checkPolicy(checkPolicyRequest);
			isPermitted = (checkPolicyResponse.getResponse().getResult().get(0).getDecision() == DecisionType.PERMIT);
		}

		return isPermitted;
	}

	/**
	 * 
	 * @param request
	 * @param response
	 * @param senderHC
	 * @return
	 */
	private boolean checkACPDocument(RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response, HomeCommunityType senderHC)
	{
		AssertionType assertion = request.getAssertion();

		logger.info("Checking if UserType Exists...");
		UserType userType = assertion.getUserInfo();
		if(NullChecker.isNullOrEmpty(userType))
		{
			logger.debug("User Info is null, returning FALSE");
			return false;
		}
		
		String ssaOID = assertion.getHomeCommunity().getHomeCommunityId();
		if(NullChecker.isNullOrEmpty(ssaOID))
		{
			logger.debug("assertion User Organization is null, returning FALSE");
			return false;
		}

		if(facilityManager.getFacilityByFacilityNumber("200NSS") != null) {
    		if(!ssaOID.contains(facilityManager.getFacilityByFacilityNumber("200NSS").getFullHomeCommunityId()))
    		{
                logger.info("SSA ID in DB is: {0}", facilityManager.getFacilityByFacilityNumber("200NSS").getHomeCommunityId());
    			// not SSA, don't do ACP Check
    			return false;
    		}
            else {
                logger.info("SSA facility not found in the database.");
                // SSA not in DB, don't do ACP Check
                return false;
            }
		}

		String excludeSectionsString = propertyLookup.getProperty("excludePatientDiscoverySections");
		if(NullChecker.isNullOrEmpty(excludeSectionsString))
		{
			excludeSectionsString = "";
		}
		
		RetrieveDocumentSetResponseType accessConsentRetrieveResponse = null;
		byte[] acpDocument = null;
		if(!excludeSectionsString.contains("1"))
		{
			/*
			 * Check request Assertion section and send Access Consent Policy
			 * (ACP) Query Document (QD)/Retrieve Document (RD) to the
			 * Originator of PD request.
			 */
			SamlAuthzDecisionStatementType authZDecisionStatement = assertion.getSamlAuthzDecisionStatement();
			logger.info("Got authZDecisionStatement from assertion.");

			if(NullChecker.isNullOrEmpty(authZDecisionStatement))
			{
				// SAML section is empty, don't do Access Consent doc query
				return false;
			}
			
			// eAuth requests will have ACP while wet sig will not
			if(NullChecker.isNullOrEmpty(authZDecisionStatement.getEvidence().getAssertion().getAccessConsentPolicy()))
			{
				// wet signature, don't do Access Consent doc query
				return false;
			}

			// get values required for ACP doc query
			if(NullChecker.isNullOrEmpty(authZDecisionStatement.getEvidence().getAssertion().getInstanceAccessConsentPolicy()))
			{
				return false;
			}

			String samlInstanceAccessPolicy = authZDecisionStatement.getEvidence().getAssertion().getInstanceAccessConsentPolicy().get(0);

			if(NullChecker.isNullOrEmpty(assertion.getUniquePatientId()))
			{
				return false;
			}

			String samlPatientId = assertion.getUniquePatientId().get(0);
			logger.debug("Starting ACP DQ for Saml Patient ID {}= " , samlPatientId);

			// do EntityDocQuery to get Access Consent doc Id from originator
			gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayQueryRequestType accessConsentQueryRequest;
			AdhocQueryResponse accessConsentQueryResponse;

			try
			{
				gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory objFactory = new gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory();
				accessConsentQueryRequest = objFactory.createRespondingGatewayCrossGatewayQueryRequestType();

				// build Access Consent Doc Query request
				AdhocQueryRequest aqr = new AdhocQueryRequest();
				aqr.setFederated(false);
				aqr.setStartIndex(BigInteger.valueOf(0));
				aqr.setMaxResults(BigInteger.valueOf(-1));

				// AdhocQuery
				AdhocQueryType aqt = new AdhocQueryType();
				aqt.setHome("urn:oid:" + propertyLookup.getProperty("HomeCommunityId"));
				aqt.setId("urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d");

				// add slots
				SlotType1 slot;
				ValueListType valueList;

				// $XDSDocumentEntryPatientId
				slot = new SlotType1();
				slot.setName("$XDSDocumentEntryPatientId");
				valueList = new ValueListType();
				valueList.getValue().add(samlPatientId);
				slot.setValueList(valueList);
				aqt.getSlot().add(slot);

				// $XDSDocumentEntryStatus
				slot = new SlotType1();
				slot.setName("$XDSDocumentEntryStatus");
				valueList = new ValueListType();
				valueList.getValue().add("urn:oasis:names:tc:ebxmlregrep:StatusType:Approved");
				slot.setValueList(valueList);
				aqt.getSlot().add(slot);

				// $XDSDocumentEntryClassCode
				slot = new SlotType1();
				slot.setName("$XDSDocumentEntryClassCode");
				valueList = new ValueListType();
				valueList.getValue().add("57016-8");
				slot.setValueList(valueList);
				aqt.getSlot().add(slot);

				// $XDSDocumentEntryEventCodeList
				slot = new SlotType1();
				slot.setName("$XDSDocumentEntryEventCodeList");
				valueList = new ValueListType();
				valueList.getValue().add(samlInstanceAccessPolicy);
				slot.setValueList(valueList);
				aqt.getSlot().add(slot);

				aqr.setAdhocQuery(aqt);

				// Response Options
				ResponseOptionType rot = new ResponseOptionType();
				rot.setReturnComposedObjects(true);
				rot.setReturnType("LeafClass");

				aqr.setResponseOption(rot);

				accessConsentQueryRequest.setAdhocQueryRequest(aqr);

				// assertion
				AssertionType asrt = createAssertion(assertion, samlPatientId);

				accessConsentQueryRequest.setAssertion(asrt);

				// target communities
				NhinTargetCommunitiesType targetCommunities = new NhinTargetCommunitiesType();
				NhinTargetCommunityType targetCommunity = new NhinTargetCommunityType();
				targetCommunity.setHomeCommunity(senderHC);
				targetCommunities.getNhinTargetCommunity().add(targetCommunity);
				accessConsentQueryRequest.setNhinTargetCommunities(targetCommunities);

				// call EntityDocQuery
				accessConsentQueryResponse = entityDocQuery.respondingGatewayCrossGatewayQuery(accessConsentQueryRequest);
			}
			catch(Exception e)
			{
				logger.error(getClass().getName(), "respondingGatewayCrossGatewayQuery", e);
	            
				return false;
			}

			if(!accessConsentQueryResponse.getStatus().equalsIgnoreCase("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success"))
			{
				logger.warn("SAML Patient Id = {}, ACP document query status = {} ", samlPatientId, accessConsentQueryResponse.getStatus());
	            
				return false;
			}

			// parse doc query response and get Document Unique Id and Repo
			// Unique Id
			String accessConsentDocumentUniqueId = null;
			String accessConsentRepositoryUniqueId = null;
			String accessConsentDocumentCreationTime = null;

			ExtrinsicObjectType extrinsicObject;
			List<SlotType1> extrinsicObjectSlotList;
			List<ExternalIdentifierType> extrinsicObjectExternalIdentifierList;

			RegistryObjectListType registryObjectList = accessConsentQueryResponse.getRegistryObjectList();

			List<JAXBElement<? extends IdentifiableType>> identifiableList;
			if(registryObjectList != null)
			{
				identifiableList = registryObjectList.getIdentifiable();
			}
			else
			{
				logger.debug("DQ response registryObjectList is null. Returning to regular PD...");
				return false;
			}

			if(identifiableList == null)
			{
				logger.debug("DQ response identifiableList is null. Returning to regular PD...");
				return false;
			}

			for(JAXBElement<? extends IdentifiableType> identifiable : identifiableList)
			{
				if(identifiable.getValue() instanceof ExtrinsicObjectType)
				{
					extrinsicObject = (ExtrinsicObjectType) identifiable.getValue();

					// get Slots
					extrinsicObjectSlotList = extrinsicObject.getSlot();
					for(SlotType1 extrinsicObjectSlot : extrinsicObjectSlotList)
					{
						// search for slot.name = 'repositoryUniqueId'
						if(extrinsicObjectSlot.getName().equalsIgnoreCase("repositoryUniqueId"))
						{
							accessConsentRepositoryUniqueId = extrinsicObjectSlot.getValueList().getValue().get(0);
						}
						// search for slot.name = 'creationTime'
						if(extrinsicObjectSlot.getName().equalsIgnoreCase("creationTime"))
						{
							accessConsentDocumentCreationTime = extrinsicObjectSlot.getValueList().getValue().get(0);
						}
					}

					// get ExternalIdentifiers
					extrinsicObjectExternalIdentifierList = extrinsicObject.getExternalIdentifier();
					for(ExternalIdentifierType extrinsicObjectExternalIdentifier : extrinsicObjectExternalIdentifierList)
					{
						if(extrinsicObjectExternalIdentifier.getName().getLocalizedString().get(0).getValue().equalsIgnoreCase("XDSDocumentEntry.uniqueId"))
						{
							accessConsentDocumentUniqueId = extrinsicObjectExternalIdentifier.getValue();
						}
					}
				}
			}

			logger.debug("accessConsentDocumentUniqueId = {} ", accessConsentDocumentUniqueId);
	        logger.debug("accessConsentRepositoryUniqueId = {} ", accessConsentRepositoryUniqueId);
	        logger.debug("accessConsentDocumentCreationTime = {} ", accessConsentDocumentCreationTime);

			if(NullChecker.isNullOrEmpty(accessConsentDocumentUniqueId))
			{
				logger.warn("SAML Patient Id ={} ACP Document Unique Id is not provided", samlPatientId);
	            return false;
			}

			if(NullChecker.isNullOrEmpty(accessConsentRepositoryUniqueId))
			{
				logger.warn("SAML Patient Id={} ACP Repository Unique Id is not provided", samlPatientId);
	            return false;
			}

			if(NullChecker.isNullOrEmpty(accessConsentDocumentCreationTime))
			{
				logger.warn("SAML Patient Id ={} for whcih ACP Creation Time is not provided", samlPatientId);
	            return false;
			}

			logger.debug("Starting ACP DR for Saml Patient ID {}= ", samlPatientId);

			// do EntityDocRetrieve to get Access Consent doc from originator
			gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayRetrieveRequestType accessConsentRetrieveRequest;

			try
			{
				gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory objFactory = new gov.hhs.fha.nhinc.common.nhinccommonentity.ObjectFactory();
				accessConsentRetrieveRequest = objFactory.createRespondingGatewayCrossGatewayRetrieveRequestType();

				// build Access Consent Doc Retrive request
				RetrieveDocumentSetRequestType rdsrt = new RetrieveDocumentSetRequestType();

				// Document Request
				DocumentRequest dr = new RetrieveDocumentSetRequestType.DocumentRequest();

				dr.setHomeCommunityId(ssaOID);
				dr.setRepositoryUniqueId(accessConsentRepositoryUniqueId);
				dr.setDocumentUniqueId(accessConsentDocumentUniqueId);
				rdsrt.getDocumentRequest().add(dr);
				accessConsentRetrieveRequest.setRetrieveDocumentSetRequest(rdsrt);

				// Assertion
				AssertionType asrt = createAssertion(assertion, samlPatientId);
				accessConsentRetrieveRequest.setAssertion(asrt);

				// target communities
				NhinTargetCommunitiesType targetCommunities = new NhinTargetCommunitiesType();
				NhinTargetCommunityType targetCommunity = new NhinTargetCommunityType();
				HomeCommunityType ssaHC = senderHC;
				ssaHC.setHomeCommunityId(ssaOID);
				targetCommunity.setHomeCommunity(ssaHC);
				targetCommunities.getNhinTargetCommunity().add(targetCommunity);
				accessConsentRetrieveRequest.setNhinTargetCommunities(targetCommunities);
				
				// call EntityDocRetieve
				accessConsentRetrieveResponse = entityDocRetrieve.respondingGatewayCrossGatewayRetrieve(accessConsentRetrieveRequest);
			}
			catch(Throwable t)
			{
				return false;
			}

			// check if patient authorized
			if(accessConsentRetrieveResponse.getRegistryResponse() == null)
			{
				return false;
			}
			else
			{
				if(accessConsentRetrieveResponse.getRegistryResponse().getStatus() == null)
				{
					if(accessConsentRetrieveResponse.getRegistryResponse().getRegistryErrorList() != null)
					{
						if(accessConsentRetrieveResponse.getRegistryResponse().getRegistryErrorList().getRegistryError() != null)
						{
							if(!accessConsentRetrieveResponse.getRegistryResponse().getRegistryErrorList().getRegistryError().isEmpty())
							{
								return false;
							}
						}
					}
				}
				else
				{
					if(!accessConsentRetrieveResponse.getRegistryResponse().getStatus().equalsIgnoreCase("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success"))
					{
						return false;
					}
				}
			}

			// See if the document is valid, that is creationTime > today - 1
			// year
			// creationTime format is YYYYMMDD
			Date accessConsentDocumentCreationDate;
			try
			{
				accessConsentDocumentCreationDate = new SimpleDateFormat("yyyyMMdd").parse(accessConsentDocumentCreationTime);
			}
			catch(ParseException ex)
			{
				return false;
			}

			Date yearAgo = new Date(System.currentTimeMillis() - MILLIS_IN_YEAR);
			if(accessConsentDocumentCreationDate.before(yearAgo))
			{
				return false;
			}

			// call VLER-DAS to save the ACP doc
			if(NullChecker.isNullOrEmpty(accessConsentRetrieveResponse.getDocumentResponse()))
			{
				return false;
			}

			acpDocument = getDocumentAsBytes(accessConsentRetrieveResponse.getDocumentResponse().get(0).getDocument());

			if(NullChecker.isNullOrEmpty(acpDocument))
			{
				return false;
			}
		}
		else
		{
			// SSA DQ/DR section skipped for testing
			accessConsentRetrieveResponse = new RetrieveDocumentSetResponseType();
			DocumentResponse docResponse = new DocumentResponse();
			String testDocString = "Test eAuth";
			acpDocument = testDocString.getBytes();
			DataHandler dataHandler = new DataHandler(acpDocument, "application/octet-stream");
			docResponse.setDocument(dataHandler);
			accessConsentRetrieveResponse.getDocumentResponse().add(docResponse);
		}

		DasOperationSendACPDocumentResponse res = null;
		if(!excludeSectionsString.contains("2"))
		{
			// get url from the property table
			String endPoint = propertyLookup.getProperty("vler.das.eauthsubmit.url");
			if(NullChecker.isNullOrEmpty(endPoint))
			{
				logger.error("VLER DAS ACP URL vler.das.eauthsubmit.url is not configured in the VA Adapter Properties table");
				return false;
			}

			// instantiate Das Http client
			DasDAO dasDAO;
			dasDAO = new DasDAOHttpImpl(endPoint);

			DasOperationSendACPDocumentRequest req;

			// call VLER DAS ACP push
			try
			{
				req = new DasOperationSendACPDocumentRequest();
				req.setMessage(acpDocument);

				res = dasDAO.doOperationSendACPDocument(req);

				auditACPDocument(assertion, accessConsentRetrieveResponse, res);
			}
			catch(DasException e)
			{
				return false;
			}
		}
		else
		{
			// DAS storage section skipped for testing
			res = new DasOperationSendACPDocumentResponse();
			res.setDocId("0000000000");
		}

		if(!excludeSectionsString.contains("3"))
		{
			try
			{
				if(!updateVapConsent(request, response, new Date(), res.getDocId()))
				{
					return false;
				}
			}
			catch(Exception e)
			{
				return false;
			}
		}

		return true;
	}

	private ConsentDirectiveAuthorizationResponseType authorizeVapConsent(RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response, ServiceConsumerContextType scct, Date docStartDate, String dasDocId)
	{
		logger.info("Entering authorizeVapConsent...");
		PRPAMT201310UV02Patient patient = HL7Parser201306.extractSubjectPatient(response);

		ConsentDirectiveData data = new ConsentDirectiveData();

		Patient patientInfo = HL7Parser201305.extractMpiPatientFromMessage(request.getPRPAIN201305UV02());
		data.setPatientRoleSsn(patientInfo.getSsn());
		data.setPatientRoleGivenName(patientInfo.getPersonnames().get(0).getFirstName());
		data.setPatientRoleFamilyName(patientInfo.getPersonnames().get(0).getLastName());
		data.setPatientRoleMiddleName(patientInfo.getPersonnames().get(0).getMiddleName());

		if(patient != null && patient.getId() != null && patient.getId().size() > 0)
		{
			II icn = HL7Parser201306.extractICNId(patient.getId());
			data.setIcn(HL7Parser201306.extractICNValue(icn.getExtension()));
		}

		if(patient == null)
		{
			logger.debug("patient is null");
		}
		else if(patient.getId() == null)
		{
			logger.debug("patient.getId() is null");
		}
		else
		{
			logger.debug("patient.getId() length: {}", patient.getId().size());
		}

		// For VA user information
		AssertionType assertionType = createAssertion(request.getAssertion(), data.getIcn());
		// Set author
		data.setAuthorPersonOid(assertionType.getUserInfo().getRoleCoded().getCode());
		data.setAuthorPersonOrgOid(assertionType.getUserInfo().getOrg().getHomeCommunityId());

		// Set the status to active
		data.setComponentStatusCode("active");
		data.setComponentPurposeOfUseDisplayName("COVERAGE");

		try
		{
			final String beginDateString = HL7DateUtil.yyyyMMddhhmmssZ(docStartDate);

			// Set the effective date
			data.setEffectiveDateTime(beginDateString);
			// Create the begin and end date
			data.setDocumentationBeginTime(beginDateString);
			java.util.Calendar dateCal = java.util.Calendar.getInstance();

			dateCal.setTime(docStartDate);
			// Add a year from creation date
			dateCal.add(java.util.Calendar.YEAR, 1);
			data.setDocumentationEndTime(HL7DateUtil.yyyyMMddhhmmssZ(dateCal.getTime()));
		}
		catch(final ParseException ex)
		{
			logger.error("Date parsing exception while creating VAP consent authorization request.");
			throw new RuntimeException(ex);
		}

		try
		{
			final byte[] consentDirectiveDocumentBytes = this.makeConsentDirectiveDocumentString(data);
			// Create the request and send to the consent management service
			final ConsentDirectiveAuthorizationRequestType consentAuthRequest = new ConsentDirectiveAuthorizationRequestType();
			consentAuthRequest.setDocument(consentDirectiveDocumentBytes);

			consentAuthRequest.setServiceConsumerContext(scct);
			consentAuthRequest.setDasDocumentId(dasDocId);

			logger.debug("Attempting to authorize new consent to VAP...");
			return vapConsentManagement.processConsentDirectiveAuthorization(consentAuthRequest);
		}
		catch(ConsentManagementServiceFaultMessage ex)
		{
			logger.error("Exception attempting to authorize VAP consent");
		}

		return null;
	}

	private boolean updateVapConsent(RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response, Date docStartDate, String dasDocId)
	{
		logger.debug("Entering updateVapConsent...");
		// First try to retrieve existing VAP consent directory for SSA auth
		ConsentDirectiveQueryRequestType consentDirectiveQuery = new ConsentDirectiveQueryRequestType();
		PRPAMT201310UV02Patient patient = HL7Parser201306.extractSubjectPatient(response);
		if(patient != null && patient.getId() != null && patient.getId().size() > 0)
		{
			II icn = HL7Parser201306.extractICNId(patient.getId());
			logger.debug("icn: {}", icn.toString());
			consentDirectiveQuery.setPatientId(HL7Parser201306.extractICNValue(icn.getExtension()));
			logger.debug("icn.getExtension(): {}", icn.getExtension());
		}

		if(patient == null)
		{
			logger.debug("patient is null");
		}
		else if(patient.getId() == null)
		{
			logger.debug("patient.getId() is null");
		}
		else
		{
			logger.debug("patient.getId() length: {}", patient.getId().size());
		}

		ConsentType ssaAuthType = ConsentType.SSA_AUTHORIZATION;
		ServiceConsumerContextType scct = new ServiceConsumerContextType();
		scct.setConsentType(ssaAuthType);
		scct.setUser(request.getAssertion().getUserInfo().getUserName());

        if(facilityManager.getFacilityByFacilityNumber("200NSS") != null) {
            scct.setFacility(facilityManager.getFacilityByFacilityNumber("200NSS").getFacilityName());
        }
        else {
            scct.setFacility("SSA");
        }

		scct.setServiceConsumerType(ServiceConsumer.EXCHANGE);
		consentDirectiveQuery.setServiceConsumerContext(scct);

		consentDirectiveQuery.setQueryParam(ConsentDirectiveQueryParamType.ACTIVE);

		ConsentDirectiveQueryResponseType consentDirectiveResponse;

		logger.info("Attempting to get current VAP consent directive for SSA authorization...");
		try
		{
			consentDirectiveResponse = vapConsentManagement.getConsentDirectives(consentDirectiveQuery);
		}
		catch(ConsentManagementServiceFaultMessage ex)
		{
			logger.error("Couldn't get VAP consent directive.");

			return false;
		}
		logger.info("Received VAP consent directive response");

		if(!NullChecker.isNullOrEmpty(consentDirectiveResponse) && !NullChecker.isNullOrEmpty(consentDirectiveResponse.getConsentDirectiveReference()) && !consentDirectiveResponse.getConsentDirectiveReference().isEmpty())
		{
			logger.info("Current VAP consent directive extracted");

			// Check if eAuth is newer than current consent directive
			logger.info("Checking if eAuth is newer than the current SSA authorization...");
			ConsentDirectiveReferenceType currentConsentReference = consentDirectiveResponse.getConsentDirectiveReference().get(0);
			Date currentConsentDate = currentConsentReference.getOptinDate();
			if(docStartDate.after(currentConsentDate))
			{
				logger.info("eAuth is newer tham current SSA authorization");
				// Delete existing consent directive first
				ConsentDirectiveRevocationRequestType revocationRequest = new ConsentDirectiveRevocationRequestType();
				revocationRequest.setOptoutReason(ConsentDirectiveOptOutReasonType.NEW_AUTHORIZATION);
				scct.setConsentType(ConsentType.SSA_REVOCATION);
				revocationRequest.setServiceConsumerContext(scct);

				// Create ConsentDirectiveData for revocation request from
				// response
				ConsentDirectiveData currentConsentData = new ConsentDirectiveData();
				currentConsentData.setDocId(currentConsentReference.getConsentDirId());
				currentConsentData.setIcn(currentConsentReference.getPatientIen());

				// Extract Patient info
				Patient patientInfo = HL7Parser201305.extractMpiPatientFromMessage(request.getPRPAIN201305UV02());
				currentConsentData.setPatientRoleSsn(patientInfo.getSsn());
				currentConsentData.setPatientRoleGivenName(patientInfo.getPersonnames().get(0).getFirstName());
				currentConsentData.setPatientRoleFamilyName(patientInfo.getPersonnames().get(0).getLastName());
				currentConsentData.setPatientRoleMiddleName(patientInfo.getPersonnames().get(0).getMiddleName());
				try
				{
					currentConsentData.setDocumentationBeginTime(HL7DateUtil.yyyyMMddhhmmssZ(currentConsentReference.getOptinDate()));
					currentConsentData.setDocumentationEndTime(HL7DateUtil.yyyyMMddhhmmssZ(currentConsentReference.getExpirationDate()));
				}
				catch(ParseException ex)
				{
					logger.error("Could not parse current consent Opt-in or expiration date");
					return false;
				}
				currentConsentData.setComponentStatusCode("aborted");
				currentConsentData.setComponentPurposeOfUseDisplayName("COVERAGE");
				final byte[] consentDirectiveDocumentBytes = this.makeConsentDirectiveDocumentString(currentConsentData);
				revocationRequest.setDocument(consentDirectiveDocumentBytes);
				logger.info("Consent revocation request created for VAP");

				try
				{
					logger.debug("Attempting to revoke old VAP consent...");
					ConsentDirectiveRevocationResponseType revocationResponse = vapConsentManagement.processConsentDirectiveRevocation(revocationRequest);

					if(NullChecker.isNullOrEmpty(revocationResponse) || NullChecker.isNullOrEmpty(revocationResponse.getConsentDirectiveReference()))
					{
						// Consent revocation failed
						logger.info("Failed to revoke old consent. Returning to regulare PD flow...");
						return false;
					}
				}
				catch(ConsentManagementServiceFaultMessage ex)
				{
					logger.error("Error attempting to revoke old VAP consent");
					return false;
				}
				logger.info("Old VAP consent successfully revoked");

				scct.setConsentType(ssaAuthType);
				ConsentDirectiveAuthorizationResponseType consentManagementResponse = authorizeVapConsent(request, response, scct, docStartDate, dasDocId);

				if(NullChecker.isNullOrEmpty(consentManagementResponse) || NullChecker.isNullOrEmpty(consentManagementResponse.getConsentDirectiveReference()))
				{
					// Consent authorization failed
					logger.error("VAP consent authorization failed.");
					return false;
				}
			}
		}
		else
		{
			logger.debug("No current SSA authorization consent directive. Attempting to authorize...");
			ConsentDirectiveAuthorizationResponseType consentManagementResponse = authorizeVapConsent(request, response, scct, docStartDate, dasDocId);

			if(NullChecker.isNullOrEmpty(consentManagementResponse) || NullChecker.isNullOrEmpty(consentManagementResponse.getConsentDirectiveReference()))
			{
				// Consent authorization failed
				logger.error("VAP consent authorization failed.");
				return false;
			}
		}

		return true;
	}

	/**
	 * Convert from the ConsentDirectiveData to the CDA R2 XML Privacy consent
	 * directive document and then convert that to string.
	 */
	private byte[] makeConsentDirectiveDocumentString(final ConsentDirectiveData data)
	{
		try
		{
			// Convert the ConsentDirectiveData to XML document
			JAXBContext consentDirectiveJaxbContext = JAXBContext.newInstance(ConsentDirectiveData.class);

			StringWriter stringWriter = new StringWriter();
			Marshaller consentDirectiveMarshaller = consentDirectiveJaxbContext.createMarshaller();
			consentDirectiveMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			consentDirectiveMarshaller.marshal(data, stringWriter);
			String consentDirectiveDocumentString = stringWriter.toString();

			return consentDirectiveDocumentString.getBytes();
		}
		catch(final JAXBException ex)
		{
			throw new RuntimeException(ex);
		}
	}

	public II extractPatientIdFrom201305(PRPAIN201305UV02 request)
	{
		II patId = null;
		String aaId;

		if(request != null && request.getControlActProcess() != null)
		{
			aaId = getAAOID(request);

			if(!NullChecker.isNullOrEmpty(aaId))
			{
				if(request.getControlActProcess().getQueryByParameter() != null && request.getControlActProcess().getQueryByParameter().getValue() != null && request.getControlActProcess().getQueryByParameter().getValue().getParameterList() != null && !NullChecker.isNullOrEmpty(request.getControlActProcess().getQueryByParameter().getValue().getParameterList().getLivingSubjectId()))
				{
					for(PRPAMT201306UV02LivingSubjectId livingSubId : request.getControlActProcess().getQueryByParameter().getValue().getParameterList().getLivingSubjectId())
					{
						for(II id : livingSubId.getValue())
						{
							if(id != null && !NullChecker.isNullOrEmpty(id.getRoot()) && !NullChecker.isNullOrEmpty(id.getExtension()) && aaId.contentEquals(id.getRoot()))
							{
								patId = new II();
								patId.setRoot(id.getRoot());
								patId.setExtension(id.getExtension());

								// break out of inner loop
								break;
							}
						}

						// If the patient id was found then break out of outer
						// loop
						if(patId != null)
						{
							break;
						}
					}
				}
			}
		}

		return patId;
	}

	private void addPatientCorrelation(RespondingGatewayPRPAIN201305UV02RequestType request, PRPAIN201306UV02 response, Patient searchResultPatient, boolean correlationAlreadyExists)
	{
		PRPAIN201301UV02 prpain201301UV02 = AdapterHL7PRPA201301Transforms.createPRPA201301(request.getPRPAIN201305UV02(), getHomeCommunityId());
		/*
		 * 04/30/2014 - SR345 - With the old RPC calls the results were mapped
		 * to a PRPAIN201306UV02 response and the ICN was inserted as the first
		 * element in the Subject1->Patient->Id list. Now that we are using the
		 * MVI web service and using the PRPAIN201306UV02 response they
		 * returned/populated there's no guarantee ICN will be first in the
		 * list. Therefore need to search for it.
		 */
		// II localPatientId =
		// response.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId().get(0);
		II localPatientId = HL7Parser201306.extractICNId(response);
		prpain201301UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId().add(localPatientId);
		AddPatientCorrelationRequestType r = new AddPatientCorrelationRequestType();
		r.setPRPAIN201301UV02(prpain201301UV02);
		r.setAssertion(request.getAssertion());
		adapterPatientCorrelation.addPatientCorrelation(r, searchResultPatient, correlationAlreadyExists);
	}

	private PRPAIN201306UV02 createPatientNotFoundResponse(RespondingGatewayPRPAIN201305UV02RequestType request, String homeCommunityId, String assigningAuthorityId)
	{
		return HL7DbParser201306.BuildMessageFromMpiPatients(null, request.getPRPAIN201305UV02(), homeCommunityId, assigningAuthorityId);
	}

	private PRPAIN201306UV02 createErrorResponse(RespondingGatewayPRPAIN201305UV02RequestType request, String homeCommunityId, String assigningAuthorityId)
	{
		PRPAIN201306UV02 ret = HL7DbParser201306.BuildMessageFromMpiPatients(null, request.getPRPAIN201305UV02(), homeCommunityId, assigningAuthorityId);

		// set Applicaton Error code
		CS appError = new CS();
		appError.setCode("AE");
		ret.getAcknowledgement().get(0).setTypeCode(appError);

		// set Query Error Code
		ret.getControlActProcess().getQueryAck().getQueryResponseCode().setCode("QE");

		return ret;
	}

	private String getSenderOID(PRPAIN201305UV02 request)
	{
		String ret = null;

		if(request.getSender() != null && request.getSender().getDevice() != null && request.getSender().getDevice().getAsAgent() != null && request.getSender().getDevice().getAsAgent().getValue() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId() != null && !NullChecker.isNullOrEmpty(request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId()))
		{
			ret = request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();
		}

		return ret;
	}

	private String getAAOID(PRPAIN201305UV02 request)
	{
		return HL7Parser201305.extractRemoteAssigningAuthorityID(request);
	}

	private String getReceiverOID(PRPAIN201305UV02 request)
	{
		String ret = null;

		if(request != null && !NullChecker.isNullOrEmpty(request.getReceiver()) && request.getReceiver().get(0) != null && request.getReceiver().get(0).getDevice() != null && request.getReceiver().get(0).getDevice().getAsAgent() != null && request.getReceiver().get(0).getDevice().getAsAgent().getValue() != null && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && !NullChecker.isNullOrEmpty(request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId()) && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0) != null && !NullChecker.isNullOrEmpty(request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot()))
		{
			ret = request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();
		}

		return ret;
	}

	private AssertionType createAssertion(AssertionType fromAssertion, String patientId)
	{
		AssertionType asrt = new AssertionType();
		Facility homeFacility = getHomeFacility();

		asrt.setAuthorized(true);

		HomeCommunityType homeCommunity = new HomeCommunityType();
		homeCommunity.setHomeCommunityId(homeFacility.getFullHomeCommunityId());
		homeCommunity.setName(homeFacility.getFacilityName());
		asrt.setHomeCommunity(homeCommunity);

		UserType user = new UserType();
		CeType roleCoded = new CeType();
		roleCoded.setCode("224608005");
		user.setUserName("VA_USER");
		PersonNameType personName = new PersonNameType();
		personName.setGivenName("VA");
		personName.setFamilyName("User");
		user.setPersonName(personName);

		if(user.getPersonName() != null && (NullChecker.isNotNullOrEmpty(user.getPersonName().getFamilyName()) || NullChecker.isNotNullOrEmpty(user.getPersonName().getSecondNameOrInitials()) || NullChecker.isNotNullOrEmpty(user.getPersonName().getGivenName())))
		{

			user.setUserName(user.getUserName() + ", CN=" + createFullName(user.getPersonName()) + ", O=" + homeFacility.getFacilityName());
		}
		else
		{
			user.setUserName(user.getUserName() + ", CN=" + user.getUserName() + ", O=" + homeFacility.getFacilityName());
		}

		roleCoded.setCodeSystem("2.16.840.1.113883.6.96");
		roleCoded.setCodeSystemName("SNOMED_CT");

		user.setRoleCoded(roleCoded);
		user.setOrg(homeCommunity);
		asrt.setUserInfo(user);

		asrt.getUniquePatientId().add(patientId);

		CeType p = new CeType();
		p.setCodeSystem("2.16.840.1.113883.3.18.7.1");
		p.setCodeSystemName("nhin-purpose");
		p.setDisplayName("COVERAGE");
		if(fromAssertion.getPurposeOfDisclosureCoded() != null)
		{
			p.setCode(fromAssertion.getPurposeOfDisclosureCoded().getCode());
		}
		else
		{
			p.setCode("COVERAGE");
		}
		asrt.setPurposeOfDisclosureCoded(p);

		SamlIssuerType sit = new SamlIssuerType();
		sit.setIssuer(propertyLookup.getProperty("AssertionIssuer"));
		sit.setIssuerFormat(propertyLookup.getProperty("AssertionIssuerFormat"));
		asrt.setSamlIssuer(sit);

		asrt.setMessageId("urn:uuid:" + UUID.randomUUID().toString());

		return asrt;
	}

	private Facility getHomeFacility()
	{
		return facilityManager.getFacilityByFacilityNumber("VA");
	}

	private Facility getHomeFacility(String homeCommunityId)
	{
		Facility facility;

		facility = facilityManager.getFacilityByFullHomeCommunityId(homeCommunityId);

		if(NullChecker.isNotNullOrEmpty(facility))
		{
			return facility;
		}

		facility = facilityManager.getFacilityByHomeCommunityId(homeCommunityId);

		return facility;
	}

	private String getHomeCommunityId()
	{
		String ret = null;
		Facility facility = facilityManager.getFacilityByFacilityNumber("VA");

		if(facility != null)
		{
			ret = facility.getHomeCommunityId();
		}

		return ret;
	}

	private String getHomeCommunityId(AssertionType assertion)
	{
		if(NullChecker.isNullOrEmpty(assertion))
		{
			logger.error("Assertion section is empty");
			return null;
		}

		if(NullChecker.isNullOrEmpty(assertion.getHomeCommunity()))
		{
			logger.error("HomeCommunity section is empty");
			return null;
		}

		return assertion.getHomeCommunity().getHomeCommunityId();
	}

	private String getFacilityNumber(String hcid)
	{
		String ret = null;
		Facility facility = facilityManager.getFacilityByHomeCommunityId(hcid);

		if(facility != null)
		{
			ret = facility.getFacilityNumber();
		}

		return ret;
	}

	private String getAssigningAuthorityId()
	{
		return propertyLookup.getProperty("AssigningAuthority");
	}

	private String createFullName(PersonNameType personName)
	{
		StringBuilder ret = new StringBuilder();

		if(NullChecker.isNotNullOrEmpty(personName.getGivenName()))
		{
			ret.append(personName.getGivenName());
		}

		if(NullChecker.isNotNullOrEmpty(personName.getSecondNameOrInitials()))
		{
			if(ret.length() > 0)
			{
				ret.append(' ');
			}
			ret.append(personName.getSecondNameOrInitials());
		}

		if(NullChecker.isNotNullOrEmpty(personName.getFamilyName()))
		{
			if(ret.length() > 0)
			{
				ret.append(' ');
			}
			ret.append(personName.getFamilyName());
		}

		return ret.toString();
	}

	private void auditACPDocument(AssertionType assertion, RetrieveDocumentSetResponseType body, DasOperationSendACPDocumentResponse response)
	{
		logger.info("In auditACPDocument...");

		// create record in the DOCUMENT table
		Document document = new Document();

		byte[] acpBytes = getDocumentAsBytes(body.getDocumentResponse().get(0).getDocument());
		document.setDocumentUniqueId(response.getDocId());
		document.setRawData(acpBytes);
		document.setSize(acpBytes.length);
		document.setAvailabilityStatus("urn:oasis:names:tc:ebxmlregrep:StatusType:Approved");
		document.setLastAccessedTime(new Date());
		document.setCreationTime(new Date());
		document.setClassCode("57016-8");
		document.setClassCodeDisplayName("Privacy Policy Acknowledgement");
		document.setClassCodeScheme("1.3.6.1.4.1.19376.1.2.3");
		document.setMimeType("text/xml");
		document.setPatientId(assertion.getUniquePatientId().get(0));
		document.setFormatCodeDisplayName("Privacy Policy Acknowledgement");
		document.setSourcePatientId(assertion.getUniquePatientId().get(0));
		document.setPatientSSN(assertion.getSSN());

		if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo()))
		{

			if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getOrg()))
			{
				// document.setPatientPreferredFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
				document.setPatientPreferredFacilityName(assertion.getUserInfo().getOrg().getName());
			}

			if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getPersonName()))
			{
				PersonNameType userName = assertion.getUserInfo().getPersonName();
				document.setPatientGivenName(userName.getGivenName());
				document.setPatientLastName(userName.getFamilyName());
			}
		}
		logger.info("Document object successfully created");

		documentRepository.storeDocument(document);
		logger.info("Document stored to DAS");

		logger.info("Building audit object for document submission...");
		// create record in the AUDIT table
		Audit audit = new Audit();

		audit.setAction("DocumentSubmissionIn");
		audit.setAuditTime(new Date());
		audit.setDocumentId(document.getDocumentUniqueId());
		audit.setRemoteDocumentId(document.getDocumentUniqueId());
		// audit.setRemoteDocumentRepositoryId(d.getRepositoryUniqueId());
		audit.setRemoteOrganizationId(getHomeCommunityId(assertion));

		if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo()))
		{
			audit.setUserId(assertion.getUserInfo().getUserName());

			if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getRoleCoded()))
			{
				audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
			}

			if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getOrg()))
			{
				audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
				audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
			}

			if(NullChecker.isNotNullOrEmpty(assertion.getUserInfo().getPersonName()))
			{
				PersonNameType userName = assertion.getUserInfo().getPersonName();
				if(!NullChecker.isNullOrEmpty(userName.getFullName()))
				{
					audit.setUserName(userName.getFullName());
				}
				else
				{
					audit.setUserName(userName.getGivenName() + " " + userName.getFamilyName());
				}
			}
		}

		if(NullChecker.isNotNullOrEmpty(assertion.getPurposeOfDisclosureCoded()))
		{
			audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
		}

		audit.setOrganizationId(getHomeCommunityId(assertion));
		audit.setPatientId(assertion.getUniquePatientId().get(0));
		audit.setPatientSSN(document.getPatientSSN());
		if(NullChecker.isNotNullOrEmpty(assertion.getPersonName()))
		{
			audit.setPatientGivenName(assertion.getPersonName().getGivenName());
			audit.setPatientLastName(assertion.getPersonName().getFamilyName());
		}
		audit.setPatientFacilityNumber(getHomeFacility(getHomeCommunityId(assertion)).getFacilityNumber());
		audit.setPatientFacilityName(getHomeFacility(getHomeCommunityId(assertion)).getFacilityName());

		audit.setDetails("VLER DAS Doc Id: " + response.getDocId());
		logger.info("Audit object created");

		auditManager.storeAudit(audit);
		logger.info("Audit stored");
	}

	private byte[] getDocumentAsBytes(DataHandler doc)
	{
		if(NullChecker.isNullOrEmpty(doc))
		{
			logger.error("Document section is empty");
			return null;
		}

		byte[] docBytes = null;
		InputStream is = null;
		try
		{
			is = doc.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
			// String encoding = isr.getEncoding();
			docBytes = IOUtils.toByteArray(isr);
		}
		catch(IOException e)
		{
			logger.error("Error getting Document from ProvideAndRegisterDocumentSet");
			return null;
		}
		// fix for fortify issue - Unreleased Resource: Streams RTC ticket #
		// 163005
		finally
		{
			if(is != null)
			{
				try
				{
					is.close();
				}
				catch(IOException e)
				{
					logger.error("Error getting Document from ProvideAndRegisterDocumentSet");
				}
			}
		}

		if(NullChecker.isNullOrEmpty(docBytes))
		{
			logger.error("Document content is empty");
			return null;
		}

		return docBytes;
	}
}
