package gov.va.med.nhin.adapter.adaptergateway.patientdiscovery;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;

import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.jws.WebService;
import javax.xml.ws.BindingType;

import org.apache.commons.collections.CollectionUtils;
import org.hl7.v3.AddPatientCorrelationRequestType;
import org.hl7.v3.CommunityPRPAIN201306UV02ResponseType;
import org.hl7.v3.II;
import org.hl7.v3.MCCIMT000100UV01Receiver;
import org.hl7.v3.PRPAIN201301UV02;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject1;
import org.hl7.v3.PRPAMT201301UV02Patient;
import org.hl7.v3.PRPAMT201310UV02Patient;
import org.hl7.v3.RespondingGatewayPRPAIN201305UV02RequestType;
import org.hl7.v3.RespondingGatewayPRPAIN201306UV02ResponseType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.NhinTargetCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PRPA201301Transforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PRPA201305Transforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7PatientTransforms;
import gov.hhs.fha.nhinc.transform.subdisc.HL7ReceiverTransforms;
import gov.va.med.nhin.adapter.adaptergateway.docquery.WorkManagerExecutorServiceLocal;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.audit.requests.RequestActions;
import gov.va.med.nhin.adapter.audit.requests.RequestAudit;
import gov.va.med.nhin.adapter.audit.requests.RequestAuditEntity;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.OperationOnOff;
import gov.va.med.nhin.adapter.logging.AdapterPDError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.mpi.hl7parsers.HL7DbParser201306;
import gov.va.med.nhin.adapter.mvi.AdapterMviPortTypeLocal;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.patientcorrelation.PatientCorrelationPortTypeLocal;
import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author David Vazquez
 */
@WebService(serviceName = "EntityPatientDiscovery", portName = "EntityPatientDiscoveryPortSoap", endpointInterface = "gov.hhs.fha.nhinc.entitypatientdiscovery.EntityPatientDiscoveryPortType", targetNamespace = "urn:gov:hhs:fha:nhinc:entitypatientdiscovery", wsdlLocation = "META-INF/wsdl/EntityPatientDiscovery.wsdl")
@BindingType(value = javax.xml.ws.soap.SOAPBinding.SOAP12HTTP_BINDING)
@Stateless(name = "AdapterGatewayPatientDiscovery")
public class AdapterGatewayPatientDiscovery implements EntityPatientDiscoveryPortTypeLocal
{
	private static final Logger logger = LoggerFactory.getLogger(AdapterGatewayPatientDiscovery.class.getName());

	private class SendRequestResponse
	{

		public PRPAIN201306UV02 prpain201306UV02;
		public String remoteHomeCommunityId;
		public SendRequestException sre = null;

		public boolean hasError()
		{
			if(sre == null)
			{
				return false;
			}
			return true;
		}
	}

	private class SendRequestException extends Exception
	{

		private static final long serialVersionUID = 1L;

		public PRPAIN201305UV02 prpain201305UV02;
		public String remoteHomeCommunityId;
	}

	private class SendRequestCallable implements Callable<SendRequestResponse>
	{
		private final PRPAIN201305UV02 prpain201305UV02;
		private final AssertionType assertion;
		private final String remoteHomeCommunityId;

		public SendRequestCallable(PRPAIN201305UV02 prpain201305UV02, AssertionType assertion, String remoteHomeCommunityId)
		{
			this.prpain201305UV02 = prpain201305UV02;
			this.assertion = assertion;
			this.remoteHomeCommunityId = remoteHomeCommunityId;
		}

		public SendRequestResponse call() throws Exception
		{
			return sendRequest(prpain201305UV02, assertion, remoteHomeCommunityId);
		}
	}

	private PatientCorrelationPortTypeLocal adapterPatientCorrelation;
	private FacilityManager facilityManager;
	private AdapterMviPortTypeLocal adapterMvi;
	private NHINPatientDiscoverySender nhinPatientDiscoverySender;
	private ExecutorService executorService;
	private RequestAudit requestAudit;
	private boolean successfulPD = false;

	private AuditManager auditManager;

	@EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
	public void setAuditManager(AuditManager auditManager)
	{
		this.auditManager = auditManager;
	}

	@EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
	public void setFacilityManager(FacilityManager facilityManager)
	{
		this.facilityManager = facilityManager;
	}

	@EJB(beanInterface = AdapterMviPortTypeLocal.class, beanName = "AdapterMVI")
	public void setAdapterMvi(AdapterMviPortTypeLocal adapterMvi)
	{
		this.adapterMvi = adapterMvi;
	}

	@EJB(beanInterface = PatientCorrelationPortTypeLocal.class, beanName = "AdapterPatientCorrelation")
	public void setAdapterPatientCorrelation(PatientCorrelationPortTypeLocal adapterPatientCorrelation)
	{
		this.adapterPatientCorrelation = adapterPatientCorrelation;
	}

	@EJB(beanInterface = NHINPatientDiscoverySender.class, beanName = "NhincProxyNHINPatientDiscoverySender")
	public void setNhinPatientDiscoverySender(NHINPatientDiscoverySender nhinPatientDiscoverySender)
	{
		this.nhinPatientDiscoverySender = nhinPatientDiscoverySender;
	}

	@EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "WorkManagerExecutorService")
	public void setExecutorService(ExecutorService executorService)
	{
		this.executorService = executorService;
	}

	@EJB(beanInterface = RequestAudit.class, beanName = "RequestAudit")
	public void setRequestAudit(RequestAudit requestAudit)
	{
		this.requestAudit = requestAudit;
	}

	public RespondingGatewayPRPAIN201306UV02ResponseType respondingGatewayPRPAIN201305UV02(RespondingGatewayPRPAIN201305UV02RequestType respondingGatewayPRPAIN201305UV02Request)
	{
		this.successfulPD = false;
		String partnerCommunityId = respondingGatewayPRPAIN201305UV02Request.getAssertion().getHomeCommunity().getHomeCommunityId();

		logger.debug("EnteringAdapterGatewayPatientDiscovery Call  and Partner homecommunity ID is {} ", partnerCommunityId);

		RespondingGatewayPRPAIN201306UV02ResponseType ret = new RespondingGatewayPRPAIN201306UV02ResponseType();
		ExecutorCompletionService<SendRequestResponse> completionService = new ExecutorCompletionService<SendRequestResponse>(executorService);


                int numSubmitted = 0;
                II localPatientId = getLocalPatientId(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02());
                AssertionType assertion = respondingGatewayPRPAIN201305UV02Request.getAssertion();

                List<Facility> facilities = getFacilitiesFromRequest(respondingGatewayPRPAIN201305UV02Request);
                if(CollectionUtils.isEmpty(facilities))
                {
                        String[] excludedFacilities = {"VA"}; //remove VA facility to avoid self-querying
                        facilities = facilityManager.getAllFacilities(excludedFacilities);
                }

                for(Facility facility : facilities)
                {
                        String remoteHomeCommunityId = facility.getHomeCommunityId();

                        if(facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.ONBOARD))
                        {
                                if(facilityManager.isPartnerAllowed(remoteHomeCommunityId, OperationOnOff.OUT_PD))
                                {
                                        PRPAIN201305UV02 prpain201305UV02 = createPRPAIN201305UV02(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02(), remoteHomeCommunityId);
                                        logger.info("Sending PRPAIN201305UV02 request for localPatientId: '" + localPatientId.getExtension() + "' to remoteHomeCommunityId: '" + remoteHomeCommunityId + "'");
                                        SendRequestCallable callable = new SendRequestCallable(prpain201305UV02, respondingGatewayPRPAIN201305UV02Request.getAssertion(), remoteHomeCommunityId);
                                        completionService.submit(callable);
                                        ++numSubmitted;
                                }
                                else
                                {
                                        addResultToResponse(ret, createErrorResponse(createPRPAIN201305UV02(respondingGatewayPRPAIN201305UV02Request.getPRPAIN201305UV02(), remoteHomeCommunityId), remoteHomeCommunityId, remoteHomeCommunityId));
                                        AdapterPDError.queryError(null, ErrorMessage.OUT_PD_DISABLED, remoteHomeCommunityId);
                                }
                        }
                        else
                        {
                                AdapterPDError.queryError(null, ErrorMessage.OUT_PD_NOT_A_PARTNER, remoteHomeCommunityId);
                        }
                }

                for(int i = 0; i < numSubmitted; ++i)
                {
                        try
                        {
                                SendRequestResponse response = completionService.take().get();
                                if(!response.hasError())
                                {
                                        processResult(response.prpain201306UV02, localPatientId, response.remoteHomeCommunityId, assertion);
                                        addResultToResponse(ret, response.prpain201306UV02);
                                }
                                else
                                {
                                        SendRequestException sre = response.sre;
                                        addResultToResponse(ret, createErrorResponse(sre.prpain201305UV02, sre.remoteHomeCommunityId, sre.remoteHomeCommunityId));

                                        // CCR 177986
                                        logger.error("Error sending PatientDiscovery to {} - {}", new Object[] { sre.remoteHomeCommunityId, sre });
                                }
                        }
                        catch(InterruptedException ie)
                        {
                                logger.error("processing an outbound patient discovery request was interrupted {}.", ie);
                        }
                        catch(ExecutionException e)
                        {
                                logger.error("An exception occured processing an outbound patient discovery request.", e);
                        }
                }
	

		// Audit Entire PD outbound process status
		// If At least one partner responds with queryResponseCode of 'OK', its
		// a success.
		try
		{
			auditPDStatus(respondingGatewayPRPAIN201305UV02Request.getAssertion(), localPatientId);
		}
		catch(Throwable ex)
		{
			logger.error("Exeption while auditing PD Outbound process status to database. Please correct this.", ex);
		}

		return ret;
	}

	private void auditPDStatus(AssertionType assertion, II localPatientId)
	{
		Audit audit = new Audit();
		audit.setAction("PatientDiscoveryOut");
		if(this.successfulPD)
		{
			audit.setDetails("Success");
		}
		else
		{
			audit.setDetails("Failure - None of the partners reponded with patient information");
		}
		audit.setAuditTime(new Date());
		audit.setUserId(assertion.getUserInfo().getUserName());
		audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
		audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
		audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
		if(assertion.getUserInfo().getPersonName() != null)
		{
			PersonNameType personName = assertion.getUserInfo().getPersonName();
			if(!NullChecker.isNullOrEmpty(personName.getFullName()))
			{
				audit.setUserName(personName.getFullName());
			}
			else
			{
				audit.setUserName(personName.getGivenName() + " " + personName.getFamilyName());
			}
		}
		audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
		audit.setOrganizationId(assertion.getHomeCommunity().getHomeCommunityId());
		audit.setPatientId(localPatientId.getExtension());
		// audit.setPatientSSN(localPatientId.getPatientSSN());
		// audit.setPatientGivenName(document.getPatientGivenName());
		// audit.setPatientLastName(document.getPatientLastName());
		// audit.setPatientFacilityNumber(document.getPatientPreferredFacilityNumber());
		// audit.setPatientFacilityName(document.getPatientPreferredFacilityName());

		// audit.setDocSpecType(document.getFormatCodeDisplayName());
		auditManager.storeAudit(audit);
	}

	private List<Facility> getFacilitiesFromRequest(RespondingGatewayPRPAIN201305UV02RequestType request)
	{
		List<Facility> facilities = new ArrayList<Facility>();

		if(request.getNhinTargetCommunities() != null && request.getNhinTargetCommunities().getNhinTargetCommunity() != null && request.getNhinTargetCommunities().getNhinTargetCommunity().size() > 0)
		{
			List<NhinTargetCommunityType> nhinTargetCommunityList = request.getNhinTargetCommunities().getNhinTargetCommunity();
			for(NhinTargetCommunityType nhinTargetCommunity : nhinTargetCommunityList)
			{
				if(nhinTargetCommunity.getHomeCommunity() != null)
				{
					if(nhinTargetCommunity.getHomeCommunity().getHomeCommunityId() != null)
					{
						String remoteHomeCommunityId = nhinTargetCommunity.getHomeCommunity().getHomeCommunityId();
						Facility facility = facilityManager.getFacilityByHomeCommunityId(remoteHomeCommunityId);
						facilities.add(facility);
					}
				}
			}
		}

		return facilities;
	}

	private II getLocalPatientId(PRPAIN201305UV02 request)
	{
		II ret = null;
		List<II> patientIds = request.getControlActProcess().getQueryByParameter().getValue().getParameterList().getLivingSubjectId().get(0).getValue();
		String aa = getAAOID(request);
		for(II patientId : patientIds)
		{
			if(aa.equals(patientId.getRoot()))
			{
				ret = patientId;
				break;
			}
		}
		return ret;
	}

	private String getAAOID(PRPAIN201305UV02 request)
	{
		String oid = null;

		if(request != null && request.getControlActProcess() != null && !NullChecker.isNullOrEmpty(request.getControlActProcess().getAuthorOrPerformer()) && request.getControlActProcess().getAuthorOrPerformer().get(0) != null && request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice() != null && request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue() != null && !NullChecker.isNullOrEmpty(request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId()) && request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0) != null && !NullChecker.isNullOrEmpty(request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0).getRoot()))
		{
			oid = request.getControlActProcess().getAuthorOrPerformer().get(0).getAssignedDevice().getValue().getId().get(0).getRoot();
		}

		return oid;
	}

	private void auditError(final PRPAIN201305UV02 request, final String remoteHCID, final String errorMsg)
	{
		RequestAuditEntity auditEntity = new RequestAuditEntity();
		auditEntity.setStartTime(new Date());
		II ii = getQueryId(request);
		if(ii == null)
		{
			auditEntity.setUuid("Error getting UUID");
		}
		else
		{
			auditEntity.setUuid(ii.getExtension());
		}

		ii = getPID(request);

		if(ii == null)
		{
			auditEntity.setPid("Error getting PID");
		}
		else
		{
			auditEntity.setPid(ii.getExtension());
		}

		auditEntity.setHcid(remoteHCID);
		auditEntity.setAction(RequestActions.PATIENT_DISCOVERY.getValue());

		auditEntity.setStopTime(new Date());
		auditEntity.setStatus(errorMsg);
		requestAudit.storeAudit(auditEntity);
	}

	private II getQueryId(final PRPAIN201305UV02 prpain201305UV02)
	{
		if(prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId() != null)
		{
			return prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId();
		}
		return null;
	}

	private II getPID(final PRPAIN201305UV02 prpain201305UV02)
	{
		if(prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId() != null)
		{
			return prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId();
		}
		return null;
	}

	private SendRequestResponse sendRequest(PRPAIN201305UV02 prpain201305UV02, AssertionType assertion, String remoteHomeCommunityId)
	{
		SendRequestResponse ret = new SendRequestResponse();
		RequestAuditEntity auditEntity = new RequestAuditEntity();
		auditEntity.setStartTime(new Date());
		if(prpain201305UV02 != null && prpain201305UV02.getControlActProcess() != null && prpain201305UV02.getControlActProcess().getQueryByParameter() != null && prpain201305UV02.getControlActProcess().getQueryByParameter().getValue() != null)
		{
			if(prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId() != null)
			{
				auditEntity.setUuid(prpain201305UV02.getControlActProcess().getQueryByParameter().getValue().getQueryId().getExtension());
			}
			else
			{
				auditEntity.setUuid("Error getting UUID");
			}
		}
		auditEntity.setHcid(remoteHomeCommunityId);
		auditEntity.setAction(RequestActions.PATIENT_DISCOVERY.getValue());

		try
		{
			ret.prpain201306UV02 = nhinPatientDiscoverySender.sendRequest(prpain201305UV02, assertion, remoteHomeCommunityId);
			ret.remoteHomeCommunityId = remoteHomeCommunityId;
			auditEntity.setStopTime(new Date());
			if(ret.prpain201306UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId() != null)
			{
				auditEntity.setPid(ret.prpain201306UV02.getControlActProcess().getQueryByParameter().getValue().getParameterList().getId().getExtension());
			}
			else
			{
				auditEntity.setPid("Error getting pid");
			}
			String queryResponseCode = ret.prpain201306UV02.getControlActProcess().getQueryAck().getQueryResponseCode().getCode();
			auditEntity.setStatus(queryResponseCode);
			// check if successful PD is false and query response code is 'ok'
			// from the partner
			if((!this.successfulPD) && queryResponseCode != null && queryResponseCode.equalsIgnoreCase("ok"))
			{
				this.successfulPD = true;
			}
			requestAudit.storeAudit(auditEntity);
		}
		catch(Throwable e)
		{
			// Handle all errors on a per thread basis independently above.
			SendRequestException sre = new SendRequestException();
			sre.prpain201305UV02 = prpain201305UV02;
			sre.remoteHomeCommunityId = remoteHomeCommunityId;
			sre.initCause(e);
			ret.sre = sre;
			auditEntity.setStatus("Failed: " + e.getMessage());
			requestAudit.storeAudit(auditEntity);
		}
		return ret;
	}

	private PRPAIN201305UV02 createPRPAIN201305UV02(PRPAIN201305UV02 fromRequest, String homeCommunityId)
	{
		PRPAIN201305UV02 ret = new PRPAIN201305UV02();
		ret.setAcceptAckCode(fromRequest.getAcceptAckCode());
		ret.setControlActProcess(fromRequest.getControlActProcess());
		ret.setCreationTime(fromRequest.getCreationTime());
		ret.setITSVersion(fromRequest.getITSVersion());
		ret.setId(fromRequest.getId());
		ret.setInteractionId(fromRequest.getInteractionId());
		ret.setProcessingCode(fromRequest.getProcessingCode());
		ret.setProcessingModeCode(fromRequest.getProcessingModeCode());
		ret.setSecurityText(fromRequest.getSecurityText());
		ret.setSender(fromRequest.getSender());
		ret.setSequenceNumber(fromRequest.getSequenceNumber());
		ret.setTypeId(fromRequest.getTypeId());
		ret.setVersionCode(fromRequest.getVersionCode());

		if(NullChecker.isNotNullOrEmpty(fromRequest.getAttachmentText()))
		{
			ret.getAttachmentText().addAll(fromRequest.getAttachmentText());
		}

		if(NullChecker.isNotNullOrEmpty(fromRequest.getAttentionLine()))
		{
			ret.getAttentionLine().addAll(fromRequest.getAttentionLine());
		}

		if(NullChecker.isNotNullOrEmpty(fromRequest.getNullFlavor()))
		{
			ret.getNullFlavor().addAll(fromRequest.getNullFlavor());
		}

		if(NullChecker.isNotNullOrEmpty(fromRequest.getProfileId()))
		{
			ret.getProfileId().addAll(fromRequest.getProfileId());
		}

		if(NullChecker.isNotNullOrEmpty(fromRequest.getRealmCode()))
		{
			ret.getRealmCode().addAll(fromRequest.getRealmCode());
		}

		ret.getReceiver().add(createNewReceiver(homeCommunityId));

		if(NullChecker.isNotNullOrEmpty(fromRequest.getRespondTo()))
		{
			ret.getRespondTo().addAll(fromRequest.getRespondTo());
		}

		if(NullChecker.isNotNullOrEmpty(fromRequest.getTemplateId()))
		{
			ret.getTemplateId().addAll(fromRequest.getTemplateId());
		}

		return ret;
	}

	private MCCIMT000100UV01Receiver createNewReceiver(String homeCommunityId)
	{
		return HL7ReceiverTransforms.createMCCIMT000100UV01Receiver(homeCommunityId);
	}

	private void processResult(PRPAIN201306UV02 result, II localPatientId, String homeCommunityId, AssertionType assertion)
	{
		if(result.getControlActProcess() != null && !NullChecker.isNullOrEmpty(result.getControlActProcess().getSubject()))
		{
			for(Iterator<PRPAIN201306UV02MFMIMT700711UV01Subject1> it = result.getControlActProcess().getSubject().iterator(); it.hasNext();)
			{
				PRPAIN201306UV02MFMIMT700711UV01Subject1 subject = it.next();
				if(!(subject.getRegistrationEvent() != null && subject.getRegistrationEvent().getSubject1() != null && subject.getRegistrationEvent().getSubject1().getPatient() != null && processPatient(subject.getRegistrationEvent().getSubject1().getPatient(), localPatientId, homeCommunityId, assertion)))
				{
					it.remove();
					logger.warn("Could NOT processResult.  Failed while looping through the PRPAIN201306UV02 result's subject.");
				}
			}
		}
		else
		{
			logger.warn("Could NOT processResult due to null check failures on PRPAIN201306UV02 result.");
		}
	}

	private boolean processPatient(PRPAMT201310UV02Patient prpamt201310UV02Patient, II localPatientId, String remoteHomeCommunityId, AssertionType assertion)
	{
		boolean ret = false;
		Map searchResultsMap = findMatch(prpamt201310UV02Patient, remoteHomeCommunityId, assertion);
		PRPAIN201306UV02 mviResults = null;

		if((searchResultsMap != null && (mviResults = (PRPAIN201306UV02) searchResultsMap.get("PRPAIN201306UV02")) != null) && doIDsMatch(localPatientId, HL7Parser201306.extractPatientIds(mviResults)))
		{
			ret = true;
			logger.info("Found a match for localPatientId: '" + localPatientId.getExtension() + "'");

			/*
			 * TODO complete ths code so we can check if this remote patient ID
			 * is already correlated....so we don't make unneeded 1301 MVI call.
			 */

			/*
			 * boolean correlationAlreadyExists =
			 * HL7Parser201306.extractMatchingId(mviResults,
			 * remotePatientIdRequest.getExtension(), "NI",
			 * sendingFacilityNumber, sendingHCID) != null;
			 * 
			 * if (correlationAlreadyExists) { logger.log(Level.INFO,
			 * "Remote Patient ID: ''{0}'' " +
			 * "is already correlated in MVI for HCID: ''{1}''.", new
			 * Object[]{remotePatientIdRequest, sendingHCID}); } //even if the
			 * remote Patient ID is already correlated we will call
			 * addPatientCorrelation to enter Audit log entry.
			 */
			Patient searchResultPatient = (Patient) searchResultsMap.get("SEARCH_RESULT_PATIENT");
			storeCorrelation(prpamt201310UV02Patient, localPatientId, remoteHomeCommunityId, assertion, searchResultPatient);
		}
		else
		{
			ret = false;

			// CCR 177986
			logger.warn("Did NOT find a match for localPatientId {}: ", localPatientId.getExtension());
		}

		return ret;
	}

	@SuppressWarnings("rawtypes")
	private Map findMatch(PRPAMT201310UV02Patient prpamt201310UV02Patient, String remoteHomeCommunityId, AssertionType assertion)
	{
		PRPAMT201301UV02Patient prpamt201301UV02Patient = HL7PatientTransforms.createPRPAMT201301UVPatient(prpamt201310UV02Patient);

		String localHomeCommunityId = getHomeCommunityId();
		PRPAIN201305UV02 prpain201305UV02 = HL7PRPA201305Transforms.createPRPA201305(prpamt201301UV02Patient, remoteHomeCommunityId, localHomeCommunityId, localHomeCommunityId);
		RespondingGatewayPRPAIN201305UV02RequestType request = new RespondingGatewayPRPAIN201305UV02RequestType();
		request.setPRPAIN201305UV02(prpain201305UV02);
		request.setAssertion(assertion);
		return adapterMvi.findCandidatesMap(request);
	}

	private String getHomeCommunityId()
	{
		String ret = null;
		Facility facility = facilityManager.getFacilityByFacilityNumber("VA");
		if(facility != null)
		{
			ret = facility.getHomeCommunityId();
		}
		return ret;
	}

	private boolean doIDsMatch(II localPatientId, List<II> ids)
	{
		boolean ret = false;
		logger.debug("doIDsMatch - localPatientId.getRoot(): " + localPatientId.getRoot());
		logger.debug("doIDsMatch - localPatientId.getExtension(): " + localPatientId.getExtension());

		for(II id : ids)
		{
			String icnValue = HL7Parser201306.extractICNValue(id.getExtension());
			logger.debug("id root: ''{}''; id extension: ''{}'';  id/icn value: ''{}''", id.getRoot(), id.getExtension(), icnValue); // CCR
																																		// 177986
			// removed id.root(i.e. the OID) equals comparison since for ICNs
			// it's always the VA's OID. And this was an issue since in each of
			// the Exchange environment
			// the VA OID's have a ".1" or ".2" etc. added to distinguish
			// environments for the connect gateway purposes.
			if(// localPatientId.getRoot().equals(id.getRoot())&&
			localPatientId.getExtension().equals(icnValue))
			{
				// CCR 177986
				logger.info("Found ID Match for ICN {}: ", icnValue);

				ret = true;
				break;
			}
		}

		if(!ret)
		{
			// CCR 177986
			logger.warn("Did not find ID Match for ICN {}: ", localPatientId.getExtension());
		}

		return ret;
	}

	private void storeCorrelation(PRPAMT201310UV02Patient prpamt201310UV02Patient, II localPatientId, String remoteHomeCommunityId, AssertionType assertion, Patient searchResultPatient)
	{
		PRPAMT201301UV02Patient prpamt201301UV02Patient = HL7PatientTransforms.createPRPAMT201301UVPatient(prpamt201310UV02Patient);
		String localHomeCommunityId = getHomeCommunityId();
		PRPAIN201301UV02 prpain201301UV02 = HL7PRPA201301Transforms.createPRPA201301(prpamt201301UV02Patient, localHomeCommunityId, remoteHomeCommunityId, localHomeCommunityId);
		prpain201301UV02.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId().add(localPatientId);
		AddPatientCorrelationRequestType request = new AddPatientCorrelationRequestType();
		request.setPRPAIN201301UV02(prpain201301UV02);
		request.setAssertion(assertion);
		// TODO the boolean s/b a flag to indicate whether or not the
		// correlation already exists so that we can avoid an
		// uncecessary 1301 call to MVI. See AdapterPatientDiscovery for example
		// of its use.
		adapterPatientCorrelation.addPatientCorrelation(request, searchResultPatient, false);
	}

	private void addResultToResponse(RespondingGatewayPRPAIN201306UV02ResponseType response, PRPAIN201306UV02 result)
	{
		CommunityPRPAIN201306UV02ResponseType communityResponse = new CommunityPRPAIN201306UV02ResponseType();
		communityResponse.setPRPAIN201306UV02(result);
		response.getCommunityResponse().add(communityResponse);
	}

	private PRPAIN201306UV02 createErrorResponse(PRPAIN201305UV02 prpain201305UV02, String homeCommunityId, String assigningAuthorityId)
	{
		PRPAIN201306UV02 ret = HL7DbParser201306.BuildMessageFromMpiPatients(null, prpain201305UV02, homeCommunityId, assigningAuthorityId);
		ret.getControlActProcess().getQueryAck().getQueryResponseCode().setCode("QE");
		return ret;
	}
}
