package gov.va.med.nhin.adapter.terminology;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.annotation.Resource;
import javax.ejb.Stateless;
import javax.jws.WebService;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author David Vazquez
 */
@WebService(serviceName = "TerminologyService", wsdlLocation = "META-INF/wsdl/Terminology.wsdl", portName = "TerminologyPort", endpointInterface = "gov.va.med.nhin.adapter.terminology.TerminologyPortType", targetNamespace = "urn:gov:va:med:nhin:adapter:terminology")
@Stateless(name = "Terminology", mappedName = "Terminology")
public class TerminologyBean implements TerminologyLocal, TerminologyRemote, TerminologyPortType
{
	private static final Logger logger = LoggerFactory.getLogger(TerminologyBean.class.getName());

	@Resource(name = "jdbc/NHIN")
	private DataSource dataSource;

	@PersistenceContext
	private EntityManager entityManager;

	public String lookup(String codeSet, String code, String field)
	{
		String ret = "";
		Connection connection = null;

		try
		{
			connection = getConnection();
			Codeset codeset = getCodeset(codeSet);
			if(codeset != null)
			{
				ret = getValue(connection, codeset, code, field);
			}
		}
		catch(SQLException se)
		{
			throw new RuntimeException("A database error occurred.", se);
		}
		finally
		{
			closeConnection(connection);
		}

		return ret;
	}

	public String translate(String sourceCodeset, String code, String targetCodeset, String field)
	{
		String ret = "";
		Connection connection = null;

		try
		{
			connection = getConnection();
			Translation translation = getTranslation(sourceCodeset, code, targetCodeset);
			if(translation != null)
			{
				Codeset codeset = getCodeset(translation.getTargetCodeset());
				if(codeset != null)
				{
					ret = getValue(connection, codeset, translation.getTargetCode(), field);
				}
			}
		}
		catch(SQLException se)
		{
			throw new RuntimeException("A database error occurred.", se);
		}
		finally
		{
			closeConnection(connection);
		}

		return ret;
	}

	private Connection getConnection() throws SQLException
	{
		return dataSource.getConnection();
	}

	private void closeConnection(Connection connection)
	{
		try
		{
			if(connection != null)
			{
				connection.close();
			}
		}
		catch(SQLException se)
		{
			logger.error("SQL Error occurred in closeConnection(): ", se);
		}
	}

	private Codeset getCodeset(String codeset)
	{
		Codeset ret = null;
		Query query = entityManager.createNamedQuery("Codeset.findByCodeset");
		query.setParameter("codeset", codeset);
		
		try
		{
			ret = (Codeset) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			logger.error("Error occurred in getCodeset(): ", nre);
		}
		
		return ret;
	}

	private String getValue(Connection connection, Codeset codeset, String code, String field) throws SQLException
	{
		DatabaseMetaData meta = connection.getMetaData();
		String tablName = codeset.getLookupTable();
		ResultSet columns = meta.getColumns(null, null, tablName, null);

		String columnNameWhere = null;
		String keyField = null;
		while(columns.next())
		{
			if(columns.getString("COLUMN_NAME").equals(codeset.getKeyField()))
			{
				columnNameWhere = columns.getString("COLUMN_NAME");

			}
			if(columns.getString("COLUMN_NAME").equals(field))
			{
				keyField = columns.getString("COLUMN_NAME");
			}
		}
		String ret = null;

		if(columnNameWhere != null && keyField != null)
		{
			// String sql = "select " + field + " from " +
			// codeset.getLookupTable() + " where " + codeset.getKeyField() + "=
			// ?";
			String sql = "select " + keyField + " from " + codeset.getLookupTable() + " where " + columnNameWhere + "= ?";
			PreparedStatement statement = connection.prepareStatement(sql);
			statement.setString(1, code);
			ResultSet rs = statement.executeQuery();
			if(rs.next())
			{
				ret = rs.getString(1);
			}
		}
		
		logger.debug("value {}", ret);

		return ret;
	}

	private Translation getTranslation(String sourceCodeset, String sourceCode, String targetCodeset)
	{
		Translation ret = null;
		Query query = entityManager.createNamedQuery("Translation.findTranslation");
		query.setParameter("sourceCodeset", sourceCodeset);
		query.setParameter("sourceCode", sourceCode);
		query.setParameter("targetCodeset", targetCodeset);
		
		try
		{
			ret = (Translation) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			logger.error("Error occurred in getTranslation(): ", nre);
		}
		
		return ret;
	}
}
