package gov.va.med.nhin.adapter.logging;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.va.med.nhin.adapter.utils.NullChecker;
import oasis.names.tc.xacml._2_0.context.schema.os.DecisionType;
import oasis.names.tc.xacml._2_0.context.schema.os.RequestType;

/**
 * @author serverchmarc
 * 
 */
public final class CheckPolicy
{
	private static Logger logger = LoggerFactory.getLogger(CheckPolicy.class.getName());

	public static enum AttrType
	{
		RESOURCE,
		SUBJECT,
		ACTION;
	}

	public static enum PolicyAttrs
	{
		XDS_POLICY_CHECK_ASSIGNING_AUTHORITY(
				"Assigning Authority",
				"urn:gov:hhs:fha:nhinc:assigning-authority-id",
				AttrType.RESOURCE),
		XDS_POLICY_CHECK_RESOURCE_ID(
				"Policy Check Resource",
				"urn:oasis:names:tc:xacml:1.0:resource:resource-id",
				AttrType.RESOURCE),
		SUBJECT_STRUCTURED_ROLE_NS(
				"Structured Role",
				"urn:oasis:names:tc:xacml:2.0:subject:role",
				AttrType.SUBJECT),
		SUBJECT_FUNCTIONAL_ROLE_NS(
				"Functional Role",
				"urn:oasis:names:tc:xspa:1.0:subject:functional_role",
				AttrType.SUBJECT),
		ACTION_ATTRIBUTE_ID(
				"Action ID",
				"urn:oasis:names:tc:xacml:1.0:action:action-id",
				AttrType.ACTION),
		SUBJECT_HOME_COMMUNITY_ATTRIBUTE_ID(
				"Subject HCID",
				"urn:gov:hhs:fha:nhinc:home-community-id",
				AttrType.SUBJECT),
		SUBJECT_PURPOSE_FOR_USE_ATTRIBUTE_ID(
				"Subject purpose of use",
				"urn:gov:hhs:fha:nhinc:purpose-for-use",
				AttrType.SUBJECT),
		SUBJECT_SUBJECT_ID_ATTRIBUTE_ID(
				"Subject ID",
				"urn:oasis:names:tc:xacml:1.0:subject:subject-id",
				AttrType.SUBJECT),
		RESOURCE_ATTRIBUTE_ID(
				"Resource attr ID",
				"urn:oasis:names:tc:xacml:1.0:resource:resource-id",
				AttrType.RESOURCE),
		RESOURCE_DOCUMENT_ATTRIBUTE_ID(
				"Resource document attr ID",
				"urn:gov:hhs:fha:nhinc:document-id",
				AttrType.RESOURCE),
		RESOURCE_HOME_COMMUNITY_ATTRIBUTE_ID(
				"Resource HCID",
				"urn:gov:hhs:fha:nhinc:home-community-id",
				AttrType.RESOURCE),
		RESOURCE_SUBJECT_ID_ATTRIBUTE_ID(
				"Resource Subject ID attr ID",
				"urn:oasis:names:tc:xacml:1.0:subject:subject-id",
				AttrType.RESOURCE);

		private PolicyAttrs(final String name, final String value, final AttrType type)
		{
			this.name = name;
			this.value = value;
			this.type = type;
		}

		private String name;
		private String value;
		private AttrType type;

		public String getName()
		{
			return name;
		}

		public String getValue()
		{
			return value;
		}

		public AttrType getType()
		{
			return type;
		}

		public String toString()
		{
			return name;
		}
	}

	public static String checkPolicyError(final CheckPolicyRequestType requst, final CheckPolicyResponseType response)
	{
		StringBuilder sb = new StringBuilder();
		RequestType requestType = requst.getRequest();

		sb.append("Transaction Error: Check Policy Returned false.\n");

		for(PolicyAttrs attr : PolicyAttrs.values())
		{
			String attrValue;
			switch(attr.getType())
			{
			case ACTION:
				attrValue = LoggingUtils.extractValueFromAction(requestType.getAction(), attr.getValue());
				break;
			case SUBJECT:
				attrValue = LoggingUtils.extractValueFromSubject(requestType.getSubject(), attr.getValue());
				break;
			case RESOURCE:
				attrValue = LoggingUtils.extractValueFromResource(requestType.getResource(), attr.getValue());
				break;
			default:
				attrValue = null;
				break;
			}
			if(StringUtils.isNotEmpty(attrValue))
			{
				sb.append("\t").append(attr.getName()).append(":").append(attrValue).append("\n");
			}
		}
		sb.append("\tReturned: ").append(response.getResponse().getResult().get(0).getDecision().toString());
		return sb.toString();
	}

	public static boolean checkPolicy(final CheckPolicyRequestType checkPolicyRequest, final CheckPolicyResponseType checkPolicyResponse)
	{
		boolean checkReturn = checkPolicyResponse != null && checkPolicyResponse.getResponse() != null && NullChecker.isNotNullOrEmpty(checkPolicyResponse.getResponse().getResult()) && checkPolicyResponse.getResponse().getResult().get(0).getDecision() == DecisionType.PERMIT;

		if(!checkReturn)
		{
			logger.debug("CheckPolicy Error {}", CheckPolicy.checkPolicyError(checkPolicyRequest, checkPolicyResponse));
		}
		return checkReturn;
	}

	/**
	 * Will check to see if the first result has an obligation, and if that
	 * obligation matches the passed value.
	 * 
	 * @param checkPolicyResponse
	 * @param value
	 * @return
	 */
	public static boolean checkObligation(final CheckPolicyResponseType checkPolicyResponse, final String value)
	{
		boolean ret = checkPolicyResponse != null && value != null && checkPolicyResponse.getResponse() != null && NullChecker.isNotNullOrEmpty(checkPolicyResponse.getResponse().getResult()) && checkPolicyResponse.getResponse().getResult().get(0).getObligations() != null && NullChecker.isNotNullOrEmpty(checkPolicyResponse.getResponse().getResult().get(0).getObligations().getObligation()) && checkPolicyResponse.getResponse().getResult().get(0).getObligations().getObligation().get(0).getObligationId() != null && StringUtils.equalsIgnoreCase(checkPolicyResponse.getResponse().getResult().get(0).getObligations().getObligation().get(0).getObligationId(), value);
		logger.trace("Check Obligation returned {} for value: {} ", ret, value);
		return ret;
	}
}
