package gov.va.med.nhin.adapter.facilitymanager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 * DAO for the FACILITIES table.
 * 
 * @author David Vazquez
 */
@Stateless(name = "FacilityManager", mappedName = "FacilityManager")
public class FacilityManagerBean implements FacilityManagerRemote, FacilityManagerLocal
{
	private static final Logger logger = LoggerFactory.getLogger(FacilityManagerBean.class.getName());

	private EntityManager entityManager;

	@PersistenceContext
	public void setEntityManager(EntityManager entityManager)
	{
		this.entityManager = entityManager;
	}

	public List<Facility> getAllFacilities()
	{
		Query query = entityManager.createNamedQuery("Facility.findAll");
		
		return query.getResultList();
	}

	public Facility getMyFacility()
	{
		Query query = entityManager.createNamedQuery("Facility.findByFacilityNumber");
		query.setParameter("facilityNumber", "VA");
		return getSingleFacilityResult(query);
	}

	public List<Facility> getAllFacilities(String[] excludeFacilityNumbers)
	{
		StringBuilder whereClause = new StringBuilder();
		Map<String, Object> setParams = new HashMap<String, Object>();

		if(!NullChecker.isNullOrEmpty(excludeFacilityNumbers))
		{
			whereClause.append(" WHERE f.facilityNumber NOT IN (:facility0");
			setParams.put("facility0", excludeFacilityNumbers[0]);

			for(int i = 1; i < excludeFacilityNumbers.length; ++i)
			{
				whereClause.append(",:facility").append(i);
				setParams.put("facility" + i, excludeFacilityNumbers[i]);
			}

			whereClause.append(")");
		}

		Query query = entityManager.createQuery("SELECT f from Facility f" + whereClause.toString());

		for(Map.Entry<String, Object> e : setParams.entrySet())
		{
			query.setParameter(e.getKey(), e.getValue());
		}

		return query.getResultList();
	}

	public Facility getFacilityByFacilityNumber(String facilityNumber)
	{
		Query query = entityManager.createNamedQuery("Facility.findByFacilityNumber");
		query.setParameter("facilityNumber", facilityNumber);
		return getSingleFacilityResult(query);
	}

	public Facility getFacilityByHomeCommunityId(String oid)
	{
		Query query = entityManager.createNamedQuery("Facility.findByHomeCommunityId");
		query.setParameter("homeCommunityId", oid);
		return getSingleFacilityResult(query);
	}

	public Facility getFacilityByFullHomeCommunityId(String fullHomeCommunityId)
	{
		Query query = entityManager.createNamedQuery("Facility.findByFullHomeCommunityId");
		query.setParameter("fullHomeCommunityId", fullHomeCommunityId);
		return getSingleFacilityResult(query);
	}

	public List<Facility> getFacilitiesContainingAssigningAuthority(String assigningAuthorityOID)
	{
		logger.info("In FacilityManagerBean::getFacilitiesContainingAssigningAuthority");
		List<Facility> ret = new ArrayList<Facility>();
		Query query = entityManager.createNamedQuery("AssigningAuthority.findByAssigningAuthorityOid");
		query.setParameter("assigningAuthorityOid", assigningAuthorityOID);
		AssigningAuthority result = getSingleAAResult(query);
		if(result != null)
		{
			for(FacilityToAA f2aa : result.getFacilitiesToAAs())
			{
				ret.add(f2aa.getFacility());
				logger.info("Facility {} added to return", f2aa.getFacility().getFacilityNumber()); //CCR 177986
			}
		}
		else
		{
			logger.info("AssiningAuthority query result is null");
		}
		return ret;
	}

	public String getPrefferedQueryName(final String loinc, List<FacSupDoctypes> adapterFacSupDoctypesList, List<FacSupDoctypes> partnerFacSupDoctypesList)
	{
		String partnerSupportedDocSpecType = null;
		String docSpecTypeToBeGenerated = null;

		for(FacSupDoctypes partnerFSD : partnerFacSupDoctypesList)
		{
			// Make sure that we are testing against the correct LOINC.
			if(StringUtils.equals(loinc, partnerFSD.getLoincCode()))
			{
				partnerSupportedDocSpecType = partnerFSD.getQueryName();

				for(FacSupDoctypes myFSD : adapterFacSupDoctypesList)
				{

					if(StringUtils.equals(loinc, partnerFSD.getLoincCode()) && myFSD.getQueryName().equalsIgnoreCase(partnerSupportedDocSpecType) && partnerFSD.getIsPrefType().equalsIgnoreCase("Y"))
					{
						return partnerSupportedDocSpecType;
					}
					else if(myFSD.getQueryName().equalsIgnoreCase(partnerSupportedDocSpecType))
					{
						docSpecTypeToBeGenerated = myFSD.getQueryName();
					}
				}
			}
		}

		return docSpecTypeToBeGenerated;
	}

	@SuppressWarnings("unchecked")
	public String getDocumentTypeToBeGenerated(String loincode, String partnerHomeCommunityId)
	{
		Facility myFacility = null;
		List<FacSupDoctypes> partnerFacSupDoctypesList = null;
		List<FacSupDoctypes> myFacSupDoctypesList = null;

		String myHomeCommunityId = null;

		myFacility = this.getMyFacility();
		myHomeCommunityId = myFacility.getHomeCommunityId();

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci AND fsd.loinCd = :code");
		query.setParameter("hci", partnerHomeCommunityId);
		query.setParameter("code", loincode);
		partnerFacSupDoctypesList = query.getResultList();

		query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci AND fsd.loinCd = :code");
		query.setParameter("hci", myHomeCommunityId);
		query.setParameter("code", loincode);
		myFacSupDoctypesList = query.getResultList();

		return getPrefferedQueryName(loincode, myFacSupDoctypesList, partnerFacSupDoctypesList);
	}

	/**
	 * Get all possible queryNames for a particular Home Community.
	 * 
	 * @param partnerHomeCommunityId
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public List<String> getQueryNamesToGenerateDocuments(final String partnerHomeCommunityId)
	{
		Facility myFacility = null;
		List<FacSupDoctypes> partnerFacSupDoctypesList = null;
		List<FacSupDoctypes> myFacSupDoctypesList = null;

		List<String> commonQueryNames = new ArrayList<String>();

		String myHomeCommunityId = null;

		myFacility = this.getMyFacility();
		myHomeCommunityId = myFacility.getHomeCommunityId();

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci");
		query.setParameter("hci", normalizeHCID(partnerHomeCommunityId));
		partnerFacSupDoctypesList = query.getResultList();

		query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci");
		query.setParameter("hci", myHomeCommunityId);
		myFacSupDoctypesList = query.getResultList();

		// For all of the LOINCs we support find the preferred queryName.
		for(FacSupDoctypes adapterFSD : myFacSupDoctypesList)
		{
			String queryName = getPrefferedQueryName(adapterFSD.getLoincCode(), myFacSupDoctypesList, partnerFacSupDoctypesList);
			if(StringUtils.isNotBlank(queryName))
			{
				commonQueryNames.add(queryName);
			}
		}

		return commonQueryNames;
	}

	@SuppressWarnings("unchecked")
	public List<String> getQueryNamesToGenerateDocuments(String partnerHomeCommunityId, List<String> loinCodeListIn)
	{
		List<String> validatedLoinCdList = new ArrayList<String>();

		List<String> parsedLoincs = listFromQueryParam(loinCodeListIn);

		Facility myFacility = this.getMyFacility();
		String myHomeCommunityId = normalizeHCID(myFacility.getHomeCommunityId());
		myFacility = this.getMyFacility();

		Query query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci");
		query.setParameter("hci", normalizeHCID(partnerHomeCommunityId));
		List<FacSupDoctypes> partnerFacSupDoctypesList = query.getResultList();

		query = entityManager.createQuery("SELECT fsd FROM FacSupDoctypes fsd WHERE fsd.homeCommunityId = :hci");
		query.setParameter("hci", myHomeCommunityId);
		List<FacSupDoctypes> myFacSupDoctypesList = query.getResultList();

		// Start with the list that came in the request.
		for(String loinCodeIn : parsedLoincs)
		{
			String queryName = getPrefferedQueryName(loinCodeIn, myFacSupDoctypesList, partnerFacSupDoctypesList);
			if(StringUtils.isNotBlank(queryName))
			{
				validatedLoinCdList.add(queryName);
			}
		}
		return validatedLoinCdList;
	}

	public void addAssigningAuthorityToFacility(String homeCommunityId, String assigningAuthorityOid, String assigningAuthorityName)
	{
		logger.debug("addAssigningAuthorityToFacility() invoked");

		Query query = entityManager.createNamedQuery("Facility.findByHomeCommunityId");
		query.setParameter("homeCommunityId", homeCommunityId);
		Facility facility = getSingleFacilityResult(query);

		if(facility != null)
		{
			FacilityToAA facilityToAA = null;
			if(!NullChecker.isNullOrEmpty(facility.getFacilitiesToAAs()))
			{
				for(FacilityToAA f2aa : facility.getFacilitiesToAAs())
				{
					if(f2aa.getAssigningAuthority().getAssigningAuthorityOid().equals(assigningAuthorityOid))
					{
						facilityToAA = f2aa;
						break;
					}
				}
			}

			if(facilityToAA == null)
			{
				Query aaQuery = entityManager.createNamedQuery("AssigningAuthority.findByAssigningAuthorityOid");
				aaQuery.setParameter("assigningAuthorityOid", assigningAuthorityOid);
				AssigningAuthority assigningAuthority = getSingleAAResult(aaQuery);

				if(assigningAuthority == null)
				{
					assigningAuthority = new AssigningAuthority();
					assigningAuthority.setAssigningAuthorityOid(assigningAuthorityOid);
					assigningAuthority.setAssigningAuthorityName(assigningAuthorityName);
					assigningAuthority.setFacilitiesToAAs(new ArrayList<FacilityToAA>());
					entityManager.persist(assigningAuthority);
				}

				facilityToAA = new FacilityToAA();
				facilityToAA.setAssigningAuthority(assigningAuthority);
				facilityToAA.setFacility(facility);
				facility.getFacilitiesToAAs().add(facilityToAA);
				assigningAuthority.getFacilitiesToAAs().add(facilityToAA);
				entityManager.persist(facilityToAA);
			}
		}

		logger.debug("addAssigningAuthorityToFacility() exited");
	}

	private Facility getSingleFacilityResult(Query query)
	{
		Facility ret;

		try
		{
			ret = (Facility) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			ret = null;
		}

		return ret;
	}

	private AssigningAuthority getSingleAAResult(Query query)
	{
		AssigningAuthority ret;

		try
		{
			ret = (AssigningAuthority) query.getSingleResult();
		}
		catch(NoResultException nre)
		{
			ret = null;
		}

		return ret;
	}

	/**
	 * Incoming queryParameters can be of type list with a single element. Or
	 * can be a list of values depending on spec version. Need to normalize the
	 * data into a list of LOINCS.
	 * 
	 * @param incoming
	 * @return
	 */
	private List<String> listFromQueryParam(final List<String> incoming)
	{
		List<String> loincs = new ArrayList<String>();

		if(CollectionUtils.isEmpty(incoming))
		{
			return Collections.emptyList();
		}

		for(String value : incoming)
		{
			String[] values = parseOutAllQueryParam(value);

			for(String single : values)
			{
				loincs.add(extractLoinc(single));
			}
		}

		return loincs;
	}

	private String[] parseOutAllQueryParam(final String value)
	{
		String val = parseSingleQueryParam(value);
		String[] parsed = StringUtils.split(val, ",");
		return parsed;
	}

	private String parseSingleQueryParam(final String value)
	{
		String parsed = value;
		// Remove leading paren.
		parsed = StringUtils.remove(parsed, "(");
		// Remove trailing paren.
		parsed = StringUtils.remove(parsed, ")");
		// Remove single qoutes.
		parsed = StringUtils.remove(parsed, "'");
		return parsed;
	}

	/**
	 * Will return just the LOINC from an inbound request. Some values come in
	 * as just LOINC some come in as LOINC^^TEMPLATID.
	 * 
	 * @param value
	 * @return
	 */
	private String extractLoinc(final String value)
	{
		if(!StringUtils.contains(value, "^^"))
		{
			// Did not find a delimiter return value.
			return value;
		}

		// There is a delimiter need to parse out the LOINC.
		String[] split = value.split("\\^\\^");
		return split[0];

	}

	private String normalizeHCID(final String hcid)
	{
		return StringUtils.remove(hcid, "urn:oid:");
	}

	/**
	 * TODO
	 * @param hcid
	 * @param operation
	 * @return true if
	 */
	public boolean isPartnerAllowed(final String hcid, final OperationOnOff operation)
	{
		@SuppressWarnings("rawtypes")
		List results;

		if(StringUtils.isBlank(hcid) || operation == null)
		{
			return false;
		}
		
		String fullHCID = hcid;
		
		if(!fullHCID.startsWith("urn:oid:"))
		{
			fullHCID = "urn:oid:" + fullHCID;
		}
		
		StringBuilder querySb = new StringBuilder();
		querySb.append("SELECT f, fo from Facility as f, Operations as fo ");
		querySb.append("WHERE f.fullHomeCommunityId = :fullHomeCommunityId AND f.facilityId = fo.id ");
		querySb.append("AND fo.");
		querySb.append(operation.getParam());
		querySb.append(" = :opValue");
		
		Query query = entityManager.createQuery(querySb.toString());
		query.setParameter("fullHomeCommunityId", fullHCID);
		query.setParameter("opValue", "Y");
		results = query.getResultList();

		if(CollectionUtils.isNotEmpty(results) && results.size() > 0)
		{
			return true;
		}

		return false;
	}
}
