package gov.va.med.nhin.adapter.datamanager.translators;

import java.net.URL;
import java.util.List;
import java.util.Properties;

import javax.naming.InitialContext;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.cache.CacheManager;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;
import gov.va.med.nhin.adapter.utils.config.PropertiesType;
import gov.va.med.nhin.adapter.utils.config.PropertyTypeUtils;
import gov.va.med.sts.webservice.ct.ConceptDetailTransfer;
import gov.va.med.sts.webservice.ct.CtService;
import gov.va.med.sts.webservice.ct.DesignationDetailTransfer;
import gov.va.med.sts.webservice.ct.PropertyTransfer;
import gov.va.med.sts.webservice.ct.RelationshipTransfer;
import gov.va.med.sts.webservice.ct.Sts;

/**
 *
 * @author David Vazquez
 */
public class STSLookupDataTranslator implements DataTranslator
{
	private static final Logger logger = LoggerFactory.getLogger(STSLookupDataTranslator.class.getName());

	static private final String DesignationsConstant = "Designations";
	static private final String PropertiesConstant = "Properties";
	static private final String AssociationsConstant = "Associations";

	private CtService ctService;
	private InitialContext context;

	public Object translate(Object input, Object result, Reference translation, DataQuery dataQuery)
	{
		logger.debug("translate() invoked", "translate");

		Object ret = "";

		try
		{
			if(NullChecker.isNotNullOrEmpty(input))
			{
				Long codeSystemVuid = new Long(translation.getProperty("CodeSystemVuid"));
				String versionName = translation.getProperty("VersionName");
				String field = translation.getProperty("field");
				String collection = translation.getProperty("Collection");
				boolean bypassCache = translation.getProperty("bypassCache") != null;
				String key = codeSystemVuid.toString() + versionName + input.toString();
				String connectionFilename = translation.getProperty("connectionFilename");
				String connection = translation.getProperty("connection");

				List<DesignationDetailTransfer> designations = null;
				List<PropertyTransfer> properties = null;
				List<RelationshipTransfer> associations = null;

				ConceptDetailTransfer detail = null;

				if(!bypassCache)
				{
					detail = (ConceptDetailTransfer) getCacheManager(translation).get(key);
				}

				if(detail == null)
				{
					Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
					Properties connectionProperties = (Properties) propertiesCollection.get(connection);

					Sts stsPort = getPort(connectionProperties.getProperty("wsdlURL"));

					detail = stsPort.returnConceptDetails(codeSystemVuid, versionName, (String) input);

					if(!bypassCache)
					{
						getCacheManager(translation).put(key, detail);
					}
				}

				if(collection.equalsIgnoreCase(DesignationsConstant))
				{
					if(NullChecker.isNotNullOrEmpty(detail.getDesignations()) && NullChecker.isNotNullOrEmpty(detail.getDesignations().getDesignation()))
					{
						designations = detail.getDesignations().getDesignation();
						for(DesignationDetailTransfer designation : designations)
						{
							if(designation.getType().equalsIgnoreCase(field))
							{
								ret = designation.getName();
							}
						}
					}
					else
					{
						throw new DataManagerException("An error occured performing lookup.  Configuration issue exists: Designations specified but empty.");
					}
				}
				else if(collection.equalsIgnoreCase(PropertiesConstant))
				{
					if(NullChecker.isNotNullOrEmpty(detail.getProperties()) && NullChecker.isNotNullOrEmpty(detail.getProperties().getProperty()))
					{
						properties = detail.getProperties().getProperty();
						for(PropertyTransfer property : properties)
						{
							if(property.getType().equalsIgnoreCase(field))
							{
								ret = property.getName();
							}
						}
					}
					else
					{
						throw new DataManagerException("An error occured performing lookup.  Configuration issue exists: Properties specified but empty.");
					}
				}
				else if(collection.equalsIgnoreCase(AssociationsConstant))
				{
					if(NullChecker.isNotNullOrEmpty(detail.getAssociations()) && NullChecker.isNotNullOrEmpty(detail.getAssociations().getAssociation()))
					{
						associations = detail.getAssociations().getAssociation();
						for(RelationshipTransfer association : associations)
						{
							if(association.getType().equalsIgnoreCase(field))
							{
								ret = association.getName();
							}
						}
					}
					else
					{
						throw new DataManagerException("An error occured performing lookup.  Configuration issue exists: Associations specified but empty.");
					}
				}
			}
		}
		catch(Exception e)
		{
			throw new DataManagerException("An error occured performing lookup.", e);
		}

		return ret;
	}

	private synchronized Sts getPort(String CtServiceWSDL) throws java.lang.Exception
	{
		if(ctService == null)
		{
			ctService = new CtService(new URL(CtServiceWSDL), new QName("urn:gov:va:med:sts:webservice:ct", "ctService"));
		}

		return ctService.getStsPort();
	}

	private CacheManager getCacheManager(Reference translationType) throws java.lang.Exception
	{
		if(context == null)
		{
			context = getInitialContext(translationType);
		}

		CacheManager ret = (CacheManager) context.lookup("CacheManager#gov.va.med.nhin.adapter.cache.CacheManagerRemote");
		return ret;
	}

	private InitialContext getInitialContext(Reference translationType) throws java.lang.Exception
	{
		InitialContext ret;

		PropertiesType propertiesType = translationType.getProperties("initialContextProperties");
		if(propertiesType != null)
		{
			Properties properties = PropertyTypeUtils.toProperties(propertiesType);
			ret = new InitialContext(properties);
		}
		else
		{
			ret = new InitialContext();
		}

		return ret;
	}
}
