package gov.va.med.nhin.adapter.datamanager.translators;

import java.net.URL;
import java.util.List;
import java.util.Properties;

import javax.naming.InitialContext;
import javax.xml.namespace.QName;

import org.apache.commons.beanutils.PropertyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.cache.CacheManager;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.PropertiesCollectionFactory;
import gov.va.med.nhin.adapter.utils.config.PropertiesType;
import gov.va.med.nhin.adapter.utils.config.PropertyTypeUtils;
import gov.va.med.sts.webservice.ct.CtService;
import gov.va.med.sts.webservice.ct.MapEntryValueListTransfer;
import gov.va.med.sts.webservice.ct.MapEntryValueTransfer;
import gov.va.med.sts.webservice.ct.SourcesTransfer;
import gov.va.med.sts.webservice.ct.Sts;

/**
 *
 * @author David Vazquez
 */
public class STSDataTranslator implements DataTranslator
{
	private static final Logger logger = LoggerFactory.getLogger(STSDataTranslator.class.getName());

	private CtService ctService;
	private InitialContext context;

	public Object translate(Object input, Object result, Reference translationType, DataQuery dataQuery)
	{
		logger.debug("translate() invoked", "translate");

		Object ret = "";

		try
		{
			if(!NullChecker.isNullOrEmpty(input))
			{
				String inputAsString = (String) input;
				Long mapSetVuid = new Long(translationType.getProperty("MapSetVuid"));
				String mapSetVersionName = translationType.getProperty("MapSetVersionName");
				String field = translationType.getProperty("field");
				boolean bypassCache = translationType.getProperty("bypassCache") != null;
				String connectionFilename = translationType.getProperty("connectionFilename");
				String connection = translationType.getProperty("connection");

				int mapEntryOrder = 0;

				String targetDesignationTypeName;
				if(translationType.getProperty("TargetDesignationTypeName") == null)
				{
					targetDesignationTypeName = null;
				}
				else
				{
					targetDesignationTypeName = translationType.getProperty("TargetDesignationTypeName");
				}

				String sourceDesignationTypeName = null;
				if(translationType.getProperty("SourceDesignationTypeName") == null)
				{
					sourceDesignationTypeName = null;
				}
				else
				{
					sourceDesignationTypeName = translationType.getProperty("SourceDesignationTypeName");
				}

				// Specific to the blood pressure VUID (4500634)
				int delimiter = inputAsString.indexOf("~");
				if(delimiter != -1)
				{
					mapEntryOrder = Integer.parseInt(inputAsString.substring(delimiter + 1));
					inputAsString = inputAsString.substring(0, delimiter);
				}
				else
				{
					mapEntryOrder = 1;
				}

				if(mapEntryOrder != -1)
				{
					String key = mapSetVuid.toString() + mapSetVersionName + sourceDesignationTypeName + targetDesignationTypeName + inputAsString;
					MapEntryValueListTransfer mapEntryValueListTransfer = null;

					if(!bypassCache)
					{
						mapEntryValueListTransfer = (MapEntryValueListTransfer) getCacheManager(translationType).get(key);
					}

					if(mapEntryValueListTransfer == null)
					{
						Properties propertiesCollection = PropertiesCollectionFactory.getPropertiesCollection(connectionFilename);
						Properties connectionProperties = (Properties) propertiesCollection.get(connection);

						Sts stsPort = getPort(connectionProperties.getProperty("wsdlURL"));

						SourcesTransfer sources = new SourcesTransfer();
						sources.getSource().add(inputAsString);

						mapEntryValueListTransfer = new SerializableMapEntryValueListTransfer(stsPort.listMapEntriesFromSource(mapSetVuid, mapSetVersionName, sources, sourceDesignationTypeName, targetDesignationTypeName, null, null));

						if(!bypassCache)
						{
							getCacheManager(translationType).put(key, mapEntryValueListTransfer);
						}
					}

					if(!NullChecker.isNullOrEmpty(mapEntryValueListTransfer.getMapEntryValues()))
					{
						MapEntryValueTransfer entry = getEntry(mapEntryValueListTransfer.getMapEntryValues(), mapEntryOrder);
						if(entry != null)
						{
							ret = PropertyUtils.getProperty(entry, field);
						}
					}
				}
			}
		}
		catch(Exception e)
		{
			logger.error("Error occurred while translating. message={}", input.toString(), e.getMessage());
			throw new DataManagerException("An error occured when translating.", e);
		}

		return ret;
	}

	private synchronized Sts getPort(String CtServiceWSDL) throws java.lang.Exception
	{
		if(ctService == null)
		{
			ctService = new CtService(new URL(CtServiceWSDL), new QName("urn:gov:va:med:sts:webservice:ct", "ctService"));
		}

		return ctService.getStsPort();
	}

	private CacheManager getCacheManager(Reference translationType) throws java.lang.Exception
	{
		if(context == null)
		{
			context = getInitialContext(translationType);
		}

		CacheManager ret = (CacheManager) context.lookup("CacheManager#gov.va.med.nhin.adapter.cache.CacheManagerRemote");
		return ret;
	}

	private InitialContext getInitialContext(Reference translationType) throws java.lang.Exception
	{
		InitialContext ret;

		PropertiesType propertiesType = translationType.getProperties("initialContextProperties");
		if(propertiesType != null)
		{
			Properties properties = PropertyTypeUtils.toProperties(propertiesType);
			ret = new InitialContext(properties);
		}
		else
		{
			ret = new InitialContext();
		}

		return ret;
	}

	private MapEntryValueTransfer getEntry(List<MapEntryValueTransfer> entries, int mapEntryOrder)
	{
		MapEntryValueTransfer ret = null;

		for(MapEntryValueTransfer entry : entries)
		{
			if(entry.getMapEntryOrder() == mapEntryOrder)
			{
				ret = entry;
				break;
			}
		}

		return ret;
	}
}
