﻿using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.Integration.VIMT.Vista.Links;
using VA.TMP.Integration.VIMT.Vista.StateObject;
using VA.TMP.OptionSets;
using VIMT.VIASchedulingService.Messages;
using VA.TMP.Integration.VIMT.Messages.Vista;

namespace VA.TMP.Integration.VIMT.Vista.Mappers
{
    public static class CancelAppointmentMapper
    {
        #region CancelCode Helpers
        /// <summary>
        /// Maps the TMP Status Reason to the corresponding Vista Cancel Code
        /// </summary>
        /// <param name="tmpStatusReason"></param>
        /// <returns></returns>
        public static string MapCancelCodeToVista(int tmpStatusReason)
        {
            var cancelCode = string.Empty;
            switch (tmpStatusReason)
            {
                case 917290001: //Technology Failure
                case 917290008: //Scheduling Error
                case 917290000: //Clinic Canceled
                    cancelCode = "C";
                    break;
                case 9: //Patient Canceled
                    cancelCode = "PC";
                    break;
                case 10: //Patient No Show
                    cancelCode = "N";
                    break;
                default:
                    cancelCode = "C";
                    break;
            }
            return cancelCode;
        }

        /// <summary>
        /// Maps the TMP Status Reason to the corresponding Vista Cancel Reason
        /// </summary>
        /// <param name="tmpStatusReason"></param>
        /// <returns></returns>
        public static string MapCancelReasonToVista(int tmpStatusReason)
        {
            var cancelReason = string.Empty;
            switch (tmpStatusReason)
            {
                case 917290000: //Clinic Canceled
                    cancelReason = "13";
                    break;
                case 917290008: //Scheduling Error
                    cancelReason = "8";
                    break;
                case 917290001: //Technology Failure
                case 9: //Patient Canceled
                case 10: //Patient No Show
                    cancelReason = "11";
                    break;
                default: //fallback for unexpected or future statuses to be added
                    cancelReason = "11";
                    break;
            }
            return cancelReason;
        }

        public static VistaCancelReason MapCancelReasonToTmp(string vistaReason)
        {
            switch(vistaReason)
            {
                case "13":
                    return VistaCancelReason.CLINIC_CANCELLED;
                case "8":
                    return VistaCancelReason.SCHEDULING_CONFLICT_ERROR;
                case "11":
                    return VistaCancelReason.OTHER;
                default:
                    return VistaCancelReason.OTHER;
            }
        }
        
        public static VistaCancelCode MapCancelCodeToTmp(string vistaCode)
        {
            switch (vistaCode)
            {
                case "N":
                    return VistaCancelCode.NO_SHOW;
                case "C":
                    return VistaCancelCode.CANCELLED_BY_CLINIC;
                case "NA":
                    return VistaCancelCode.NO_SHOW_AND_AUTO_RE_BOOK;
                case "CA":
                    return VistaCancelCode.CANCELLED_BY_CLINIC_AND_AUTO_RE_BOOK;
                case "I":
                    return VistaCancelCode.INPATIENT_APPOINTMENT;
                case "PC":
                    return VistaCancelCode.CANCELLED_BY_PATIENT;
                case "PCA":
                    return VistaCancelCode.CANCELLED_BY_PATIENT_AND_AUTO_RE_BOOK;
                case "NT":
                    return VistaCancelCode.NO_ACTION_TAKEN;
                default:
                    return VistaCancelCode.CANCELLED_BY_CLINIC_AND_AUTO_RE_BOOK;
            }
        }

        public static serviceappointment_statuscode GetAppointmentStatus(CancelGroupAppointmentStateObject state)
        {
            int statusCodeValue;
            if (state.CancelGroupAppointmentRequest.WholeAppointmentCanceled)
                statusCodeValue = state.CrmAppointment.cvt_IntegrationBookingStatus.Value;
            else
                statusCodeValue = state.CancelGroupAppointmentRequest.CancelIndividualStatus;
            return (serviceappointment_statuscode)statusCodeValue;
        }

        public static serviceappointment_statuscode GetAppointmentStatus(CancelAppointmentStateObject state)
        {
            int statusCodeValue;
            if (state.CancelAppointmentRequest.WholeAppointmentCanceled)
                statusCodeValue = state.ServiceAppointment.StatusCode.Value;
            else
                statusCodeValue = state.CancelAppointmentRequest.IndividualCancelStatus;
            return (serviceappointment_statuscode)statusCodeValue;
        }
        #endregion

        #region Cancel Request Mappers
        public static VIMTVIASchedCancApcancelAppointmentRequest CancelAppointmentRequest(CancelGroupAppointmentStateObject state, Side side, log4net.ILog logger)
        {
            var facility = side == Side.Patient ? state.PatientFacility : state.ProviderFacility;
            var clinic = side == Side.Patient ? state.PatientClinic : state.ProviderClinic;
            var localDateTime = PipelineUtilities.ConvertUtcToLocal(facility, state.ServiceAppointment.ScheduledStart.Value);

            //5 minute offset for provider side when appt is clinic based.  Happens whether its inter or intrafacility - if this assumption is wrong, then also check "intraFacilityRequiresOffset" before adding 5 minutes
            //var intraFacilityRequiresOffset = state.PatFacility?.mcs_StationNumber == state.ProFacility?.mcs_StationNumber;
            if (side == Side.Provider && (state.AppointmentType == AppointmentType.CLINIC_BASED || state.AppointmentType == AppointmentType.GROUP))
                localDateTime = localDateTime.AddMinutes(5);

            var status = GetAppointmentStatus(state);

            var apptReqInfo = new VIMTVIASchedCancApReqappointment
            {
                mcs_clinicId = clinic.cvt_ien,
                mcs_appointmentLength = state.ServiceAppointment.ScheduledDurationMinutes.ToString(),
                mcs_purpose = state.AppointmentPurpose,
                mcs_timestamp = localDateTime.ToString("yyyyMMdd.HHmm").TrimEnd('0'),
                mcs_appointmentType = state.VistaAppointmentType,
                mcs_cancelCode = MapCancelCodeToVista((int)status),
                mcs_cancelReason = MapCancelReasonToVista((int)status),
                mcs_remarks = status.ToString()
            };

            return new VIMTVIASchedCancApcancelAppointmentRequest
            {
                UserId = state.UserId,
                OrganizationName = state.OrganizationName,
                VIMTVIASchedCancApReqappointmentInfo = apptReqInfo,
                VIMTVIASchedCancApReqqueryBeanInfo = VistaMapperHelper.GetQueryBean(state, side),
                Debug = true,
                LogSoap = true
            };
        }

        public static VIMTVIASchedCancApcancelAppointmentRequest CancelAppointmentRequest(CancelAppointmentStateObject state, Side side, log4net.ILog logger)
        {
            var facility = side == Side.Patient ? state.PatFacility : state.ProFacility;
            var clinic = side == Side.Patient ? state.PatClinic : state.ProClinic;
            var localDateTime = PipelineUtilities.ConvertUtcToLocal(facility, state.ServiceAppointment.ScheduledStart.Value);
            //TO DO - check assumption: 5 minute offset for provider side when appt is clinic based.  Happens whether its inter or intrafacility.  If this assumption is wrong, then also check "intraFacilityRequiresOffset"
            //var intraFacilityRequiresOffset = state.PatFacility?.mcs_StationNumber == state.ProFacility?.mcs_StationNumber;
            if (side == Side.Provider && (state.AppointmentType == AppointmentType.CLINIC_BASED || state.AppointmentType == AppointmentType.GROUP))
                localDateTime = localDateTime.AddMinutes(5);
            var status = GetAppointmentStatus(state);
            var apptReqInfo = new VIMTVIASchedCancApReqappointment
            {
                mcs_clinicId = clinic.cvt_ien,
                mcs_appointmentLength = state.ServiceAppointment.ScheduledDurationMinutes.ToString(),
                mcs_purpose = state.VistaAppointmentPurpose, 
                mcs_timestamp = localDateTime.ToString("yyyyMMdd.HHmm").TrimEnd('0'),
                mcs_appointmentType = state.VistaAppointmentType, 
                mcs_cancelCode = MapCancelCodeToVista((int)status),   
                mcs_cancelReason = MapCancelReasonToVista((int)status), 
                mcs_remarks = status.ToString()
            };

            return new VIMTVIASchedCancApcancelAppointmentRequest
            {
                UserId = state.UserId,
                OrganizationName = state.OrganizationName,
                VIMTVIASchedCancApReqappointmentInfo = apptReqInfo,
                VIMTVIASchedCancApReqqueryBeanInfo = VistaMapperHelper.GetQueryBean(state, side),
                Debug = true,
                LogSoap = true
            };
        }
        #endregion
    }
}
