﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using VA.TMP.DataModel;
using VIMT.VideoVisit.Messages;

namespace VA.TMP.Integration.VIMT.VideoVisit.Mappers
{
    /// <summary>
    /// Class to map a Service Appointment to VVS Appointment.
    /// </summary>
    internal class VideoVisitMapper
    {
        private readonly IOrganizationService _organizationService;
        private readonly ServiceAppointment _serviceAppointment;
        private readonly List<Guid> _contacts;
        private readonly List<SystemUser> _systemUser;
        private readonly Appointment _appointment;
        private readonly bool _isGroup;

        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="serviceAppointment">CRM Service Appointment.</param>
        /// <param name="contacts">CRM Contacts.</param>
        /// <param name="systemUsers">CRM Sysem Users.</param>
        /// <param name="isGroup">Is group appointment.</param>
        /// <param name="appointment">Optional appointment.</param>
        public VideoVisitMapper(IOrganizationService organizationService, ServiceAppointment serviceAppointment, List<Guid> contacts, List<SystemUser> systemUsers, bool isGroup, Appointment appointment = null)
        {
            _organizationService = organizationService;
            _serviceAppointment = serviceAppointment;
            _contacts = contacts;
            _systemUser = systemUsers;
            _appointment = appointment;
            _isGroup = isGroup;
        }

        /// <summary>
        /// Map a schema Apppointment Initiation data to TMP entities.
        /// </summary>
        /// <returns>Appointment.</returns>
        internal VIMTTmpCreateAppointmentRequestData Map()
        {
            var vimtAppointmentKind = MappingResolvers.GetAppointmentKind(_serviceAppointment, _organizationService);
            var createAppointmentRequest = new VIMTTmpCreateAppointmentRequestData
            {
                Version = "1.0",
                Id = _appointment?.Id.ToString() ?? _serviceAppointment.Id.ToString(), //1 appointment
                Duration = _serviceAppointment.ScheduledDurationMinutes ?? 0,
                DateTime = _serviceAppointment.ScheduledStart.ToString(),
                SchedulingRequestType = VIMTTmpSchedulingRequestType.OTHER_THAN_NEXT_AVAILABLE_PATIENT_REQUESTED,
                SchedulingRequestTypeSpecified = true,
                Type = VIMTTmpAppointmentType.REGULAR,
                TypeSpecified = true,             
                DesiredDate = _serviceAppointment.ScheduledStart.ToString(),
                DesiredDateSpecified = true,
                AppointmentKind = vimtAppointmentKind,
                SourceSystem = "TMP",
                SamlToken = "", // TODO
            };

            if (HasBookingNotes(_serviceAppointment)) createAppointmentRequest.BookingNotes = _serviceAppointment.cvt_SchedulingInstructions.Length > 150 ? _serviceAppointment.cvt_SchedulingInstructions.Substring(0,150) : _serviceAppointment.cvt_SchedulingInstructions;

            createAppointmentRequest.Patients = MappingResolvers.ResolveContacts(_contacts, _organizationService, _serviceAppointment, vimtAppointmentKind,
                _isGroup, _isGroup && (vimtAppointmentKind != VIMTTmpAppointmentKind.MOBILE_ANY && vimtAppointmentKind != VIMTTmpAppointmentKind.MOBILE_GFE )? _appointment.Id : Guid.Empty).ToArray();
            createAppointmentRequest.Providers = MappingResolvers.MapProviders(_systemUser, _organizationService, _serviceAppointment, vimtAppointmentKind).ToArray();
            return createAppointmentRequest;
        }

        internal VIMTTmpUpdateAppointmentRequestData MapUpdate(VIMTTmpCreateAppointmentRequestData source)
        {
            return new VIMTTmpUpdateAppointmentRequestData
            {
                AppointmentKind = source.AppointmentKind,
                BookingNotes = source.BookingNotes,
                DateTime = source.DateTime,
                DesiredDate = source.DesiredDate,
                DesiredDateSpecified = source.DesiredDateSpecified,
                Duration = source.Duration,
                Id = source.Id,
                Patients = source.Patients,
                Providers = source.Providers,
                SchedulingRequestType = source.SchedulingRequestType,
                SchedulingRequestTypeSpecified = source.SchedulingRequestTypeSpecified,
                SourceSystem = source.SourceSystem,
                Status = source.Status,
                Type = source.Type,
                TypeSpecified = source.TypeSpecified,
                Version = source.Version,
                SamlToken = "", // TODO
            };
        }

        internal VIMTTmpUpdateAppointmentRequestData MapUpdate()
        {
            var vimtAppointmentKind = MappingResolvers.GetAppointmentKind(_serviceAppointment, _organizationService);
            var updateAppointmentRequest = new VIMTTmpUpdateAppointmentRequestData
            {
                Version = "1.0",
                Id = _appointment?.Id.ToString() ?? _serviceAppointment.Id.ToString(), //1 appointment
                Duration = _serviceAppointment.ScheduledDurationMinutes ?? 0,
                DateTime = _serviceAppointment.ScheduledStart.ToString(),
                SchedulingRequestType = VIMTTmpSchedulingRequestType.OTHER_THAN_NEXT_AVAILABLE_PATIENT_REQUESTED,
                SchedulingRequestTypeSpecified = true,
                Type = VIMTTmpAppointmentType.REGULAR,
                TypeSpecified = true,
                DesiredDate = _serviceAppointment.ScheduledStart.ToString(),
                DesiredDateSpecified = true,
                AppointmentKind = vimtAppointmentKind,
                SourceSystem = "TMP",
                SamlToken = "", // TODO
            };

            if (HasBookingNotes(_serviceAppointment)) updateAppointmentRequest.BookingNotes = _serviceAppointment.cvt_SchedulingInstructions;

            updateAppointmentRequest.Patients = MappingResolvers.ResolveContacts(_contacts, _organizationService, _serviceAppointment, vimtAppointmentKind,
                _isGroup, _isGroup ? _appointment.Id : Guid.Empty).ToArray();
            updateAppointmentRequest.Providers = MappingResolvers.MapProviders(_systemUser, _organizationService, _serviceAppointment, vimtAppointmentKind).ToArray();
            return updateAppointmentRequest;
        }

        /// <summary>
        /// Determine whether the Service Appointment has booking notes.
        /// </summary>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <returns>Whether the Service Appointment has booking notes.</returns>
        private static bool HasBookingNotes(ServiceAppointment serviceAppointment)
        {
            return !string.IsNullOrEmpty(serviceAppointment.cvt_SchedulingInstructions);
        }
    }
}