﻿using System;
using System.ComponentModel.Composition;
using System.Diagnostics;
using VA.TMP.Integration.VIMT.HealthShare.Processor;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.Integration.VIMT.Shared;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.MessageHandler
{
    /// <summary>
    /// HealthShare Make Cancel Outbound Handler for VIMT.
    /// </summary>
    [Export(typeof(IMessageHandler))]
    [ExportMetadata("MessageType", MessageRegistry.TmpHealthShareMakeCancelOutboundRequestMessage)]
    public class MakeCancelOutboundHandler : RequestResponseHandler
    {
        private TmpHealthShareMakeCancelOutboundRequestMessage _request;
        private string _serializedRequest;
        private Stopwatch _timer;

        /// <summary>
        /// Handle the Make Cancel Outbound Request.
        /// </summary>
        /// <param name="message">TmpHealthShareMakeCancelOutboundRequestMessage.</param>
        /// <returns>TmpHealthShareMakeCancelOutboundResponseMessage.</returns>
        public override IMessageBase HandleRequestResponse(object message)
        {
            _timer = new Stopwatch();

            try
            {
                _request = (TmpHealthShareMakeCancelOutboundRequestMessage)message ?? throw new Exception("ERROR: The message object parameter in MakeCancelOutboundHandler is null");

                _timer.Start();
                _serializedRequest = PipelineUtilities.SerializeInstance(_request);
                
                var requestProcessor = new MakeCancelOutboundProcessor();
                var response = (TmpHealthShareMakeCancelOutboundResponseMessage)requestProcessor.Execute(_request);

                if (response.ExceptionOccured) Logger.Instance.Error(string.Format("ERROR: HealthShare Make Cancel Outbound: {0}", response.ExceptionMessage));

                _timer.Stop();
                response.VimtProcessingMs = (int)_timer.ElapsedMilliseconds;

                return response;
            }
            catch (Exception ex)
            {
                return LogAndCreateResponseMessage(ex);
            }
        }

        /// <summary>
        /// Logs an error.
        /// </summary>
        /// <param name="exception">Exception.</param>
        /// <returns>TmpHealthShareMakeCancelOutboundResponseMessage.</returns>
        private TmpHealthShareMakeCancelOutboundResponseMessage LogAndCreateResponseMessage(Exception exception)
        {
            Logger.Instance.Error("ERROR: HealthShare Make Cancel Outbound Exception:", exception);

            var exceptionMessage = exception.Message;
            exceptionMessage += PipelineUtilities.BuildErrorMessage(exception);

            var response = new TmpHealthShareMakeCancelOutboundResponseMessage
            {
                ExceptionOccured = true,
                ExceptionMessage = exceptionMessage,
                SerializedInstance = _serializedRequest
            };

            response.PatientIntegrationResultInformation.Add(
                new PatientIntegrationResultInformation
                {
                    ExceptionOccured = true,
                    ExceptionMessage = exceptionMessage
                });

            return response;
        }
    }
}