﻿using System;
using System.ComponentModel.Composition;
using System.Diagnostics;
using VA.TMP.Integration.VIMT.HealthShare.Processor;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.Integration.VIMT.Shared;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.MessageHandler
{
    /// <summary>
    /// HealthShare Make Cancel Inbound Handler for VIMT.
    /// </summary>
    [Export(typeof(IMessageHandler))]
    [ExportMetadata("MessageType", MessageRegistry.TmpHealthShareMakeCancelInboundRequestMessage)]
    public class MakeCancelInboundHandler : RequestResponseHandler
    {
        private TmpHealthShareMakeCancelInboundRequestMessage _request;
        private string _serializedRequest;
        private Stopwatch _timer;

        /// <summary>
        /// Handle the Make Cancel Inbound Request.
        /// </summary>
        /// <param name="message">TmpHealthShareMakeCancelInboundRequestMessage.</param>
        /// <returns>TmpHealthShareMakeCancelInboundResponseMessage.</returns>
        public override IMessageBase HandleRequestResponse(object message)
        {
            _timer = new Stopwatch();

            try
            {
                _request = (TmpHealthShareMakeCancelInboundRequestMessage)message ?? throw new Exception("ERROR: The message object parameter in MakeCancelInboundHandler is null");

                _timer.Start();
                _serializedRequest = PipelineUtilities.SerializeInstance(_request);

                var requestProcessor = new MakeCancelInboundProcessor();
                var response = (TmpHealthShareMakeCancelInboundResponseMessage)requestProcessor.Execute(_request);

                if (response.ExceptionOccured) Logger.Instance.Error(string.Format("ERROR: HealthShare Make Cancel Inbound: {0}", response.ExceptionMessage));

                response.VimtRequest = _serializedRequest;
                response.VimtResponse = PipelineUtilities.SerializeInstance(response);
                _timer.Stop();
                response.VimtProcessingMs = (int)_timer.ElapsedMilliseconds;

                return response;
            }
            catch (Exception ex)
            {
                return LogAndCreateResponseMessage(ex);
            }
        }

        /// <summary>
        /// Logs an error.
        /// </summary>
        /// <param name="exception">Exception.</param>
        /// <returns>TmpHealthShareMakeCancelOutboundResponseMessage.</returns>
        private TmpHealthShareMakeCancelInboundResponseMessage LogAndCreateResponseMessage(Exception exception)
        {
            Logger.Instance.Error("ERROR: HealthShare Make Cancel Inbound Exception:", exception);

            var exceptionMessage = exception.Message;
            exceptionMessage += PipelineUtilities.BuildErrorMessage(exception);

            return new TmpHealthShareMakeCancelInboundResponseMessage
            {
                ExceptionOccured = true,
                ExceptionMessage = exceptionMessage,
                VimtRequest = _serializedRequest
            };
        }
    }
}