﻿using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using VA.TMP.DataModel;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.Mappers
{
    internal static class MappingResolver
    {
        /// <summary>
        /// Site Resolver Function
        /// </summary>
        /// <param name="organizationService">Organization Service</param>
        /// <param name="stationNumber">The Clinic's Station Number</param>
        /// <returns>The Site's Entity Reference</returns>
        internal static EntityReference SiteResolver(IOrganizationService organizationService, string stationNumber)
        {
            EntityReference siteReference = null;
            using (var srv = new Xrm(organizationService))
            {
                var site = srv.mcs_siteSet.FirstOrDefault(x => x.mcs_StationNumber == stationNumber);
                if (site == null)
                {
                    Logger.Instance.Debug($"Site with Station Number: {stationNumber} does not exist");
                    return null;
                }

                if (site.mcs_siteId != null) siteReference = new EntityReference(mcs_site.EntityLogicalName, site.mcs_siteId.Value);
            }
            return siteReference;
        }

        /// <summary>
        /// Clinic Resolver Function - Retrives the clinic Id matching the Clinic IEN provided
        /// </summary>
        /// <param name="organizationService">Organization Service</param>
        /// <param name="clinicIen">The Clinic's Ien Number</param>
        /// <param name="stationNumber">The Facility Station Number</param>
        /// <returns>The Resource(Clinic) ID</returns>
        internal static Guid? ClinicResolver(IOrganizationService organizationService, string clinicIen, string stationNumber)
        {
            Guid? clinicId = null;
            using (var srv = new Xrm(organizationService))
            {
                var resources = (from r in srv.mcs_resourceSet
                    join site in srv.mcs_siteSet on r.mcs_RelatedSiteId.Id equals site.mcs_siteId.Value
                    where r.cvt_ien == clinicIen
                                 select new {r.mcs_resourceId, site.mcs_StationNumber});

                foreach(var res in resources)
                {
                    if(res.mcs_StationNumber == stationNumber)
                    {
                        clinicId = res.mcs_resourceId.Value;
                        break;
                    }
                }
            }
            return clinicId;
        }

        internal static EntityReference FacilityResolver(IOrganizationService organizationService, string stationNumber)
        {
            EntityReference facilityReference = null;
            using (var srv = new Xrm(organizationService))
            {
                var facility = srv.mcs_facilitySet.FirstOrDefault(x => x.mcs_StationNumber == stationNumber);
                if (facility == null)
                {
                    Logger.Instance.Debug($"Facility with Station Number: {stationNumber} does not exist");
                    return null;
                }
                if (facility.mcs_facilityId != null) facilityReference = new EntityReference(mcs_facility.EntityLogicalName, facility.mcs_facilityId.Value);
            }
            return facilityReference;
        }

        internal static EntityReference VisnResolver(IOrganizationService organizationService, int visnNumber)
        {
            var visnString = $"VISN {visnNumber:D2}";
            EntityReference visnReference = null;
            using (var srv = new Xrm(organizationService))
            {
                var visn = srv.BusinessUnitSet.FirstOrDefault(x => x.Name == visnString);
                if (visn == null)
                {
                    Logger.Instance.Debug($"Visn with Name: {visnString} does not exist");
                    return null;
                }
                if (visn.BusinessUnitId != null) visnReference = new EntityReference(BusinessUnit.EntityLogicalName, visn.BusinessUnitId.Value);
            }
            return visnReference;
        }

        internal static EntityReference DefaultProviderResolver(IOrganizationService organizationService, string email)
        {
            EntityReference defaultProviderReference = null;
            using (var srv = new Xrm(organizationService))
            {
                var defaultProvider = srv.SystemUserSet.FirstOrDefault(x => x.InternalEMailAddress == email);
                if (defaultProvider == null)
                {
                    Logger.Instance.Debug($"Default Provider (User) with Email: {email} does not exist");
                    return null;
                }
                if (defaultProvider.SystemUserId != null) defaultProviderReference = new EntityReference(SystemUser.EntityLogicalName, defaultProvider.SystemUserId.Value);
            }
            return defaultProviderReference;
        }
    }
}