﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.OptionSets;
using VRM.Integration.Servicebus.Core;


namespace VA.TMP.Integration.VIMT.HealthShare.Mappers
{
    internal class MakeCancelMapper
    {
        private readonly TmpHealthShareMakeAndCancelAppointmentRequestMessage _request;
        private readonly OrganizationServiceProxy _organizationServiceProxy;

        public MakeCancelMapper(TmpHealthShareMakeAndCancelAppointmentRequestMessage request, OrganizationServiceProxy organizationServiceProxy)
        {
            _request = request;
            _organizationServiceProxy = organizationServiceProxy;
        }

        internal Appointment Map()
        {
            var appointment = new Appointment
            {
                cvt_PatientICN = _request.PatientIcn,
                ScheduledDurationMinutes = _request.Duration,
                cvt_VisitStatus = _request.VisitStatus,
                cvt_FacilityText = _request.Facility,
                cvt_ClinicIEN = _request.ClinicIen,
                cvt_ClinicName = _request.ClinicName,
                cvt_ProviderEmail = _request.ProviderEmail,
                cvt_ConsultId = _request.ConsultId,
                cvt_ConsultName = _request.ConsultName,
                cvt_CancelReason = _request.CancelReason,
                cvt_CancelCode = _request.CancelCode,
                cvt_CancelRemarks = _request.CancelRemarks,
                cvt_SchedulerName = _request.SchedulerName,
                cvt_SchedulerEmail = _request.SchedulerEmail,
                Subject = _request.ConsultName
            };

            MapScheduledStart(appointment);

            if (_request.Duration < 0)
            {
                Logger.Instance.Error("ERROR: Duration in the Request is Invalid");
                throw new ArgumentNullException("Duration", "ERROR: Duration in the HealthShare Request is NULL/INVALID");
            }

            using (var service = new Xrm(_organizationServiceProxy))
            {
                MapProviderDetails(service, appointment);
                MapSchedulerDetails(appointment, service);
                MapFacilitySiteDetails(appointment, service);
                MapPatientDetails(service, appointment);
            }

            return appointment;
        }

        private void MapScheduledStart(Appointment appointment)
        {
            var isParseSuccessful = DateTime.TryParse(_request.StartTime, out var startTime);
            if (!string.IsNullOrWhiteSpace(_request.StartTime) && isParseSuccessful)
            {
                appointment.ScheduledStart = startTime;
            }
            else
            {
                Logger.Instance.Error("ERROR: StartTime in the Request is null/Invalid");
                throw new ArgumentNullException("StartTime", "ERROR: StartTime in the Request Json is NULL/INVALID");
            }
        }

        private void MapPatientDetails(Xrm service, Appointment appointment)
        {
            if (string.IsNullOrWhiteSpace(_request.PatientIcn))
            {
                Logger.Instance.Info("INFO: Patient ICN is missing in the request");
            }
            else
            {
                var pId = service.mcs_personidentifiersSet.FirstOrDefault(i => i.mcs_identifier == _request.PatientIcn &&
                                                                                  i.mcs_assigningauthority == "USVHA");
                if (pId != null)
                {
                    var optParty = new ActivityParty {["partyid"] = pId.mcs_patient};
                    var op = new List<ActivityParty> {optParty};
                    appointment.OptionalAttendees = op;
                }
                else
                {
                    Logger.Instance.Debug($"Cannot find patient with IEN matching {_request.PatientIcn}");
                }
            }
        }

        private void MapFacilitySiteDetails(Appointment appointment, Xrm service)
        {
            if (string.IsNullOrWhiteSpace(_request.Facility))
            {
                Logger.Instance.Info("INFO: Facility is missing in the request");
            }
            else
            {
                var facility = service.mcs_facilitySet.FirstOrDefault(i => i.mcs_StationNumber == _request.Facility.Substring(0, 3));

                var site = service.mcs_siteSet.FirstOrDefault(i => i.mcs_StationNumber == _request.Facility);
                if (site != null)
                {
                    appointment.cvt_FacilityText = $"{site.mcs_UserNameInput}  {site.FormattedValues["mcs_type"]}({site.mcs_StationNumber})";
                    appointment.cvt_Site = new EntityReference(mcs_site.EntityLogicalName, site.Id);
                }

                if (facility != null)
                {
                    appointment.cvt_Facility = new EntityReference("mcs_facility", facility.Id);
                    appointment.cvt_FacilityText = $"{facility.mcs_name}  ({facility.mcs_StationNumber})";
                    appointment.OwnerId = GetSchedulerGroup(facility, service);
                }
                else
                {
                    appointment.cvt_FacilityText = "";
                    //Log it when we can't find a facility
                    Logger.Instance.Debug($"Cannot find Facility with Station Number = {_request.Facility}");
                }

                appointment.Subject = _request.ConsultName + " @ " + appointment.cvt_FacilityText;
            }
        }

        private void MapSchedulerDetails(Appointment appointment, Xrm service)
        {
            if (string.IsNullOrWhiteSpace(_request.SchedulerEmail))
            {
                Logger.Instance.Info("INFO: SchedulerEmail is missing in the request");
            }
            else
            {
                var scheduler =
                    service.SystemUserSet.FirstOrDefault(i => i.InternalEMailAddress == _request.SchedulerEmail);

                if (scheduler != null)
                {
                    appointment.cvt_SchedulerUser = new EntityReference("systemuser", scheduler.Id);
                }
                else
                {
                    //log it when we can't find a scheduler
                    Logger.Instance.Debug($"Cannot find systemuser with email matching {_request.SchedulerEmail} for Scheduler {_request.SchedulerName}");
                }
            }
        }

        private void MapProviderDetails(Xrm service, Appointment appointment)
        {
            //PROVIDER
            if (string.IsNullOrEmpty(_request.ProviderEmail))
            {
                Logger.Instance.Info("INFO: ProviderEmail is missing in the request");
            }
            else
            {
                var provider = service.SystemUserSet.FirstOrDefault(i => i.InternalEMailAddress == _request.ProviderEmail);

                if (provider != null)
                {
                    appointment.cvt_Provider = new EntityReference("systemuser", provider.Id);

                    var reqParty = new ActivityParty {["partyid"] = new EntityReference("systemuser", provider.Id)};

                    var ap = new List<ActivityParty> {reqParty};
                    appointment.RequiredAttendees = ap;
                }
                else
                {
                    //log it when we can't find a provider
                    Logger.Instance.Debug($"Cannot find systemuser with email matching {_request.ProviderEmail} for Provider");
                }
            }
        }

        private EntityReference GetSchedulerGroup(mcs_facility facility, Xrm srv)
        {
            EntityReference schedulerTeam = null;
            if (facility != null)
            {
                var teams = (from t in srv.TeamSet
                    join f in srv.mcs_facilitySet on t.cvt_Facility.Id equals f.mcs_facilityId.Value
                    where f.mcs_facilityId.Value == facility.Id 
                    select new Team{Id = t.Id, cvt_Type = t.cvt_Type});
                //t.cvt_Type.Value == Teamcvt_Type.Scheduler.GetHashCode()
                foreach (var team in teams)
                {
                    if (team.cvt_Type!= null && team.cvt_Type.Value == Teamcvt_Type.Scheduler.GetHashCode())
                    {
                        schedulerTeam = new EntityReference(Team.EntityLogicalName, team.Id);
                        break;
                    }
                }

                if (schedulerTeam == null)
                {
                    //log it when we can't find a scheduler team
                    Logger.Instance.Debug($"INFO: Scheduler Team for the facility {facility.mcs_name} with Station Number {_request.Facility} is not found in TMP");
                }
            }

            return schedulerTeam;
        }
    }
}