﻿using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.OptionSets;
using Vimt.HealthShare.Messages.Messages;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.Mappers
{
    /// <summary>
    /// Mapper class for LOB to EC Request.
    /// </summary>
    internal class GetConsultsLobEcMapper
    {
        /// <summary>
        /// Map LOB to EC Request.
        /// </summary>
        /// <returns>VimtHealthShareGetConsultsRequest.</returns>
        internal VimtHealthShareGetConsultsRequest Map(TmpHealthShareGetConsultsRequest request, string uniqueId, bool isPatientStation, IOrganizationService service)
        {
            if (request == null) throw new Exception("Error calling GetConsultsLobEcMapper. The request is null");

            var vimtRequest = new VimtHealthShareGetConsultsRequest
            {
                UniqueId = uniqueId,
                StationNumber = (isPatientStation) ? request.PatientLoginStationNumber : request.ProviderLoginStationNumber,
                Side = (isPatientStation) ? "Patient" : "Provider"
            };

            if (request.PatientIds != null && request.PatientIds.Any())
            {
                var patientId = request.PatientIds.FirstOrDefault();
                if (patientId != Guid.Empty)
                {
                    using (var context = new Xrm(service))
                    {
                        vimtRequest.PatientDfn = GetPatientDfn(patientId, vimtRequest.StationNumber, context);
                        vimtRequest.PatientIcn = GetPatientIcn(patientId, context);
                    }
                }
            }

            // Substitute PatientIcn with PatientDfn in case PatientIcn is null/empty
            if (string.IsNullOrWhiteSpace(vimtRequest.PatientIcn))
            {
                vimtRequest.PatientIcn = vimtRequest.PatientDfn;
                Logger.Instance.Debug($"Patient ICN for the {vimtRequest.Side} station '{vimtRequest.StationNumber}' is not set. Hence substituting it with Patient DFN");
            }

            if (string.IsNullOrWhiteSpace(vimtRequest.PatientDfn) || vimtRequest.StationNumber == 0)
            {
                Logger.Instance.Debug($"Patient DFN '{vimtRequest.PatientDfn}' or Station Number '{vimtRequest.StationNumber}' for {vimtRequest.Side} is not set. " + $"Hence not invoking the HealthShare GetConsultsForPatient for {vimtRequest.Side} station");
                vimtRequest = null;
            }

            return vimtRequest;
        }

        /// <summary>
        /// Retrieve the Patient's DFN
        /// </summary>
        /// <param name="contactId">Patient Contact Guid</param>
        /// <param name="stationNumber">The Station Number</param>
        /// <param name="context">The Xrm Context</param>
        /// <returns>The Patient DFN from the Patient Identifier based on the station number</returns>
        private static string GetPatientDfn(Guid contactId, int stationNumber, Xrm context)
        {
            string patientDfn = null;

            try
            {
                var personIdentifier = context.mcs_personidentifiersSet.FirstOrDefault(x =>
                    x.mcs_patient.Id == contactId && 
                    x.mcs_assigningauthority == "USVHA" &&
                    x.mcs_assigningfacility == stationNumber.ToString() &&
                    x.mcs_identifiertype.Value == (int) mcs_personidentifiersmcs_identifiertype.ParticipantIdentifier_PI);

                if (personIdentifier == null)
                {
                    var errorMessage = $"ERROR: The patient DFN is not available for the Patient with Id: {contactId} and Station Number : {stationNumber}";
                    Logger.Instance.Error(errorMessage);
                }
                else
                {
                    patientDfn = personIdentifier.mcs_identifier;
                }
            }
            catch (Exception ex)
            {
                throw new Exception($"Error occured while accessing Patient DFN details for the patient with Id: {contactId}\nDetails: {ex.Message}");
            }

            return patientDfn;
        }

        /// <summary>
        /// Retrieve the Patient's ICN
        /// </summary>
        /// <param name="contactId">Patient Contact Guid</param>
        /// <param name="context">Xrm Context</param>
        /// <returns>The Patient ICN from the Patient Identifier for the Patient Contact</returns>
        private static string GetPatientIcn(Guid contactId, Xrm context)
        {
            string patientIcn = null;

            try
            {
                var personIdentifier = context.mcs_personidentifiersSet.FirstOrDefault(x =>
                    x.mcs_patient.Id == contactId &&
                    x.mcs_assigningauthority == "USVHA" &&
                    x.mcs_identifiertype.Value == (int) mcs_personidentifiersmcs_identifiertype.NationalIdentifier_NI);

                if (personIdentifier == null)
                {
                    var errorMessage = $"ERROR: The patient ICN is not available for the Patient with Id: {contactId}";
                    Logger.Instance.Error(errorMessage);
                }
                else
                {
                    patientIcn = personIdentifier.mcs_identifier;
                }
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException($"Error occured while accessing Patient ICN details for the patient with Id: {contactId}\nDetails: {ex.Message}");
            }

            return patientIcn;
        }
    }
}