﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.VistaIntegrationResult
{
    public class VistaIntegrationResultUpdatePostStageRunner : PluginRunner
    {
        public VistaIntegrationResultUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        private string _vimtUrl;
        private Guid _contactId;
        private int _vimtTimeout;

        public override string McsSettingsDebugField => "cvt_vistaintegrationresult";

        public override void Execute()
        {
            Logger.WriteDebugMessage($"Plugin Execution Context Depth: {PluginExecutionContext.Depth}");

            var vir = PrimaryEntity.ToEntity<cvt_vistaintegrationresult>();
            var reason = vir.cvt_CancelReason?.Value ?? -1;
            var primaryRecord = OrganizationService.Retrieve(cvt_vistaintegrationresult.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<cvt_vistaintegrationresult>();

            if (reason != -1)
            {
                var apptId = primaryRecord.cvt_Appointment?.Id ?? primaryRecord.cvt_ServiceActivity.Id;
                bool callVvs;
                bool callVista;
                DataModel.ServiceAppointment sa;

                using (var srv = new Xrm(OrganizationService))
                {
                    if (primaryRecord.cvt_Veteran == null) throw new InvalidPluginExecutionException("No Veteran Listed on Vista Integration Result, stopping integration");

                    _contactId = primaryRecord.cvt_Veteran.Id;

                    var vimtUrlObj = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlObj == null) throw new InvalidPluginExecutionException("VIMT Url cannot be null, please ensure there is a setting called \"VIMT URL\"");

                    _vimtUrl = vimtUrlObj.mcs_value;
                    _vimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(srv, Logger, GetType().Name);

                    var appt = srv.AppointmentSet.FirstOrDefault(a => a.Id == apptId);
                    var id = appt != null ? appt.cvt_serviceactivityid?.Id : apptId;

                    sa = srv.ServiceAppointmentSet.FirstOrDefault(s => s.Id == id);
                    callVista = VistaPluginHelpers.RunVistaIntegration(sa, srv, Logger);
                    callVvs = VistaPluginHelpers.RunVvs(sa, srv, Logger);
                }

                bool success;
                if (callVista)
                {
                    var vistaResponse = SendCancelToVista(apptId, reason, sa);
                    success = ProcessCancelVistaResponse(vistaResponse, apptId);
                }
                else
                {
                    Logger.WriteDebugMessage("Vista Bypassed, checking for running VVS");
                    success = true;
                }

                if (callVvs && success)
                {
                    var response = SendCancelToVvs(apptId);
                    ProcessCancelVvsResponse(response, apptId);
                }
                else
                {
                    Logger.WriteDebugMessage($"VVS bypassed ({!callVvs}) or Vista failed ({!success}), so skipping VVS call");
                }

                Logger.WriteDebugMessage(FinalizeCrmAppointment(apptId)
                    ? "Finished Cancel Individual Patient"
                    : "Failed to Remove Patient from appointment");
            }
            else Logger.WriteDebugMessage("No Cancel Reason was set.  Can't cancel to Vista/VVS");
        }

        private HealthShareMakeCancelOutboundResponseMessage SendCancelToVista(Guid apptId, int cancelReason, DataModel.ServiceAppointment sa)
        {
            //WholeAppointmentCanceled = false
            var cancelRequest = new HealthShareMakeCancelOutboundRequestMessage
            {
                ServiceAppointmentId = sa.Id,
                AppointmentId = apptId,
                Patients = new List<Guid> { _contactId },
                LogRequest = true,
                OrganizationName = PluginExecutionContext.OrganizationName,
                UserId = PluginExecutionContext.UserId,
                VistaIntegrationResultId = PrimaryEntity.Id,
                VisitStatus = VIMT.Shared.VistaStatus.CANCELED.ToString()
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(cancelRequest);
            HealthShareMakeCancelOutboundResponseMessage response = null;
            try
            {
                Logger.WriteDebugMessage("Sending Cancel To VIMT");
                response = Utility.SendReceive<HealthShareMakeCancelOutboundResponseMessage>(new Uri(_vimtUrl), MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, cancelRequest, null, _vimtTimeout, out var lag);
                response.VimtLagMs = lag;
                Logger.WriteDebugMessage("Completed Cancel Vista Pipeline");
                return response;
            }
            catch(Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Cancel Vista Appointment", errorMessage, vimtRequest, typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, apptId, OrganizationService, response?.VimtRequest, response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);

                Logger.WriteToFile(errorMessage);
                return null;
            }
        }

        private VideoVisitDeleteResponseMessage SendCancelToVvs(Guid apptId)
        {
            var cancelRequest = new VideoVisitDeleteRequestMessage
            {
                AppointmentId = apptId,
                OrganizationName = PluginExecutionContext.OrganizationName,
                UserId = PluginExecutionContext.UserId,
                LogRequest = true,
                CanceledPatients = new List<Guid> { _contactId },
                WholeAppointmentCanceled = false
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(cancelRequest);
            VideoVisitDeleteResponseMessage response = null;
            try
            {
                Logger.WriteDebugMessage("Sending Delete VVS to VIMT");
                response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(_vimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, cancelRequest, null, _vimtTimeout, out int lag);
                response.VimtLagMs = lag;
                Logger.WriteDebugMessage("VVS Delete Successfully sent to VIMT");
                return response;
            } 
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Delete Video Visit", errorMessage, vimtRequest, typeof(VideoVisitDeleteRequestMessage).FullName, typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage, apptId, OrganizationService, response?.VimtRequest, response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);

                Logger.WriteToFile(errorMessage);
                return null;
            }
        }

        private bool ProcessCancelVistaResponse(HealthShareMakeCancelOutboundResponseMessage response, Guid apptId)
        {
            if (response == null) return false;

            var errMessage = string.Empty;
            var exceptionOccured = false;
            
            foreach (var patientIntegrationResultInformation in response.PatientIntegrationResultInformation)
            {
                var errorMessage = patientIntegrationResultInformation.ExceptionOccured ? patientIntegrationResultInformation.ExceptionMessage : string.Empty;
                IntegrationPluginHelpers.CreateAppointmentIntegrationResult(
                    "Vista Cancel Individual Patient", patientIntegrationResultInformation.ExceptionOccured, errorMessage,
                    patientIntegrationResultInformation.VimtRequest, response.SerializedInstance, patientIntegrationResultInformation.VimtResponse, 
                    typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, 
                    MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, apptId, OrganizationService, response.VimtLagMs, patientIntegrationResultInformation.EcProcessingMs, 
                    response.VimtProcessingMs, false, patientIntegrationResultInformation);

                errMessage += errorMessage;
                exceptionOccured = !string.IsNullOrEmpty(errMessage);
            }

            if (!exceptionOccured && !response.ExceptionOccured) return true;

            Logger.WriteToFile("Cancel Individual Patient Failed: " + errMessage);
            return false;
        }

        private void ProcessCancelVvsResponse(VideoVisitDeleteResponseMessage response, Guid apptId)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;

            IntegrationPluginHelpers.CreateAppointmentIntegrationResult("VVS Cancel Individual Patient", response.ExceptionOccured, errorMessage, response.VimtRequest, response.SerializedInstance, response.VimtResponse, typeof(VideoVisitDeleteRequestMessage).FullName, typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage, apptId, OrganizationService, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);

            if (!response.ExceptionOccured) Logger.WriteDebugMessage("Vista Results Updated Successfully");
            else Logger.WriteToFile("Cancel Individual Patient Failed: " + errorMessage);
        }

        private bool FinalizeCrmAppointment(Guid apptId)
        {
            DataModel.ServiceAppointment sa;
            DataModel.Appointment appt;
            using (var srv = new Xrm(OrganizationService))
            {
                sa = srv.ServiceAppointmentSet.FirstOrDefault(i => i.Id == apptId);
                appt = sa == null ? srv.AppointmentSet.FirstOrDefault(a => a.Id == apptId) : null;
            }
            if (appt != null)
            {
                var updateAppt = new DataModel.Appointment
                {
                    Id = apptId,
                    OptionalAttendees = GetNewApList(appt)
                };
                try
                {
                    OrganizationService.Update(updateAppt);
                    Logger.WriteDebugMessage("Removed Patient From Appointment: " + _contactId);
                    return true;
                }
                catch (Exception ex)
                {
                    Logger.WriteDebugMessage("Unable to Remove Activity Party from appointment following successful VistA Cancel.  Error: " + CvtHelper.BuildExceptionMessage(ex));
                    return false;
                }
            }

            var updateSa = new DataModel.ServiceAppointment
            {
                Id = apptId,
                Customers = GetNewApList(sa)
            };

            try
            {
                OrganizationService.Update(updateSa);
                Logger.WriteDebugMessage("Removed Patient From SA: " + _contactId);
                return true;
            }
            catch (Exception ex)
            {
                Logger.WriteDebugMessage("Unable to Remove Activity Party from sa following successful VistA Cancel.  Error: " + CvtHelper.BuildExceptionMessage(ex));
                return false;
            }
        }

        private List<ActivityParty> GetNewApList(Entity appointment)
        {
            var appt = appointment.LogicalName == DataModel.Appointment.EntityLogicalName ? appointment.ToEntity<DataModel.Appointment>() : null;
            var sa = appt == null ? appointment.ToEntity<DataModel.ServiceAppointment>() : null;

            if (sa == null && appt == null) throw new InvalidPluginExecutionException($"Appointment with id: {appointment.Id} could not be found");

            return sa != null ? sa.Customers.Where(ap => ap.PartyId.Id != _contactId).ToList() : appt.OptionalAttendees.Where(ap => ap.PartyId.Id != _contactId).ToList();
        }
    }
}