﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle updating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentVvsUpdatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentVvsUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public ServiceAppointmentVvsUpdatePostStageRunner(IServiceProvider serviceProvider, Dictionary<Guid, bool> veterans) : base(serviceProvider)
        {
            Veterans = veterans;
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        public bool Success { get; set; }
        
        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        public Dictionary<Guid, bool> Veterans { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            Success = false;
            var sa = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            if (sa == null) throw new InvalidPluginExecutionException($"Unable to Find Service Appointment with id {PrimaryEntity.Id}");

            using (var context = new Xrm(OrganizationService))
            {
                if (VistaPluginHelpers.RunVvs(sa, context, Logger))
                {
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);

                    var vimtUrlObj = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlObj == null) throw new InvalidPluginExecutionException("VIMT Url cannot be null, please ensure there is a setting called \"VIMT URL\"");

                    VimtUrl = vimtUrlObj.mcs_value;
                    //if (sa.cvt_Type == null || !sa.cvt_Type.Value) return;
                    if (sa.StateCode == null || sa.StateCode != ServiceAppointmentState.Scheduled) throw new InvalidPluginExecutionException("Service Appointment is not in Scheduled State.");

                    if (sa.StatusCode == null 
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.Pending 
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.InterfaceVIMTFailure
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.ReservedScheduled)
                    {
                        SendVistaMessage(sa);
                    }
                    else
                        Logger.WriteDebugMessage("Service Activity not in Proper 'Pending' or 'Interface VIMT Failure' status for writing Vista Results back into TMP, skipping VVS");
                }
                else
                {
                    Logger.WriteDebugMessage("VVS Integration Bypassed, Updating Service Activity Status");
                    IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.ReservedScheduled);
                    Success = true;
                }
            }
        }

        private void SendVistaMessage(DataModel.ServiceAppointment serviceAppointment)
        {
            // Default the success flag to false so that it is false until it successfully reaches a completed Vista Booking
            Success = false;
            List<Guid> changedPatients;
            using (var context = new Xrm(OrganizationService)) VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);
            Logger.WriteDebugMessage("Beginning VistA Book/Cancel");
            bool isBookRequest;

            if (Veterans != null)
            {
                Logger.WriteDebugMessage("Using Patients passed in from Proxy Add Plugin");
                isBookRequest = Veterans.Any(kvp => kvp.Value);
                changedPatients = Veterans.Where(kvp => kvp.Value == isBookRequest).Select(kvp => kvp.Key).ToList();
            }
            else
            {
                Logger.WriteDebugMessage("Using Vista Integration Results to determine Patient Delta");
                changedPatients = VistaPluginHelpers.GetChangedPatients(serviceAppointment, OrganizationService, Logger, out isBookRequest);
            }

            if (isBookRequest)
            {
                bool isUpdate;
                using (var srv = new Xrm(OrganizationService))
                {
                    isUpdate = srv.mcs_integrationresultSet.Where(
                        i => i.mcs_serviceappointmentid.Id == serviceAppointment.Id
                        && i.mcs_status.Value != (int)mcs_integrationresultmcs_status.Error).ToList()
                        .Any(i => i.mcs_VimtMessageRegistryName.ToLower().Contains("accenture"));
                }

                if (!isUpdate)
                {
                    var createRequest = new VideoVisitCreateRequestMessage
                    {
                        AppointmentId = serviceAppointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        AddedPatients = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Create Booking to Vista");
                    var response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, createRequest, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    ProcessVistaCreateResponse(response, typeof(VideoVisitCreateRequestMessage), typeof(VideoVisitCreateResponseMessage), serviceAppointment);
                }
                else
                {
                    var updateRequest = new VideoVisitUpdateRequestMessage
                    {
                        AppointmentId = serviceAppointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        Contacts = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Update Booking to Vista");
                    var response = Utility.SendReceive<VideoVisitUpdateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitUpdateRequestMessage, updateRequest, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    ProcessVistaUpdateResponse(response, typeof(VideoVisitUpdateRequestMessage), typeof(VideoVisitUpdateResponseMessage), serviceAppointment);
                }
            }
            else
            {
                var patIds = "";
                changedPatients.ForEach(p => patIds += p.ToString() + "; ");
                patIds.Trim().TrimEnd(';');
                Logger.WriteToFile("No Booked Patients.  Individual patient(s) was canceled through Cancel Dialog and has already been sent to VVS in previous plugin instance.  No action needed here, ending thread.  Canceled Patients: " + patIds);
            }
        }

        private void ProcessVistaBookResponse(bool isCreate, DataModel.ServiceAppointment serviceAppointment, string errorMessage, bool exceptionOccured, string vimtRequest, string serializedInstance, string vimtResponse, int EcProcessingTime, int vimtProcessingTime, int lagTime, WriteResults writeResults)
        {
            var name = isCreate ? "Create VVS" : "Update VVS";
            var reqType = isCreate ? typeof(VideoVisitCreateRequestMessage).FullName : typeof(VideoVisitUpdateRequestMessage).FullName;
            var respType = isCreate ? typeof(VideoVisitCreateResponseMessage).FullName : typeof(VideoVisitUpdateResponseMessage).FullName;
            var regName = isCreate ? MessageRegistry.VideoVisitCreateRequestMessage : MessageRegistry.VideoVisitUpdateRequestMessage;
            IntegrationPluginHelpers.CreateIntegrationResult(name, exceptionOccured, errorMessage, vimtRequest, serializedInstance, vimtResponse, reqType, respType, regName, 
                serviceAppointment.Id, OrganizationService, lagTime, EcProcessingTime, vimtProcessingTime);

            if (exceptionOccured)
            {
                IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.InterfaceVIMTFailure);
                Logger.WriteDebugMessage("VVS Booking Failed");
                return;
            }

            Success = !exceptionOccured;
            if (Success) IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.ReservedScheduled);
            else IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.InterfaceVIMTFailure);
        }

        private void ProcessVistaCreateResponse(VideoVisitCreateResponseMessage response, Type requestType, Type responseType, DataModel.ServiceAppointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(true, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.EcProcessingMs, response.VimtProcessingMs, response.VimtLagMs, response.WriteResults);
        }

        private void ProcessVistaUpdateResponse(VideoVisitUpdateResponseMessage response, Type requestType, Type responseType, VA.TMP.DataModel.ServiceAppointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(false, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.EcProcessingMs, response.VimtProcessingMs, response.VimtLagMs, response.WriteResults);
        }
    }
}