﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    public class ServiceAppointmentVistaHealthShareUpdatePostStageRunner : PluginRunner
    {
        public ServiceAppointmentVistaHealthShareUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        public bool Success { get; set; }

        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        public override void Execute()
        {
            var saRecord = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();

            if (saRecord.cvt_Type != null && (saRecord.mcs_groupappointment == null || !saRecord.mcs_groupappointment.Value || saRecord.cvt_Type.Value)) //VA Video Connect
            {
                using (var context = new Xrm(OrganizationService))
                {
                    var runVista = VistaPluginHelpers.RunVistaIntegration(saRecord, context, Logger);
                    if (runVista)
                    {
                        if (saRecord.StateCode != null && (saRecord.StateCode.Value == ServiceAppointmentState.Scheduled || saRecord.StateCode.Value == ServiceAppointmentState.Open))
                        {
                            var orgName = PluginExecutionContext.OrganizationName;
                            var user = OrganizationService.Retrieve(SystemUser.EntityLogicalName, PluginExecutionContext.UserId, new ColumnSet(true)).ToEntity<SystemUser>();

                            Logger.WriteDebugMessage("Starting GetAndSetIntegrationSettings");
                            GetAndSetIntegrationSettings(context);
                            Logger.WriteDebugMessage("Finished GetAndSetIntegrationSettings");

                            var anyFailures = RunVistaBooking(saRecord, orgName, user);

                            if (!anyFailures) TriggerNextIntegration();
                        }
                        else
                        {
                            Success = true;
                            Logger.WriteDebugMessage("Service Activity is not in a proper status to run Vista Integration");
                        }                            
                    }
                    else
                    {
                        TriggerNextIntegration();
                        Logger.WriteDebugMessage("Vista Integration Switched Off, moving on to next integration.");
                    }
                }
            }
            else Logger.WriteDebugMessage("Clinic Based Groups do not run Vista Integration");
        }

        private void GetAndSetIntegrationSettings(Xrm context)
        {
            VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);

            var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
            if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
            else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
        }

        private bool RunVistaBooking(DataModel.ServiceAppointment sa, string orgName, SystemUser user)
        {
            var anyfailures = false;
            var veterans = VistaPluginHelpers.GetChangedPatients(sa, OrganizationService, Logger, out var isBooking);

            // Checks to ensure new patients were added, and only sends those new patients
            if (!isBooking)
            {
                Logger.WriteDebugMessage("No new patients detected, exiting Vista Booking");
                return false;
            }
            var request = new HealthShareMakeCancelOutboundRequestMessage
            {
                Patients = veterans,
                ServiceAppointmentId = PrimaryEntity.Id,
                LogRequest = true,
                OrganizationName = orgName,
                UserId = user.Id,
                VisitStatus = VIMT.Shared.VistaStatus.SCHEDULED.ToString()
            };

            Logger.WriteDebugMessage("Set up HealthShareMakeCancelOutboundRequestMessage request object.");

            HealthShareMakeCancelOutboundResponseMessage response = null;
            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            try
            {
                Logger.WriteDebugMessage($"Sending HealthShare Make Appointment Request Message to VIMT: {vimtRequest}.");
                response = Utility.SendReceive<HealthShareMakeCancelOutboundResponseMessage>(new Uri(VimtUrl), MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, request, null, VimtTimeout, out var lag);
                response.VimtLagMs = lag;
                Logger.WriteDebugMessage($"Finished Sending HealthShare Make Appointment Request Message to VIMT for Service Appointment with Id: {PrimaryEntity.Id}");
                ProcessVistaResponse(response);
                anyfailures = response.ExceptionOccured || (response.PatientIntegrationResultInformation != null && response.PatientIntegrationResultInformation.Any(x => x.ExceptionOccured));
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Make Vista Appointment", errorMessage, vimtRequest, 
                    typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, 
                    MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response?.VimtRequest, response?.VimtResponse, 
                    response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                Logger.WriteToFile(errorMessage);
                anyfailures = true;
            }

            return anyfailures;
        }

        private void ProcessVistaResponse(HealthShareMakeCancelOutboundResponseMessage response)
        {
            foreach (var patientIntegrationResultInformation in response.PatientIntegrationResultInformation)
            {
                var errorMessage = patientIntegrationResultInformation.ExceptionOccured ? patientIntegrationResultInformation.ExceptionMessage : string.Empty;
                IntegrationPluginHelpers.CreateIntegrationResult("Make Vista Appointment", patientIntegrationResultInformation.ExceptionOccured, errorMessage,
                    patientIntegrationResultInformation.VimtRequest, response.SerializedInstance, patientIntegrationResultInformation.VimtResponse,
                    typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, MessageRegistry.HealthShareMakeCancelOutboundRequestMessage,
                    PrimaryEntity.Id, OrganizationService, response.VimtLagMs, patientIntegrationResultInformation.EcProcessingMs, response.VimtProcessingMs, patientIntegrationResultInformation);
            }
        }

        private void TriggerNextIntegration()
        {
            Success = true;
        }
    }
}