﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    public class ServiceAppointmentVistaHealthShareCancelPostStageRunner : PluginRunner
    {
        public ServiceAppointmentVistaHealthShareCancelPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        /// <summary>
        /// Gets the primary entity.
        /// </summary>
        /// <returns>Returns the primary entity.</returns>
        public override Entity GetPrimaryEntity()
        {
            var primaryReference = (EntityReference)PluginExecutionContext.InputParameters["EntityMoniker"];

            return new Entity(primaryReference.LogicalName) { Id = primaryReference.Id };
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        public bool Success { get; set; }

        public override void Execute()
        {
            var saRecord = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            bool filter;

            using (var context = new Xrm(OrganizationService))
            {
                filter = VistaPluginHelpers.RunVistaIntegration(saRecord, context, Logger);
                Logger.WriteDebugMessage($"Run HealthShare Cancel after checking switches is set to {filter}");
            }

            if (saRecord.cvt_Type != null && (saRecord.mcs_groupappointment == null || !saRecord.mcs_groupappointment.Value || saRecord.cvt_Type.Value))
            {
                if (saRecord.StateCode != null && saRecord.StateCode.Value == ServiceAppointmentState.Canceled)
                {
                    if (filter)
                    {
                        var orgName = PluginExecutionContext.OrganizationName;
                        var user = OrganizationService.Retrieve(SystemUser.EntityLogicalName, PluginExecutionContext.UserId, new ColumnSet(true)).ToEntity<SystemUser>();

                        using (var context = new Xrm(OrganizationService))
                        {
                            Logger.WriteDebugMessage("Starting GetAndSetIntegrationSettings");
                            GetAndSetIntegrationSettings(context);
                            Logger.WriteDebugMessage("Finished GetAndSetIntegrationSettings");

                            var anyFailures = RunCancelVistaBooking(saRecord, orgName, user);

                            if (!anyFailures)
                            {
                                Success = true;
                            }
                            else //something went wrong
                            {
                                Success = false;
                                IntegrationPluginHelpers.ChangeEntityStatus(OrganizationService, saRecord, (int)serviceappointment_statuscode.CancelFailure);
                            }
                        }
                    }
                    else // VIA integrations are turned off
                    {
                        Logger.WriteDebugMessage("Vista Integration (through VIA/HealthShare) is turned off system wide, skipping Vista Cancel integration");
                        Success = true;
                    }
                }
                else
                {
                    Logger.WriteDebugMessage("Vista Cancel Integration skipped - Appointment not in 'Canceled' state");
                    Success = true;
                }
            }
            else
            {
                Logger.WriteDebugMessage("Cancel Request invalid for Clinic Based Group Appointments. ");
                Success = true;
            }
        }

        private void GetAndSetIntegrationSettings(Xrm context)
        {
            VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);

            var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
            if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
            else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
        }

        private bool RunCancelVistaBooking(DataModel.ServiceAppointment sa, string orgName, SystemUser user)
        {
            var anyfailures = false;
            var veterans = GetPatients(sa);

            var request = new HealthShareMakeCancelOutboundRequestMessage
            {
                Patients = veterans,
                ServiceAppointmentId = PrimaryEntity.Id,
                LogRequest = true,
                OrganizationName = orgName,
                UserId = user.Id,
                VisitStatus = VIMT.Shared.VistaStatus.CANCELED.ToString()
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            HealthShareMakeCancelOutboundResponseMessage response = null;

            try
            {
                Logger.WriteDebugMessage($"Sending HealthShare Vista Cancel Appointment Request Message to VIMT: {vimtRequest}.");
                response = Utility.SendReceive<HealthShareMakeCancelOutboundResponseMessage>(new Uri(VimtUrl), MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, request, null, VimtTimeout, out var lag);
                response.VimtLagMs = lag;
                Logger.WriteDebugMessage($"Finished Sending HealthShare Vista Cancel Appointment Request Message to VIMT for Service Appointment with Id: {PrimaryEntity.Id}");
                var status = ProcessVistaResponse(response, sa);

                Logger.WriteDebugMessage($"SA status for HealthShare Cancel is {sa.StatusCode.Value}");

                if (status != sa.StatusCode.Value) anyfailures = true;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Cancel Vista Appointment", errorMessage, vimtRequest, 
                    typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, 
                    MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response?.VimtRequest, response?.VimtResponse,
                    response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                Logger.WriteToFile(errorMessage);
                anyfailures = true;
            }

            Logger.WriteDebugMessage($"For HealthShare Cancel anyfailures is set to {anyfailures}");

            return anyfailures;
        }

        private List<Guid> GetPatients(DataModel.ServiceAppointment sa)
        {
            List<Guid> patients;
            using (var srv = new Xrm(OrganizationService))
            {
                var vistaBookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_ServiceActivity.Id == sa.Id && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.CANCELED.ToString() && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.FAILED_TO_SCHEDULE.ToString() && vir.cvt_Veteran != null).ToList();
                patients = vistaBookings.Select(vir => vir.cvt_Veteran.Id).Distinct().ToList();
            }
            return patients;
        }

        private int ProcessVistaResponse(HealthShareMakeCancelOutboundResponseMessage response, DataModel.ServiceAppointment sa)
        {
            if (response == null) return (int)serviceappointment_statuscode.CancelFailure;

            var errMessage = string.Empty;
            var exceptionOccured = false;

            foreach (var patientIntegrationResultInformation in response.PatientIntegrationResultInformation)
            {
                var errorMessage = patientIntegrationResultInformation.ExceptionOccured ? patientIntegrationResultInformation.ExceptionMessage : string.Empty;

                IntegrationPluginHelpers.CreateIntegrationResult("Cancel Vista Appointment", patientIntegrationResultInformation.ExceptionOccured, errorMessage,
                    patientIntegrationResultInformation.VimtRequest, response.SerializedInstance, patientIntegrationResultInformation.VimtResponse,
                    typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, 
                    MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response.VimtLagMs, 
                    patientIntegrationResultInformation.EcProcessingMs, response.VimtProcessingMs, patientIntegrationResultInformation);

                errMessage += errorMessage;
                exceptionOccured = !string.IsNullOrEmpty(errMessage);
            }

            if (exceptionOccured) Logger.WriteDebugMessage($"The was an error in the response for HealthShare Cancel: {errMessage}");

            var status = (serviceappointment_statuscode)sa.StatusCode.Value;
            if (response.ExceptionOccured || exceptionOccured) status = serviceappointment_statuscode.CancelFailure;

            Logger.WriteDebugMessage($"Status returned after HealthShare Cancel process response is {status}");

            return (int)status;
        }
    }
}