﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    public class ServiceAppointmentIntegrationOrchestratorPostStageRunner : PluginRunner
    {
        public ServiceAppointmentIntegrationOrchestratorPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        private Dictionary<Guid, bool> Veterans { get; set; }

        public override void Execute()
        {
            Logger.WriteDebugMessage("ServiceAppointmentIntegrationOrchestratorPostStageRunner -Execute() Initiated.");
            var isStatusChange = PluginExecutionContext.InputParameters.Contains("EntityMoniker");
            
            if (isStatusChange) ExecuteCancel();
            else
            {
                var saImage = PrimaryEntity.ToEntity<DataModel.ServiceAppointment>();
                var patientsChanged = saImage.Customers != null;

                if (patientsChanged) ExecuteBook();

                var isRetry = saImage.cvt_RetryIntegration;

                if (isRetry.HasValue && isRetry.Value) ExecuteRetry();
            }
        }

        private void ExecuteBook()
        {
            // Runs left to right and stops as soon as one fails, so ordering determines order of integrations
            if (RunProxyAdd() && RunVmr() && RunVista(true) && RunVvs()) Logger.WriteDebugMessage("Successfully completed booking integration pipeline.");
        }

        private void ExecuteCancel()
        {
            // Runs left to right and stops as soon as one fails, so ordering determines order of integrations
            if (RunCancelVmr() && RunVista(false) && RunCancelVvs()) Logger.WriteDebugMessage("Successfully completed cancel integration pipeline.");
        }

        /// <summary>
        /// Logic to execute a Retry
        /// </summary>
        private void ExecuteRetry()
        {
            if (!(bool)PrimaryEntity.Attributes["cvt_retryintegration"]) return;

            Logger.WriteDebugMessage("Retry Initiated");
            
            using (var srv = new Xrm(OrganizationService))
            {
                var failedIRs = srv.mcs_integrationresultSet.Where(ir => ir.mcs_serviceappointmentid.Id == PrimaryEntity.Id && ir.mcs_status.Value == (int)mcs_integrationresultmcs_status.Error).ToList();
                var retryIr = IntegrationPluginHelpers.FindEarliestFailure(failedIRs, Logger);

                if (retryIr == null) Logger.WriteDebugMessage("Skipping Retry as Integration Result with error status could not be found.");
                else RetryIntegrationResult(retryIr);
            }

            // Reset the Retry integration field post run retry execution
            var apt = new DataModel.ServiceAppointment { Id = PrimaryEntity.Id, cvt_RetryIntegration = false };
            OrganizationService.Update(apt);
        }

        private void RetryIntegrationResult(mcs_integrationresult integrationResult)
        {
            var typeOfIntegration = integrationResult.mcs_VimtMessageRegistryName;
            Logger.WriteDebugMessage($"Executing RetryIntegrationResult. Type of Integration: {typeOfIntegration}");

            var successfulRetry = false;
            var finishedIntegrationPipeline = false;
            var isMake = !PluginExecutionContext.InputParameters.Contains("EntityMoniker");

            switch (typeOfIntegration)
            {
                case MessageRegistry.ProxyAddRequestMessage:
                    successfulRetry = RunProxyAdd();
                    if (successfulRetry && RunVmr() && RunVista(true) && RunVvs()) finishedIntegrationPipeline = true;
                    break;
                case MessageRegistry.HealthShareMakeCancelOutboundRequestMessage:
                    successfulRetry = RunVmr();
                    if (successfulRetry && RunVista(isMake) && RunVvs()) finishedIntegrationPipeline = true;
                    break;
                case MessageRegistry.VirtualMeetingRoomCreateRequestMessage:
                    successfulRetry = RunVmr();
                    if (successfulRetry && RunVista(true)) finishedIntegrationPipeline = RunVvs();
                    break;
                case MessageRegistry.VideoVisitCreateRequestMessage:
                    successfulRetry = RunVvs();
                    if (successfulRetry) finishedIntegrationPipeline = true;
                    break;
                case MessageRegistry.VirtualMeetingRoomDeleteRequestMessage:
                    successfulRetry = RunCancelVmr();
                    if (successfulRetry && RunVista(false)) finishedIntegrationPipeline = RunCancelVvs();
                    break;
                case MessageRegistry.VideoVisitDeleteRequestMessage:
                    successfulRetry = RunCancelVvs();
                    finishedIntegrationPipeline = successfulRetry;
                    break;
                default:
                    Logger.WriteDebugMessage($"Unknown Message Type - {typeOfIntegration}, skipping retry");
                    break;
            }

            if (successfulRetry) ChangeIrStatusToRetrySuccess(integrationResult);

            Logger.WriteToFile(string.Format("Remaining Integrations {0}.", finishedIntegrationPipeline ? "Succeeded" : "Did Not Succeed"));
        }

        private void ChangeIrStatusToRetrySuccess(mcs_integrationresult ir)
        {
            Logger.WriteDebugMessage($"Updating IR {ir.mcs_VimtMessageRegistryName} to retry success");
            var updateIr = new mcs_integrationresult
            {
                Id = ir.Id,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.ErrorRetrySuccess)
            };
            OrganizationService.Update(updateIr);
        }

        private bool RunProxyAdd()
        {
            var proxyAddRunner = new ServiceAppointmentCreatePostStageRunner(ServiceProvider);
            proxyAddRunner.RunPlugin(ServiceProvider);

            if (!proxyAddRunner.Success)
            {
                Logger.WriteToFile("Proxy Add did not Succeed, ending integration pipeline");
                return false;
            }
            Veterans = proxyAddRunner.VeteransChanged;
            return true;
        }

        private bool RunVmr()
        {
            Logger.WriteDebugMessage("Running VMR");

            // Only run this on create or retry - for subsequent adding of individual person to H/M group, skip this
            if (PluginExecutionContext.MessageName == "Create" || PrimaryEntity?.Attributes?.Contains("cvt_retryintegration") == true)
            {
                if (PrimaryEntity?.Attributes?.Contains("cvt_retryintegration") == true && (bool)PrimaryEntity["cvt_retryintegration"])
                {
                    Logger.WriteDebugMessage("Checking to see if VMR retry is necessary");

                    using (var srv = new Xrm(OrganizationService))
                    {
                        var failedVmr = srv.mcs_integrationresultSet.FirstOrDefault(x => 
                            x.mcs_serviceappointmentid.Id == PrimaryEntity.Id && 
                            x.mcs_status.Value == (int)mcs_integrationresultmcs_status.Error &&
                            x.mcs_name == "Create Virtual Meeting Room");

                        // If running retry, make sure that VMR wasn't already done.  If so, skip this integration and move on to the next one
                        if (failedVmr == null)
                        {
                            Logger.WriteDebugMessage("No need to retry VMR");
                            return true;
                        }
                        else Logger.WriteDebugMessage("VMR retry will execute");
                    }                    
                }

                var vmrRunner = new ServiceAppointmentVmrUpdatePostStageRunner(ServiceProvider);
                vmrRunner.RunPlugin(ServiceProvider);

                if (!vmrRunner.Success)
                {
                    Logger.WriteToFile("Create VMR did not succeed, ending integration pipeline");
                    return false;
                }
            }
            else Logger.WriteDebugMessage("Skipping VMR for non-retry SA Update");

            return true;
        }

        private bool RunVista(bool isMake)
        {
            if (isMake)
            {
                var vistaRunner = new ServiceAppointmentVistaHealthShareUpdatePostStageRunner(ServiceProvider);
                vistaRunner.RunPlugin(ServiceProvider);

                if (!vistaRunner.Success) Logger.WriteToFile("Vista Make Appointment did not succeed, ending integration pipeline");
                return vistaRunner.Success;
            }
            else
            {
                var vistaRunner = new ServiceAppointmentVistaHealthShareCancelPostStageRunner(ServiceProvider);
                vistaRunner.RunPlugin(ServiceProvider);

                if (!vistaRunner.Success) Logger.WriteToFile("Vista Cancel Appointment did not succeed, ending integration pipeline");
                return vistaRunner.Success;
            }
        }

        private bool RunVvs()
        {
            var vvsRunner = new ServiceAppointmentVvsUpdatePostStageRunner(ServiceProvider, Veterans);
            vvsRunner.RunPlugin(ServiceProvider);

            if (!vvsRunner.Success) Logger.WriteToFile("VVS did not succeed, ending integration pipeline");

            return vvsRunner.Success;
        }

        private bool RunCancelVvs()
        {
            var cancelRunner = new ServiceAppointmentCancelPostStageRunner(ServiceProvider);
            cancelRunner.RunPlugin(ServiceProvider);

            if (!cancelRunner.Success) Logger.WriteToFile("Cancel VVS failed, ending integration pipeline");

            return cancelRunner.Success;
        }

        private bool RunCancelVmr()
        {
            var cancelRunner = new ServiceAppointmentVmrCancelPostStageRunner(ServiceProvider);
            cancelRunner.RunPlugin(ServiceProvider);

            if (!cancelRunner.Success) Logger.WriteToFile("Cancel VMR failed, ending integration pipeline");

            return cancelRunner.Success;
        }
    }
}