﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle creating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentCreatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        /// <summary>
        /// Gets or sets the ProxyAddFakeResponseType.
        /// </summary>
        private string ProxyAddFakeResponseType { get; set; }

        /// <summary>
        /// Gets or sets ProcessingCode.
        /// </summary>
        private string ProcessingCode { get; set; }

        /// <summary>
        /// Gets or sets ReturnMviMessagesInResponse.
        /// </summary>
        private bool ReturnMviMessagesInResponse { get; set; }

        /// <summary>
        /// Gets or sets PatientVeteran.
        /// </summary>
        private bool PatientVeteran { get; set; }

        /// <summary>
        /// Gets or sets PatientServiceConnected.
        /// </summary>
        private bool PatientServiceConnected { get; set; }

        /// <summary>
        /// Gets or sets PatientType.
        /// </summary>
        private int PatientType { get; set; }

        /// <summary>
        /// The Guids of Veterans that were added (bool = true) or removed (bool = false)
        /// </summary>
        public Dictionary<Guid, bool> VeteransChanged { get; set; }

        /// <summary>
        /// This indicates whether there was a proxy add failure - if not, then the next integration will be called
        /// </summary>
        public bool Success { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            Success = false;
            try
            {
                // This gets the Service Activity
                var saRecord = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();

                // Not Group or is H/M
                if (saRecord.mcs_groupappointment == null || !saRecord.mcs_groupappointment.Value || saRecord.cvt_Type.Value)
                {
                    if (saRecord.StateCode.Value == ServiceAppointmentState.Scheduled )
                    {
                        var orgName = PluginExecutionContext.OrganizationName;
                        var user = OrganizationService.Retrieve(SystemUser.EntityLogicalName, PluginExecutionContext.UserId, new ColumnSet(true)).ToEntity<SystemUser>();

                        using (var context = new Xrm(OrganizationService))
                        {
                            Logger.WriteDebugMessage("Starting GetAndSetIntegrationSettings");
                            GetAndSetIntegrationSettings(context);
                            Logger.WriteDebugMessage("Finished GetAndSetIntegrationSettings");

                            var anyFailures = RunProxyAdd(orgName, user, saRecord);

                            if (!anyFailures) Success = true;
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage(string.Format("ServiceAppointment is not scheduled (status is {1}), Proxy Add Canceled: {0}", PrimaryEntity.Id, saRecord.StatusCode.Value));
                        Success = true;
                    }
                    
                    // Moved here to resolve deadlocking issue between this function and the "Trigger Next Integration" function - when a group appointment is selected, none of the integrations run
                    CvtHelper.SetServiceAppointmentPermissions(OrganizationService, PrimaryEntity, Logger);
                    Logger.WriteDebugMessage("Set the SA Permissions (and assign).");
                }
                else
                {
                    Logger.WriteDebugMessage("Proxy Add is executed on the Appointment, not the Service Activity for Group Appointments that are Clinic Based.");
                    Success = true;
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in ServiceAppointmentCreatePostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Gets and sets the Integration Settings needed for Proxy Add.
        /// </summary>
        /// <param name="context">CRM context.</param>
        private void GetAndSetIntegrationSettings(Xrm context)
        {
            VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);
            var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
            if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
            else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");

            var proxyAddFakeResponseType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "MVI ProxyAdd Fake Response Type");
            if (proxyAddFakeResponseType != null) ProxyAddFakeResponseType = proxyAddFakeResponseType.mcs_value;
            else throw new InvalidPluginExecutionException("No ProxyAddFakeResponseType listed in Integration Settings.  Please contact the Help Desk to add ProxyAddFakeResponseType.  Proxy Add Canceled.");
            
            var processingCode = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ProcessingCode");
            if (processingCode != null) ProcessingCode = processingCode.mcs_value;
            else throw new InvalidPluginExecutionException("No ProcessingCode listed in Integration Settings.  Please contact the Help Desk to add ProcessingCode.  Proxy Add Canceled.");

            var returnMviMessagesInResponse = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ReturnMviMessagesInResponse");
            if (returnMviMessagesInResponse != null) ReturnMviMessagesInResponse = Convert.ToBoolean(returnMviMessagesInResponse.mcs_value);
            else throw new InvalidPluginExecutionException("No ReturnMviMessagesInResponse listed in Integration Settings.  Please contact the Help Desk to add ReturnMviMessagesInResponse.  Proxy Add Canceled.");

            var patientVeteran = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientVeteran");
            if (patientVeteran != null) PatientVeteran = Convert.ToBoolean(patientVeteran.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientVeteran listed in Integration Settings.  Please contact the Help Desk to add PatientVeteran.  Proxy Add Canceled.");

            var patientServiceConnected = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientServiceConnected");
            if (patientServiceConnected != null) PatientServiceConnected = Convert.ToBoolean(patientServiceConnected.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientServiceConnected listed in Integration Settings.  Please contact the Help Desk to add PatientServiceConnected.  Proxy Add Canceled.");

            var patientType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientType");
            if (patientType != null) PatientType = Convert.ToInt32(patientType.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientType listed in Integration Settings.  Please contact the Help Desk to add PatientType.  Proxy Add Canceled.");
        }

        /// <summary>
        /// Sends the Proxy Add Request to VIMT.
        /// </summary>
        /// <param name="orgName">Organization Service.</param>
        /// <param name="user">CRM User.</param>
        /// <param name="sa">Service Appointment</param>
        /// <returns>ProxyAddResponseMessage.</returns>
        private bool RunProxyAdd(string orgName, SystemUser user, DataModel.ServiceAppointment sa)
        {
            var anyfailures = false;
            var failureCount = 0;
            
            // Use the PrimaryEntity so that you can only send Customers that were modified (added or removed)
            VeteransChanged = IntegrationPluginHelpers.GetListOfNewPatients(OrganizationService, sa, Logger, PluginExecutionContext);

            var veteransAdded = VeteransChanged.Where(kvp => kvp.Value).Select(kvp => kvp.Key).ToList();

            foreach (var veteranParty in veteransAdded)
            {
                if (failureCount > 0) return true;

                Logger.WriteDebugMessage("VeteranParty: " + veteranParty);

                var request = new ProxyAddRequestMessage
                {
                    LogRequest = true,
                    ServiceAppointmentId = PrimaryEntity.Id,
                    OrganizationName = orgName,
                    FakeResponseType = ProxyAddFakeResponseType,
                    UserFirstName = user.FirstName,
                    UserLastName = user.LastName,
                    UserId = user.Id,
                    ProcessingCode = ProcessingCode,
                    ReturnMviMessagesInResponse = ReturnMviMessagesInResponse,
                    PatientVeteran = PatientVeteran,
                    PatientServiceConnected = PatientServiceConnected,
                    PatientType = PatientType,
                    VeteranPartyId = veteranParty
                };

                ProxyAddResponseMessage response = null;
                var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

                try
                {
                    Logger.WriteDebugMessage(string.Format("Sending Proxy Add Message to VIMT: {0}.", vimtRequest));
                    response = Utility.SendReceive<ProxyAddResponseMessage>(new Uri(VimtUrl), MessageRegistry.ProxyAddRequestMessage, request, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    Logger.WriteDebugMessage(string.Format("Finished Sending Proxy Add Message to VIMT for {0}", veteranParty));
                    ProcessProxyAddToVistaResponse(response);
                    if (response.ExceptionOccured) failureCount++;
                }
                catch (Exception ex)
                {
                    var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Proxy Add to Vista", errorMessage, vimtRequest, typeof(ProxyAddRequestMessage).FullName,
                        typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage, PrimaryEntity.Id, OrganizationService, response?.VimtRequest, response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                    Logger.WriteToFile(errorMessage);
                    anyfailures = true;
                }
            }

            return failureCount > 0 ? true : anyfailures;
        }

        /// <summary>
        /// Process Proxy Add to VistA response.
        /// </summary>
        /// <param name="response">Proxy Add Response.</param>
        private void ProcessProxyAddToVistaResponse(ProxyAddResponseMessage response)
        {
            if (response == null) throw new Exception("No Proxy Add Response was returned from VIMT");

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            IntegrationPluginHelpers.CreateIntegrationResult("Proxy Add to VistA", response.ExceptionOccured, errorMessage,
                response.VimtRequest, response.SerializedInstance, response.VimtResponse,
                typeof(ProxyAddRequestMessage).FullName, typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage,
                PrimaryEntity.Id, OrganizationService, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);

            if (response.ExceptionOccured) IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.InterfaceVIMTFailure);
        }
    }
}