﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle canceling a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentCancelPostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentCancelPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        public bool Success { get; set; }

        /// <summary>
        /// Gets the primary entity.
        /// </summary>
        /// <returns>Returns the primary entity.</returns>
        public override Entity GetPrimaryEntity()
        {
            var primaryReference = (EntityReference)PluginExecutionContext.InputParameters["EntityMoniker"];

            return new Entity(primaryReference.LogicalName) { Id = primaryReference.Id };
        }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            Success = false;
            try
            {
                using (var context = new Xrm(OrganizationService))
                {
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);

                    VimtUrl = context.mcs_integrationsettingSet.First(x => x.mcs_name == "VIMT URL").mcs_value;
                    var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
                    if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");

                    var serviceAppointment = context.ServiceAppointmentSet.FirstOrDefault(x => x.Id == PrimaryEntity.Id);
                    if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null.");

                    if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Canceled)
                    {
                        Logger.WriteDebugMessage("Service Activity Not in Canceled status, exiting Cancel Integrations");
                        return;
                    }

                    // Call Service to cancel the Video Visit.
                    if (VistaPluginHelpers.RunVvs(serviceAppointment, context, Logger))
                    {
                        var videoVisitDeleteResponseMessage = VistaPluginHelpers.CancelAndSendVideoVisitServiceSa(serviceAppointment, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger, VimtTimeout);
                        if (videoVisitDeleteResponseMessage == null) return;
                        ProcessVideoVisitDeleteResponseMessage(videoVisitDeleteResponseMessage);
                    }
                    else Logger.WriteDebugMessage("Bypassed VVS Cancel");

                    Success = true;
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException($"ERROR in ServiceAppointmentCancelPostStageRunner: {IntegrationPluginHelpers.BuildErrorMessage(ex)}");
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="videoVisitDeleteResponseMessage">Video Visit Delete Response Message.</param>
        private void ProcessVideoVisitDeleteResponseMessage(VideoVisitDeleteResponseMessage videoVisitDeleteResponseMessage)
        {
            Logger.WriteDebugMessage("Processing VVS Cancel Response");
            var errorMessage = videoVisitDeleteResponseMessage.ExceptionOccured ? videoVisitDeleteResponseMessage.ExceptionMessage : string.Empty;
            IntegrationPluginHelpers.CreateIntegrationResult("Cancel VVS", videoVisitDeleteResponseMessage.ExceptionOccured, errorMessage,
                videoVisitDeleteResponseMessage.VimtRequest, videoVisitDeleteResponseMessage.SerializedInstance, videoVisitDeleteResponseMessage.VimtResponse,
                typeof(VideoVisitDeleteRequestMessage).FullName, typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage,
                PrimaryEntity.Id, OrganizationService, videoVisitDeleteResponseMessage.VimtLagMs, videoVisitDeleteResponseMessage.EcProcessingMs, videoVisitDeleteResponseMessage.VimtProcessingMs);
            
            Logger.WriteDebugMessage("Finished Processing VVS Cancel Response");
        }
    }
}