﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.IntegrationResult
{
    /// <summary>
    ///  CRM Plugin Runner class to handle creating a ServiceAppointment.
    /// </summary>
    public class IntegrationResultUpdateRetryPostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public IntegrationResultUpdateRetryPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            DataModel.ServiceAppointment serviceAppointment = null;
            DataModel.Appointment appointment = null;

            try
            {
                var integrationResultEntity = PrimaryEntity.ToEntity<mcs_integrationresult>();

                using (var context = new Xrm(OrganizationService))
                {
                    // Retrieve the Integration Result and ensure it is in the correct state before proceeding.
                    var integrationResult = context.mcs_integrationresultSet.FirstOrDefault(x => x.Id == integrationResultEntity.Id);
                    if (integrationResult == null) throw new InvalidPluginExecutionException("Integration Result cannot be null.");
                    if (integrationResult.mcs_appointmentid == null && integrationResult.mcs_serviceappointmentid == null) throw new InvalidPluginExecutionException("Both AppointmentId and ServiceAppointmentId for Integration Result cannot be null.");

                    // Ensure that the Int Result Retry status = yes
                    if (integrationResult.mcs_retry == null || integrationResult.mcs_retry != true) return;

                    // Get VimtTimeout Setting
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, this.GetType().Name);

                    // Look up the VIMT URL from Integration Settings. 
                    VimtUrl = context.mcs_integrationsettingSet.First(x => x.mcs_name == "VIMT URL").mcs_value;
                    var vimtRegistryMessage = integrationResult.mcs_VimtMessageRegistryName;
                    
                    // If VVS, run separate logic - determines whether "retry" needs to be same "Create" message or is an "Update"
                    if (vimtRegistryMessage.ToLower().Contains("accenture"))
                    {
                        RetryVistaMessage(integrationResult);
                        return;
                    }

                    // Ensure that the integration result is in an error status. 
                    if (integrationResult.mcs_status == null || integrationResult.mcs_status.GetHashCode() != mcs_integrationresultmcs_status.Error.GetHashCode()) return;

                    // Use the integration result to dynamically invoke the call to VIMT. Look up the Int Result values.
                    var requestType = Type.GetType(integrationResult.mcs_VimtRequestMessageType);
                    var responseType = Type.GetType(integrationResult.mcs_VimtResponseMessageType);
                    var vimtRequest = integrationResult.mcs_vimtrequest;

                    // Retrieve the Service Appointment and ensure it is in the correct state before proceeding.
                    if (integrationResult.mcs_serviceappointmentid != null)
                    {
                        serviceAppointment = context.ServiceAppointmentSet.FirstOrDefault(x => x.Id == integrationResult.mcs_serviceappointmentid.Id);
                        if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null.");

                        // Ensure this is a Home/Mobile Service Appointment.
                        if ((serviceAppointment.cvt_Type == null || !serviceAppointment.cvt_Type.Value) && vimtRegistryMessage != MessageRegistry.ProxyAddRequestMessage) return;
                        if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                        if (serviceAppointment.StateCode.Value == ServiceAppointmentState.Open || serviceAppointment.StateCode.Value == ServiceAppointmentState.Closed) return;
                    }
                    else if (integrationResult.mcs_appointmentid != null)
                    {
                        appointment = context.AppointmentSet.FirstOrDefault(x => x.Id == integrationResult.mcs_appointmentid.Id);
                        if (appointment == null) throw new InvalidPluginExecutionException("Appointment cannot be null.");
                        if (appointment.StateCode == null) throw new InvalidPluginExecutionException("Appointment State Code cannot be null");
                        if (appointment.StateCode.Value == AppointmentState.Completed || appointment.StateCode.Value == AppointmentState.Open) return;
                    }

                    // Create the Response Object
                    var responseObject = IntegrationPluginHelpers.InvokeVimtDynamically(requestType, responseType, vimtRequest, vimtRegistryMessage, VimtUrl, VimtTimeout);
                    
                    // Cast to the appropriate Response type
                    switch (vimtRegistryMessage)
                    {
                        case MessageRegistry.ProxyAddRequestMessage:
                            var proxyAddResponse = responseObject as ProxyAddResponseMessage;
                            if (proxyAddResponse == null) throw new InvalidPluginExecutionException($"Exception casting VIMT response to type {typeof(ProxyAddResponseMessage)}");
                            if (integrationResult.mcs_serviceappointmentid != null) ProcessProxyAddResponse(proxyAddResponse, serviceAppointment);
                            else if (integrationResult.mcs_appointmentid != null) ProcessGroupProxyAddResponse(proxyAddResponse, appointment);
                            else throw new InvalidPluginExecutionException("Retry Failed: null appointment and serviceappointment for Proxy Add To Vista");
                            break;

                        case MessageRegistry.VirtualMeetingRoomCreateRequestMessage:
                            // If doing a retry for the VMR Create Request Message, we know the VVC Create was never reached, so we need to Dynamically invoke 
                            // the VVC create request message as well.
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Scheduled) return;
                            var vmrCreateResponse = responseObject as VirtualMeetingRoomCreateResponseMessage;
                            if (vmrCreateResponse == null) throw new InvalidPluginExecutionException($"Exception casting VIMT response to type {typeof(VirtualMeetingRoomCreateResponseMessage)}");
                            ProcessVirtualMeetingRoomCreateResponseMessage(vmrCreateResponse, serviceAppointment);
                            break;

                        case MessageRegistry.VirtualMeetingRoomDeleteRequestMessage:
                            // If doing a retry for the VMR Delete Request Message, we know the VVC Delete was never reached, so we need to Dynamically invoke 
                            // the VVC delete request message as well.
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Canceled) return;
                            var vmrDeleteResponse = responseObject as VirtualMeetingRoomDeleteResponseMessage;
                            if (vmrDeleteResponse == null) throw new InvalidPluginExecutionException($"Exception casting VIMT response to type {typeof(VirtualMeetingRoomDeleteResponseMessage)}");
                            ProcessVirtualMeetingRoomDeleteResponseMessage(vmrDeleteResponse, serviceAppointment);

                            RetryVistaMessage(integrationResult);
                            break;

                        default:
                            Logger.WriteToFile("No Retry has been built for this type of message: " + vimtRegistryMessage);
                            break;
                    }
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException($"ERROR in IntegrationResultUpdateRetryPostStageRunner: {IntegrationPluginHelpers.BuildErrorMessage(ex)}");
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        private void RetryVistaMessage(mcs_integrationresult integrationResult)
        {
            using (var srv = new Xrm(OrganizationService))
            {
                var isCancel = integrationResult.mcs_VimtMessageRegistryName.ToLower().Contains("delete");
                var isCreate = false;
                if (!isCancel)
                    isCreate = GetVistaMessageType(integrationResult);

                var serviceAppointment = integrationResult.mcs_serviceappointmentid != null ? srv.ServiceAppointmentSet.FirstOrDefault(s => s.Id == integrationResult.mcs_serviceappointmentid.Id) : null;
                var groupAppointment = integrationResult.mcs_appointmentid != null ? srv.AppointmentSet.FirstOrDefault(a => a.Id == integrationResult.mcs_appointmentid.Id) : null;

                // Put all prior Vista IntegrationResults for this activity into List bookings
                List<cvt_vistaintegrationresult> bookings;
                if (serviceAppointment != null)
                    bookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_ServiceActivity.Id == serviceAppointment.Id).ToList();
                else
                    bookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_Appointment.Id == groupAppointment.Id).ToList();

                if (serviceAppointment != null && isCreate)
                {
                    Logger.WriteDebugMessage("Attempting Create Retry");
                    var response = VistaPluginHelpers.CreateAndSendVideoVisitService(serviceAppointment, PluginExecutionContext.UserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger, VimtTimeout);
                    ProcessVistaCreateResponse(response, serviceAppointment);
                }
                else if (serviceAppointment != null && !isCreate && !isCancel)
                {
                    var priorBookingSucceeded = bookings.Count > 0 && bookings.FirstOrDefault(book => book.cvt_VistAStatus == VIMT.Shared.VistaStatus.FAILED_TO_SCHEDULE.ToString()) == null;
                    if (priorBookingSucceeded)
                    {
                        Logger.WriteDebugMessage("No Booking Failures to retry, exiting retry function");
                        return;
                    }

                    var response = VistaPluginHelpers.UpdateAndSendVideoVisitService(serviceAppointment, PluginExecutionContext.UserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger, VimtTimeout);
                    ProcessVistaUpdateResponse(response, serviceAppointment);
                }
                else if (serviceAppointment != null && isCancel)
                {
                    Logger.WriteDebugMessage("Cancel retry for Vista Booking of individual Service Activities is not available at this time.");
                    //var priorCancelSucceeded = bookings.Count > 0 && bookings.FirstOrDefault(book => book.cvt_VistAStatus == VistaStatus.FAILED_TO_CANCEL.ToString()) == null;
                    //if (priorCancelSucceeded)
                    //{
                    //    Logger.WriteDebugMessage("No Cancel Failures to retry, exiting retry function");
                    //    return;
                    //}

                    //var response = VistaPluginHelpers.CancelAndSendVideoVisitServiceSa(serviceAppointment, PluginExecutionContext.UserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger);
                    //ProcessVideoVisitDeleteResponseMessage(response, serviceAppointment);
                } else if (groupAppointment != null && isCreate)
                {
                    Logger.WriteDebugMessage("Attempting Group Create Retry");
                    var response = VistaPluginHelpers.CreateAndSendVideoVisitService(groupAppointment, PluginExecutionContext.UserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger, VimtTimeout);
                    ProcessVistaCreateGroupResponse(response, groupAppointment);
                }
                else if (groupAppointment != null && !isCreate && !isCancel)
                {
                    var priorBookingSucceeded = bookings.Count > 0 && bookings.FirstOrDefault(book => book.cvt_VistAStatus == VistaStatus.FAILED_TO_SCHEDULE.ToString()) == null;
                    if (priorBookingSucceeded) {
                        Logger.WriteDebugMessage("No Booking Failures to retry, exiting retry function");
                        return;
                    }

                    var response = VistaPluginHelpers.UpdateAndSendVideoVisitService(groupAppointment, PluginExecutionContext.UserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger, VimtTimeout);
                    ProcessVistaUpdateGroupResponse(response, groupAppointment);
                }
                else if (groupAppointment != null && isCancel)
                {
                    Logger.WriteDebugMessage("Cancel retry for Vista Booking of group appointments is not available at this time.");
                    //need call with AFS to determine if we re-send all canceled patients or just the ones that had failures or partial failures?  
                    //var response = VistaPluginHelpers.CancelAndSendVideoVisitServiceAppt(groupAppointment, PluginExecutionContext.OrganizationName, PluginExecutionContext.UserId, Logger, VimtUrl, OrganizationService, PrimaryEntity.Id);
                    //ProcessVistaCancelGroupResponse(response, groupAppointment);
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="integrationResult"></param>
        /// <returns>true if the retry should be a create or false if it should be an update</returns>
        private bool GetVistaMessageType(mcs_integrationresult integrationResult)
        {
            var isCreate = true;
            using (var srv = new Xrm(OrganizationService))
            {
                var vistaBooks = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_ParentResult.Id == integrationResult.Id).ToList();
                if (vistaBooks.Count > 0)
                {
                    isCreate = false;
                }
                else if (integrationResult.mcs_status.Value != (int)mcs_integrationresultmcs_status.Error)
                {
                    isCreate = false;
                }
            }
            return isCreate;
        }

        /// <summary>
        /// Update the Service Activity and create an Integration Result.
        /// </summary>
        /// <param name="virtualMeetingRoomCreateResponseMessage">Virtual Meeting Room Create Response Message.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVirtualMeetingRoomCreateResponseMessage(VirtualMeetingRoomCreateResponseMessage virtualMeetingRoomCreateResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            // Create a new Int Result. Set to Complete if there was no error. 
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Create Virtual Meeting Room",
                mcs_error = virtualMeetingRoomCreateResponseMessage.ExceptionOccured ? virtualMeetingRoomCreateResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = virtualMeetingRoomCreateResponseMessage.VimtRequest,
                mcs_integrationrequest = virtualMeetingRoomCreateResponseMessage.SerializedInstance,
                mcs_vimtresponse = virtualMeetingRoomCreateResponseMessage.VimtResponse,
                mcs_status = virtualMeetingRoomCreateResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VirtualMeetingRoomCreateRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VirtualMeetingRoomCreateResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VirtualMeetingRoomCreateRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            if (virtualMeetingRoomCreateResponseMessage.ExceptionOccured) return;

            var updateServiceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointment.Id,
                mcs_meetingroomname = virtualMeetingRoomCreateResponseMessage.MeetingRoomName,
                mcs_PatientUrl = virtualMeetingRoomCreateResponseMessage.PatientUrl,
                mcs_providerurl = virtualMeetingRoomCreateResponseMessage.ProviderUrl,
                mcs_patientpin = virtualMeetingRoomCreateResponseMessage.PatientPin,
                mcs_providerpin = virtualMeetingRoomCreateResponseMessage.ProviderPin,
                mcs_dialingalias = virtualMeetingRoomCreateResponseMessage.DialingAlias,
                mcs_miscdata = virtualMeetingRoomCreateResponseMessage.MiscData,
                cvt_VMRCompleted = true
            };
            OrganizationService.Update(updateServiceAppointment);

            // Update the existing Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
        }

        /// <summary>
        /// calls ProcessVistaBookSingleRequest - same for create and update, just different input parameter object types
        /// </summary>
        /// <param name="response">Response.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVistaCreateResponse(VideoVisitCreateResponseMessage response, DataModel.ServiceAppointment serviceAppointment)
        {
            if (response == null) return;
            ProcessVistaBookSingleRequest(response.ExceptionOccured, response.ExceptionMessage, response.WriteResults, serviceAppointment, response.VimtRequest, response.SerializedInstance, response.VimtResponse, true, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);
        }

        /// <summary>
        /// calls ProcessVistaBookSingleRequest - same for create and update, just different input parameter object types
        /// </summary>
        /// <param name="response">Response.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVistaUpdateResponse(VideoVisitUpdateResponseMessage response, DataModel.ServiceAppointment serviceAppointment)
        {
            if (response == null) return;
            ProcessVistaBookSingleRequest(response.ExceptionOccured, response.ExceptionMessage, response.WriteResults, serviceAppointment, response.VimtRequest, response.SerializedInstance, response.VimtResponse, false, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);
        }

        /// <summary>
        /// Handle the response - update integration result to retry-success status, Vista "Write Results", and update Status of SA if applicable
        /// </summary>
        /// <param name="exceptionOccured">Exception Occured.</param>
        /// <param name="exceptionMessage">Exception Message.</param>
        /// <param name="writeResults">Write Results.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVistaBookSingleRequest(bool exceptionOccured, string exceptionMessage, WriteResults writeResults, DataModel.ServiceAppointment serviceAppointment, string vimtRequest, string serializedInstance, string vimtResponse, bool isCreate, int vimtLag, int ecProcessingTime, int vimtProcessingTime)
        {
            var errorMessage = exceptionOccured ? exceptionMessage : string.Empty;
            var reqType = isCreate ? typeof(VideoVisitCreateRequestMessage).FullName : typeof(VideoVisitUpdateRequestMessage).FullName;
            var respType = isCreate ? typeof(VideoVisitCreateResponseMessage).FullName : typeof(VideoVisitUpdateResponseMessage).FullName;
            var regType = isCreate ? MessageRegistry.VideoVisitCreateRequestMessage : MessageRegistry.VideoVisitUpdateRequestMessage;
            if (exceptionOccured)
            {
                IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, serviceAppointment.Id, serviceappointment_statuscode.InterfaceVIMTFailure);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("VVS Retry", errorMessage, vimtRequest, reqType, 
                    respType, regType, serviceAppointment.Id, OrganizationService, vimtRequest, vimtResponse, vimtLag, ecProcessingTime, vimtProcessingTime);
            }
            else
            {
                IntegrationPluginHelpers.CreateIntegrationResult("VVS Retry", false, errorMessage, 
                    vimtRequest, serializedInstance, vimtResponse, reqType, respType, regType, serviceAppointment.Id, OrganizationService, vimtLag, ecProcessingTime, vimtProcessingTime);
                IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
            }
        }

        private void ProcessVistaCreateGroupResponse(VideoVisitCreateResponseMessage response, DataModel.Appointment appointment)
        {
            if (response == null) return;
            ProcessGroupBookResponse(appointment, response.ExceptionOccured, response.ExceptionMessage, response.WriteResults, response.VimtRequest, response.SerializedInstance, response.VimtResponse, true, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);
        }

        private void ProcessVistaUpdateGroupResponse(VideoVisitUpdateResponseMessage response, DataModel.Appointment appointment)
        {
            if (response == null) return;
            ProcessGroupBookResponse(appointment, response.ExceptionOccured, response.ExceptionMessage, response.WriteResults, response.VimtRequest, response.SerializedInstance, response.VimtResponse, false, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);
       }

        private void ProcessGroupBookResponse(DataModel.Appointment appointment, bool exceptionOccured, string exceptionMessage, WriteResults writeResults, string vimtRequest, string serializedInstance, string vimtResponse, bool isCreate, int vimtLag, int ecProcessing, int vimtProcessing)
        {
            var errorMessage = exceptionOccured ? exceptionMessage : string.Empty;
            var reqType = isCreate ? typeof(VideoVisitCreateRequestMessage).FullName : typeof(VideoVisitUpdateRequestMessage).FullName;
            var respType = isCreate ? typeof(VideoVisitCreateResponseMessage).FullName : typeof(VideoVisitUpdateResponseMessage).FullName;
            var regName = isCreate ? MessageRegistry.VideoVisitCreateRequestMessage : MessageRegistry.VideoVisitUpdateRequestMessage;

            if (exceptionOccured)
            {
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, appointment.Id, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
                IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Group VVS Retry", errorMessage, vimtRequest, reqType, respType, regName, appointment.Id,
                    OrganizationService, vimtRequest, vimtResponse, vimtLag, ecProcessing, vimtProcessing);
            }
            else
            {
                IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
                IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group VVS Retry", false, errorMessage, vimtRequest, serializedInstance, vimtResponse, 
                    reqType, respType, regName, appointment.Id, OrganizationService, vimtLag, ecProcessing, vimtProcessing);
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, appointment.Id, Appointmentcvt_IntegrationBookingStatus.ReservedScheduled);
            }
        }

        /// <summary>
        /// Process Virtual Meeting Room Delete Response Message.
        /// </summary>
        /// <param name="virtualMeetingRoomDeleteResponseMessage">VirtualMeetingRoomDeleteResponseMessage.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVirtualMeetingRoomDeleteResponseMessage(VirtualMeetingRoomDeleteResponseMessage virtualMeetingRoomDeleteResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Cancel Virtual Meeting Room",
                mcs_error = virtualMeetingRoomDeleteResponseMessage.ExceptionOccured ? virtualMeetingRoomDeleteResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = virtualMeetingRoomDeleteResponseMessage.VimtRequest,
                mcs_integrationrequest = virtualMeetingRoomDeleteResponseMessage.SerializedInstance,
                mcs_vimtresponse = virtualMeetingRoomDeleteResponseMessage.VimtResponse,
                mcs_status = virtualMeetingRoomDeleteResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VirtualMeetingRoomDeleteRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VirtualMeetingRoomDeleteResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VirtualMeetingRoomDeleteRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            // If Retry was unsuccessful, we will not update the existing Int Result. 
            if (virtualMeetingRoomDeleteResponseMessage.ExceptionOccured) return;
            
            // Update the existing Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
        }
        
        /// <summary>
        /// Process Proxy Add to Vista response.
        /// </summary>
        /// <param name="responseMessage">ProxyAddResponseMessage.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessProxyAddResponse(ProxyAddResponseMessage responseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Proxy Add to VistA",
                mcs_error = responseMessage.ExceptionOccured ? responseMessage.ExceptionMessage : null,
                mcs_vimtrequest = responseMessage.VimtRequest,
                mcs_integrationrequest = responseMessage.SerializedInstance,
                mcs_vimtresponse = responseMessage.VimtResponse,
                mcs_status = responseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(ProxyAddRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(ProxyAddResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.ProxyAddRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            if (responseMessage.ExceptionOccured) return;

            var updateServiceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointment.Id,
                cvt_ProxyAddCompleted = true
            };

            try
            {
                //var saStatus = serviceAppointment.cvt_Type != null ? serviceAppointment.cvt_Type.Value ? serviceappointment_statuscode.Pending : serviceappointment_statuscode.ReservedScheduled : serviceappointment_statuscode.ReservedScheduled;
                //IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, serviceAppointment.Id, saStatus);
                OrganizationService.Update(updateServiceAppointment);
                IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException($"Failed to Update Proxy Add Completed Flag to initiate VVC/VVS plugin: {ex.Message}", ex);
            }
        }

        /// <summary>
        /// Process Group Proxy Add response.
        /// </summary>
        /// <param name="responseMessage">ProxyAddResponseMessage.</param>
        /// <param name="appointment">Appointment.</param>
        private void ProcessGroupProxyAddResponse(ProxyAddResponseMessage responseMessage, DataModel.Appointment appointment)
        {
            try
            {
                IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Proxy Add To Vista", responseMessage.ExceptionOccured, responseMessage.ExceptionMessage,
                    responseMessage.VimtRequest, responseMessage.SerializedInstance, responseMessage.VimtResponse, typeof(ProxyAddRequestMessage).FullName,
                    typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage, appointment.Id, OrganizationService, responseMessage.VimtLagMs, responseMessage.EcProcessingMs, responseMessage.VimtProcessingMs, false);

                if (!responseMessage.ExceptionOccured)
                {
                    IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
                    var updateAppt = new DataModel.Appointment
                    {
                        Id = appointment.Id,
                        cvt_TriggerVVS = true
                    };
                    OrganizationService.Update(updateAppt);
                    Logger.WriteDebugMessage("Triggered VVS via Retry");
                }
                else
                    Logger.WriteDebugMessage("Retry Group Proxy Add Failed");
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException($"Failed to Process Proxy Add Group Response: {ex.Message}", ex);
            }
        }
    }
}