﻿using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VRMRest;

namespace VA.TMP.Integration.Plugins.CvtVod
{
    /// <summary>
    ///  CRM Plugin Runner class to handle updating a VOD.
    /// </summary>
    public class CvtVodCreatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public CvtVodCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                var vodEntity = PrimaryEntity.ToEntity<cvt_vod>();
                using (var context = new Xrm(OrganizationService))
                {
                    // Retrieve the VOD.
                    var vod = context.cvt_vodSet.FirstOrDefault(x => x.Id == vodEntity.Id);
                    if (vod == null) throw new InvalidPluginExecutionException("VOD cannot be null.");

                    // Check the settings record to make sure we want to Call the Accenture & Vyopta Services
                    var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
                    if (settings == null) throw new InvalidPluginExecutionException("Active Settings cannot be null.");
                    
                    if (settings.cvt_accenturevyopta != null && settings.cvt_accenturevyopta.Value)
                    {
                        var vimtUrl = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                        if (vimtUrl == null) throw new InvalidPluginExecutionException("No VIMT URL Integration Setting was found");
                        VimtUrl = vimtUrl.mcs_value;

                        VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);
                        
                        // Call Service to create the Virtual Meeting Room
                        Logger.WriteDebugMessage("Beginning CreateAndSendVirtualMeetingRoom");
                        var vodCreateResponseMessage = CreateAndSendVod(vod, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName);
                        Logger.WriteDebugMessage("Finished CreateAndSendVirtualMeetingRoom");
                        if (vodCreateResponseMessage == null) return;

                        Logger.WriteDebugMessage("Beginning ProcessVodCreateResponseMessage");
                        ProcessVodCreateResponseMessage(vodCreateResponseMessage);
                        Logger.WriteDebugMessage("Finished ProcessVodCreateResponseMessage");
                    }
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(IntegrationPluginHelpers.BuildErrorMessage(ex));
                throw new InvalidPluginExecutionException($"ERROR in CvtVodCreatePostStageRunner: {IntegrationPluginHelpers.BuildErrorMessage(ex)}");
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(IntegrationPluginHelpers.BuildErrorMessage(ex));
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(IntegrationPluginHelpers.BuildErrorMessage(ex));
                throw;
            }
        }
                
        /// <summary>
        /// Create an instance of the Virtual Meeting Room request and send to VIMT.
        /// </summary>
        /// <returns>VmrOnDemandCreateResponseMessage.</returns>
        private VmrOnDemandCreateResponseMessage CreateAndSendVod(cvt_vod vod, Guid userId, string organizationName)
        {
            // Get the PatientId and ProviderId. Guard verbosely here against null references/objects.
            var patientEmail = vod.cvt_patientemail;
            if (patientEmail == null) throw new InvalidPluginExecutionException("VOD: Patient Email cannot be null.");

            var provider = vod.cvt_provider;
            if (provider == null) throw new InvalidPluginExecutionException("VOD: Provider cannot be null.");
            var providerId = provider.Id;

            var request = new VmrOnDemandCreateRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                VideoOnDemandId = vod.Id,
                PatientId = vod.Id,
                ProviderId = providerId,
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            VmrOnDemandCreateResponseMessage response = null;
            try
            {
                response = Utility.SendReceive<VmrOnDemandCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VmrOnDemandCreateRequestMessage, request, null, VimtTimeout, out int lag);
                response.VimtLagMs = lag;
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateVodIntegrationResultOnVimtFailure("Create On-Demand Virtual Meeting Room", errorMessage, vimtRequest, 
                    typeof(VmrOnDemandCreateRequestMessage).FullName, typeof(VmrOnDemandCreateResponseMessage).FullName, MessageRegistry.VmrOnDemandCreateRequestMessage, 
                    PrimaryEntity.Id, OrganizationService, response?.VimtRequest, response?.VimtResponse, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs);
                Logger.WriteToFile($"Error Sending VOD to VIMT: {IntegrationPluginHelpers.BuildErrorMessage(ex)}");

                return null;
            }
        }

        /// <summary>
        /// Update the VOD and create an Integration Result.
        /// </summary>
        /// <param name="vmrOnDemandCreateResponseMessage">Virtual Meeting Room Create Response Message.</param>
        private void ProcessVodCreateResponseMessage(VmrOnDemandCreateResponseMessage vmrOnDemandCreateResponseMessage)
        {
            var errorMessage = vmrOnDemandCreateResponseMessage.ExceptionOccured ? vmrOnDemandCreateResponseMessage.ExceptionMessage : null;
            IntegrationPluginHelpers.CreateVodIntegrationResult("Create On-Demand Virtual Meeting Room", vmrOnDemandCreateResponseMessage.ExceptionOccured, errorMessage,
                vmrOnDemandCreateResponseMessage.VimtRequest, vmrOnDemandCreateResponseMessage.SerializedInstance, vmrOnDemandCreateResponseMessage.VimtResponse,
                typeof(VmrOnDemandCreateRequestMessage).FullName, typeof(VmrOnDemandCreateResponseMessage).FullName, MessageRegistry.VmrOnDemandCreateRequestMessage,
                PrimaryEntity.Id, OrganizationService, vmrOnDemandCreateResponseMessage.VimtLagMs, vmrOnDemandCreateResponseMessage.EcProcessingMs, vmrOnDemandCreateResponseMessage.VimtProcessingMs);
            
            if (vmrOnDemandCreateResponseMessage.ExceptionOccured) return;

            var vod = new cvt_vod
            {
                Id = PrimaryEntity.Id,
                cvt_meetingroomname = vmrOnDemandCreateResponseMessage.MeetingRoomName,
                cvt_patienturl = vmrOnDemandCreateResponseMessage.PatientUrl,
                cvt_providerurl = vmrOnDemandCreateResponseMessage.ProviderUrl,
                cvt_patientpin = vmrOnDemandCreateResponseMessage.PatientPin,
                cvt_providerpin = vmrOnDemandCreateResponseMessage.ProviderPin,
                cvt_dialingalias = vmrOnDemandCreateResponseMessage.DialingAlias,
                cvt_miscdata = vmrOnDemandCreateResponseMessage.MiscData
            };
            OrganizationService.Update(vod);
            IntegrationPluginHelpers.UpdateVodStatus(OrganizationService, PrimaryEntity.Id, OptionSets.cvt_vod_statuscode.Success);
        }
    }
}