﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.CrmePerson
{
    /// <summary>
    ///  CRM Plugin Runner class to handle Person Search and Get Identifiers.
    /// </summary>
    public class CrmePersonRetrieveMultiplePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public CrmePersonRetrieveMultiplePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField => "crme_person";

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        private int VimtTimeout { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                // Make sure that a query has query parameters
                if (!PluginExecutionContext.InputParameters.Contains("Query") || !(PluginExecutionContext.InputParameters["Query"] is QueryExpression)) return;

                var qe = (QueryExpression)PluginExecutionContext.InputParameters["Query"];
                int integrationType;
                PersonSearchPluginUtilities.TryGetConditionOptionSet(qe, "cvt_IntegrationType", out integrationType);
                if (integrationType != (int)crme_personcvt_IntegrationType.PatientSearch)
                    return;

                string mviSearchType;

                if (!PersonSearchPluginUtilities.TryGetFilterValue(qe, "crme_SearchType", out mviSearchType)) return;

                ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Clear();

                Logger.WriteGranularTimingMessage("Starting retrieval of Person(s)");

                string serverName;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "crme_url", out serverName);

                mcs_integrationsetting fakeType;
                using (var context = new Xrm(OrganizationService))
                {
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, this.GetType().Name);
                    var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
                    else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
                }
                var fakeTypeValue = string.Empty;
                switch (mviSearchType)
                {
                    case "SearchByIdentifier":
                    case "SearchByFilter":
                        using (var context = new Xrm(OrganizationService))
                            fakeType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PersonSearchFakeResponseType");

                        fakeTypeValue = fakeType == null ? "" : fakeType.mcs_value;
                        var request = PersonSearchPluginUtilities.GetPersonSearchRequest(qe, PluginExecutionContext, OrganizationService);
                        request.PersonSearchFakeResponseType = fakeTypeValue;
                        var response = VRMRest.Utility.SendReceive<PersonSearchResponseMessage>(new Uri(VimtUrl), MessageRegistry.PersonSearchRequestMessage, request, null, VimtTimeout, out int lag);
                        response.VimtLagMs = lag;
                        Logger.WriteDebugMessage("VIMT PersonSearch Response Received, processing response.");
                        CreateIntegrationResult(response, Logger);
                        var psResponse = PersonSearchPluginUtilities.GetPersonSearchResponse(request, response.RetrieveOrSearchPersonResponse, OrganizationService);
                        ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.AddRange(psResponse);
                        break;
                    case "SelectedPersonSearch":
                        using (var context = new Xrm(OrganizationService))
                            fakeType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "SelectedPersonFakeResponseType");

                        fakeTypeValue = fakeType != null ? fakeType.mcs_value : string.Empty;
                        var request2 = PersonSearchPluginUtilities.GetSelectedPersonRequest(qe, PluginExecutionContext, OrganizationService);
                        request2.ServerName = serverName;
                        request2.SelectedPersonFakeResponseType = fakeTypeValue;
                        var vimtResponse = VRMRest.Utility.SendReceive<GetPersonIdentifiersResponseMessage>(new Uri(VimtUrl), MessageRegistry.GetPersonIdentifiersRequestMessage, request2, null, VimtTimeout, out int lag2);
                        vimtResponse.VimtLagMs = lag2;
                        Logger.WriteDebugMessage("VIMT GetIds Response Received, processing Response");
                        CreateIntegrationResultOnGetPersonIdentifiers(vimtResponse, Logger);
                        var selectedPersons = PersonSearchPluginUtilities.GetSelectedPersonIds(vimtResponse, serverName);
                        ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.AddRange(selectedPersons);
                        break;
                    default:
                        throw new InvalidPluginExecutionException("Unable to Determine MVI Search Type");
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in CrmePersonRetrieveMultiplePostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        private void CreateIntegrationResult(PersonSearchResponseMessage response, MCSLogger logger)
        {
            logger.WriteDebugMessage("CreateIntegrationResultOnPersonSearch Initiated");
            IntegrationPluginHelpers.CreateIntegrationResultOnPersonSearch("Person Search - Traits", response.ExceptionOccured, response.ExceptionMessage, response.VimtRequest, response.SerializedInstance, response.VimtResponse, typeof(PersonSearchRequestMessage).FullName, typeof(PersonSearchResponseMessage).FullName, MessageRegistry.PersonSearchRequestMessage, OrganizationService, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs, logger);
            logger.WriteDebugMessage("CreateIntegrationResultOnPersonSearch Call Complete");
        }

        private void CreateIntegrationResultOnGetPersonIdentifiers(GetPersonIdentifiersResponseMessage response, MCSLogger logger)
        {
            logger.WriteDebugMessage("CreateIntegrationResultOnPersonSearchGetPersonIdentifiers Initiated");
            IntegrationPluginHelpers.CreateIntegrationResultOnGetPersonIdentifiers("Person Search - GetPersonIdentifiers", response.ExceptionOccured, response.ExceptionMessage, response.VimtRequest, response.SerializedInstance, response.VimtResponse, typeof(GetPersonIdentifiersResponseMessage).FullName, typeof(GetPersonIdentifiersResponseMessage).FullName, MessageRegistry.GetPersonIdentifiersResponseMessage, OrganizationService, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs, logger);
            logger.WriteDebugMessage("CreateIntegrationResultOnPersonSearchGetPersonIdentifiers Call Complete");
        }
    }
}