﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.CrmePerson
{
    public class CrmePersonRetrieveMultiplePostHealthShareGetConsultsRunner : PluginRunner
    {
        public CrmePersonRetrieveMultiplePostHealthShareGetConsultsRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField => "crme_person";

        private int VimtTimeout { get; set; }

        public override void Execute()
        {
            try
            {
                var qe = (QueryExpression)PluginExecutionContext.InputParameters["Query"];
                PersonSearchPluginUtilities.TryGetConditionOptionSet(qe, "cvt_IntegrationType", out var integrationTypeValue);
                if (integrationTypeValue != (int)crme_personcvt_IntegrationType.GetConsultsForPatient) return;

                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_patientsiteid", out var patientSiteId);
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "crme_contactid", out var patientContactId);
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_providersiteid", out var providerSiteId);
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_issft", out var sft);
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_ishomemobile", out var homeMobile);

                Logger.WriteDebugMessage($"Patient Site Id: {patientSiteId}");
                Logger.WriteDebugMessage($"Provider Site Id: {providerSiteId}");

                var providerStationNumber = GetStationNumberFromSiteId(providerSiteId, Logger);
                var patientStationNumber = GetStationNumberFromSiteId(patientSiteId, Logger);

                Logger.WriteDebugMessage($"Patient Station Number: {patientStationNumber}");
                Logger.WriteDebugMessage($"Provider Station Number: {providerStationNumber}");

                var isSft = Convert.ToBoolean(sft);
                var isHomeMobile = Convert.ToBoolean(homeMobile);

                string vimtUrl;
                using (var context = new Xrm(OrganizationService))
                {
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);
                    var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlSetting != null) vimtUrl = vimtUrlSetting.mcs_value;
                    else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Get Consults Canceled.");
                }

                var patientIds = new List<Guid> {new Guid(patientContactId)};

                var patientLoginStationNumber = ExtractStationNumber(patientStationNumber);
                var providerLoginStationNumber = ExtractStationNumber(providerStationNumber);

                Logger.WriteDebugMessage($"Patient Login Station Number: {patientLoginStationNumber}");
                Logger.WriteDebugMessage($"Provider Login Station Number: {providerLoginStationNumber}");

                Logger.WriteDebugMessage("Populating HealthShareGetConsultsRequest");

                var healthshareRequest = new HealthShareGetConsultsRequest
                {
                    Debug = true,
                    PatientIds = patientIds,
                    PatientLoginStationNumber = patientLoginStationNumber,
                    ProviderLoginStationNumber = providerLoginStationNumber,
                    LogRequest = true,
                    UserId = PluginExecutionContext.UserId,
                    OrganizationName = PluginExecutionContext.OrganizationName,
                    IsHomeMobile = isHomeMobile,
                    IsStoreForward = isSft,
                };

                Logger.WriteDebugMessage("About to send/receive HealthShareGetConsults message.");
                var response = VRMRest.Utility.SendReceive<HealthShareGetConsultsResponse>(new Uri(vimtUrl), MessageRegistry.HealthShareGetConsultsRequest, healthshareRequest, null, VimtTimeout, out int lag);
                Logger.WriteDebugMessage("Finished send/receive HealthShareGetConsults message.");
                response.VimtLagMs = lag;
                Logger.WriteDebugMessage("Mapping the HealthShareGetConsults response");
                var mappedResponse = MapHealthShareResponseToCrm(response, patientStationNumber, providerStationNumber, patientSiteId, providerSiteId, isSft);
                Logger.WriteDebugMessage("Response received: " + response.VimtResponse);

                var entityCollection = ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]);
                entityCollection.Entities.AddRange(mappedResponse);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException(ex.Message);
            }
        }

        /// <summary>
        /// Extract the first 3 digits of the station from the station number string provided
        /// </summary>
        /// <param name="stationString">The Station Number String</param>
        /// <returns>The Station number in the form of Integer</returns>
        private static int ExtractStationNumber(string stationString)
        {
            var stationNumber = 0;

            if (string.IsNullOrWhiteSpace(stationString)) return stationNumber;

            stationString = stationString.Trim();
            var threeChars = stationString.Length < 3 ? stationString : stationString.Substring(0, 3);
            int.TryParse(threeChars, out stationNumber);

            return stationNumber;
        }

        /// <summary>
        /// Map the Health Share Get Consults Response To Crm
        /// </summary>
        /// <param name="response">The Health Share Get Consults Response</param>
        /// <param name="patientLoginStationNumber">The Patient Login Station Number</param>
        /// <param name="providerLoginStationNumber">The Provider Login StationNumber</param>
        /// <param name="patSiteId">The Patient Site Guid</param>
        /// <param name="proSiteId">The Provide Site Guid</param>
        /// <param name="isSft">is Store Forward</param>
        /// <returns>The list of person records</returns>
        private List<crme_person> MapHealthShareResponseToCrm(HealthShareGetConsultsResponse response, string patientLoginStationNumber, string providerLoginStationNumber, string patSiteId, string proSiteId, bool isSft)
        {
            Logger.WriteDebugMessage($"MapHealthShareResponseToCrm Started. patSiteId: {patSiteId}, proSiteId:{proSiteId}, isSft: {isSft}");

            if (response == null) return null;

            var persons = new List<crme_person>
            {
                new crme_person
                {
                    cvt_ConsultIEN = VistaPluginHelpers.GetDuzFromStationCode(PluginExecutionContext.UserId, Logger, OrganizationService, patientLoginStationNumber),
                    cvt_ConsultText = VistaPluginHelpers.GetDuzFromStationCode(PluginExecutionContext.UserId, Logger, OrganizationService, providerLoginStationNumber),
                    crme_url = "neither",
                    Id = new Guid()
                }
            };

            var patStationCode = string.Empty;
            var proStationCode = string.Empty;

            using (var srv = new Xrm(OrganizationService))
            {
                var patSite = !string.IsNullOrEmpty(patSiteId) ? srv.mcs_siteSet.FirstOrDefault(s => s.Id == new Guid(patSiteId)) : null;
                var proSite = !string.IsNullOrEmpty(proSiteId) ? srv.mcs_siteSet.FirstOrDefault(s => s.Id == new Guid(proSiteId)) : null;

                if (patSite?.mcs_FacilityId != null)
                {
                    var facility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == patSite.mcs_FacilityId.Id);
                    patStationCode = facility?.mcs_StationNumber ?? string.Empty;
                }
                else Logger.WriteDebugMessage("Unable to get site for patient side.");

                if (proSite?.mcs_FacilityId != null)
                {
                    var facility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == proSite.mcs_FacilityId.Id);
                    proStationCode = facility?.mcs_StationNumber ?? string.Empty;
                }
                else Logger.WriteDebugMessage("Unable to get site for provider side.");
            }
            var sameFacilityCode = patStationCode == proStationCode;

            if (response.PatientConsults != null && response.PatientConsults.Any())
            {
                var label = "patient";
                if (sameFacilityCode && !isSft) label = "provider";

                var patPersons = MapHealthShareConsultsResponse(response, response.PatientConsults, label);
                persons.AddRange(patPersons);
                var patPersonsRtc = MapReturnToClinicOrdersResponse(response, response.PatientReturnToClinicOrders, label);
                persons.AddRange(patPersonsRtc);
            }

            if (response.ProviderConsults != null && response.ProviderConsults.Any())
            {
                var proPersons = MapHealthShareConsultsResponse(response, response.ProviderConsults, "provider");
                persons.AddRange(proPersons);
                var proPersonsRtc = MapReturnToClinicOrdersResponse(response, response.ProviderReturnToClinicOrders, "provider");
                persons.AddRange(proPersonsRtc);
            }

            Logger.WriteDebugMessage("MapHealthShareResponseToCrm Ended");
            return persons;
        }

        /// <summary>
        /// Map the HealthShare Consults Response
        /// </summary>
        /// <param name="response">The Health Share Get Consults response</param>
        /// <param name="consults">The consults from the response</param>
        /// <param name="side">Patient/Provider Side</param>
        /// <returns>List of People.</returns>
        private IEnumerable<crme_person> MapHealthShareConsultsResponse(HealthShareGetConsultsResponse response, List<TmpConsult> consults, string side)
        {
            Logger.WriteDebugMessage($"MapHealthShareConsultsResponse Started. Side {side}");

            if (consults != null && consults.Any())
            {
                if (response.ExceptionOccured)
                {
                    Logger.WriteDebugMessage($"Consults response.ExceptionMessage - {response.ExceptionMessage}");
                    yield return CreateFailedCrmePersonObject(response.ExceptionMessage, true);
                }
                else
                {
                    foreach (var consult in consults)
                    {
                        if (consult == null) continue;
                        
                        var person = new crme_person
                        {
                            cvt_ConsultIEN = consult.ConsultId,
                            cvt_ConsultStatus = GetStatusName(consult.ConsultStatus),
                            cvt_ConsultText = string.Empty,
                            cvt_ConsultTimestamp = consult.ConsultRequestDateTime,
                            cvt_ClinicallyIndicatedDate = consult.ClinicallyIndicatedDate,
                            cvt_ConsultTitle = consult.ConsultTitle,
                            Id = new Guid(),
                            crme_url = side, //arbitrarily using this variable to temporarily send the pat/pro string across to the js
                            cvt_ConsultType = true
                        };

                        yield return person;
                    }
                }
            }

            Logger.WriteDebugMessage("MapHealthShareConsultsResponse Ended");
        }

        /// <summary>
        /// Map Return to Clinic Orders.
        /// </summary>
        /// <param name="response">VIMT response.</param>
        /// <param name="returnToClinicOrders">Return to Clinic Orders.</param>
        /// <param name="side">Patient or Provider.</param>
        /// <returns>List of People.</returns>
        private IEnumerable<crme_person> MapReturnToClinicOrdersResponse(HealthShareGetConsultsResponse response, List<TmpReturnToClinicOrder> returnToClinicOrders, string side)
        {
            Logger.WriteDebugMessage($"MapReturnToClinicOrdersResponse Started. Side {side}");

            if (returnToClinicOrders != null && returnToClinicOrders.Any())
            {
                if (response.ExceptionOccured)
                {
                    Logger.WriteDebugMessage($"Return to Clinic Orders response.ExceptionMessage - {response.ExceptionMessage}");
                    yield return CreateFailedCrmePersonObject(response.ExceptionMessage, false);
                }
                else
                {
                    foreach (var returnToClinicOrder in returnToClinicOrders)
                    {
                        if (returnToClinicOrder == null) continue;
                        
                        var person = new crme_person
                        {
                            cvt_RtcId = returnToClinicOrder.RtcId,
                            cvt_RtcRequestDateTime = returnToClinicOrder.RtcRequestDateTime,
                            cvt_ClinicIen = returnToClinicOrder.ToClinicIen,
                            cvt_ClinicName = returnToClinicOrder.ClinicName,
                            cvt_ClinicallyIndicatedDate = returnToClinicOrder.ClinicallyIndicatedDate,
                            cvt_StopCodes = returnToClinicOrder.StopCodes,
                            cvt_Provider = returnToClinicOrder.Provider,
                            cvt_Comments = returnToClinicOrder.Comments,
                            Id = new Guid(),
                            crme_url = side,
                            cvt_ConsultType = false
                        };

                        var multipleRtc = returnToClinicOrder.MultiRtc;
                        if (!string.IsNullOrEmpty(multipleRtc))
                        {
                            var parts = multipleRtc.Split(',');
                            if (parts != null && parts.Length > 0)
                            {
                                person.cvt_numberofappointments = parts[0];
                                person.cvt_interval = parts.Length >= 2 ? parts[1] : string.Empty;
                                person.cvt_rtcparent = parts.Length == 3 ? parts[2] : string.Empty;
                            }
                        }

                        yield return person;
                    }
                }
            }

            Logger.WriteDebugMessage("MapReturnToClinicOrdersResponse Ended");
        }

        /// <summary>
        /// Replace/Substitute the Status Acronym with the Status Display Name
        /// </summary>
        /// <param name="consultStatus"></param>
        /// <returns></returns>
        private static string GetStatusName(string consultStatus)
        {
            string statusString;
            switch(consultStatus.Trim().ToLower())
            {
                case "dc":
                    statusString = "Discontinued";
                    break;
                case "c":
                    statusString = "Complete";
                    break;
                case "h":
                    statusString = "Hold";
                    break;
                case "p":
                    statusString = "Pending";
                    break;
                case "a":
                    statusString = "Active";
                    break;
                case "e":
                    statusString = "Expired";
                    break;
                case "s":
                    statusString = "Scheduled";
                    break;
                case "pr":
                    statusString = "Partial Results";
                    break;
                case "dly":
                    statusString = "Delayed";
                    break;
                case "u":
                    statusString = "Unreleased";
                    break;
                case "dce":
                    statusString = "Discontinued/Edit";
                    break;
                case "x":
                    statusString = "Cancelled";
                    break;
                case "l":
                    statusString = "Lapsed";
                    break;
                case "rn":
                    statusString = "Renewed";
                    break;
                default:
                    statusString = consultStatus.Contains("?") ? "Flagged" : "No Status";
                    break;
            }
            return statusString;
        }

        private static crme_person CreateFailedCrmePersonObject(string message, bool isConsult)
        {
            return new crme_person
            {
                Id = new Guid(),
                crme_url = "fail", //flag read by javascript in html page to look for a failure
                cvt_ConsultTitle = message,
                cvt_ConsultType = isConsult
            };
        }

        /// <summary>
        /// Retrieve the Station Number for the Site Id Provided
        /// </summary>
        /// <param name="siteId">The Site Guid</param>
        /// <param name="logger">The logger object for logging purposes</param>
        /// <returns>The Site's Station Number</returns>
        public string GetStationNumberFromSiteId(string siteId, MCSUtilities2011.MCSLogger logger)
        {
            logger.WriteDebugMessage("Starting GetStationNumberFromSiteId");

            var output = string.Empty;
            Guid siteGuid;

            if (siteId != string.Empty) siteGuid = Guid.Parse(siteId);
            else return output;

            using (var context = new Xrm(OrganizationService))
            {
                var siteRecord = context.mcs_siteSet.FirstOrDefault(s => s.Id == siteGuid);
                if (siteRecord != null)
                {
                    logger.WriteDebugMessage("Found TMP Site");
                    var obj = siteRecord;

                    if (obj.mcs_FacilityId != null)
                    {
                        var facilityRecord = context.mcs_facilitySet.FirstOrDefault(f => f.Id == obj.mcs_FacilityId.Id);
                        if (facilityRecord != null)
                        {
                            logger.WriteDebugMessage("Found Facility");
                            output = facilityRecord.mcs_StationNumber;
                        }
                    }
                }
            }
            return output;
        }
    }
}